﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief クライアントから消費型サービスアイテムを利用するためのAPIの定義
 */

#pragma once
#include <nn/nn_Result.h>
#include <nn/account.h>
#include <nn/ec/ec_ConsumableServiceItemApi.h>
#include <nn/ec/ec_ShopServiceAccessor.h>
#include <nn/os.h>
#include <nn/util/util_ScopeExit.h>

namespace nnt { namespace ec {

/**
    @brief 消費型サービスアイテムのダミー権利を購入する

    @details

        POST /v1/applications/{applicationId}/accounts/{nsaId}/rights
        Authorization: Basic {base64(clientId:clientSecret)}
        Content-Type:application/json

        {
            "item_id": (String; the ID of the purchased item),
            "country": "(String; 2 letters (ISO 3166-1 Alpha-2); the country of the created right)"
        }
*/
void PurchaseRight(const nn::account::NetworkServiceAccountId& nsaId, const nn::ec::ConsumableServiceItemId Items[], int itemIdCount) NN_NOEXCEPT;

//! 同期要求クラス
class ConsumableServiceItemSyncRequest
{
public:
    ConsumableServiceItemSyncRequest()
    {
    }

    void Initialize(const nn::account::Uid& user, void* pWorkMemory, size_t workMemorySize) NN_NOEXCEPT
    {
        m_User = user;
        m_pWorkMemory = pWorkMemory;
        m_pWorkMemorySize = workMemorySize;
    }

    //! 権利参照要求
    nn::Result GetConsumableRightData(
        int* pOutRightDataCount, nn::ec::ConsumableServiceItemRightData outRightDataArray[], int maxRightDataCount, int page, int perPage) NN_NOEXCEPT;

    //! 提供可能権利取得要求
    nn::Result GetProvidableRightData(
        int* pOutRightDataCount, nn::ec::ConsumableServiceItemRightData outRightDataArray[], int maxRightDataCount, const nn::ec::ConsumableServiceItemRightData rightData[], int rightDataCount) NN_NOEXCEPT;

    //! 権利消費要求
    nn::Result ConsumeRightData(
        int* pOutRightDataCount, nn::ec::ConsumableServiceItemRightData outRightDataArray[], int maxRightDataCount, const nn::ec::ConsumableServiceItemRightData rightData[], int rightDataCount) NN_NOEXCEPT;

private:

    nn::account::Uid m_User;
    void* m_pWorkMemory;
    size_t m_pWorkMemorySize;
};

//! セーブデータ読み込み
nn::Result ReadConsumableServiceItemRightData(void* buffer, size_t* pBufferSize, const char* path) NN_NOEXCEPT;

//! セーブデータ書き込み
nn::Result WriteConsumableServiceItemRightData(const char* path, const void* buffer, const size_t bufferSize) NN_NOEXCEPT;

//! 権利情報の ConsumableServiceItemConsumptionRequestId を再発行します
void RepublishConsumptionRequestIdOfConsumableServiceItemRightData(
    int* pOutRightDataCount, nn::ec::ConsumableServiceItemRightData outRightDataArray[], int maxRightDataCount, const nn::ec::ConsumableServiceItemRightData rightDataArray[], int rightDataCount) NN_NOEXCEPT;

}}
