﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <algorithm>
#include <nnt/nntest.h>
#include <nnt/result/testResult_Assert.h>

#include <nn/nn_Log.h>
#include <nn/account.h>
#include <nn/account/account_Selector.h>
#include <nn/ec/ec_ShopServiceAccessor.h>
#include <nn/fs.h>
#include <nn/oe.h>
#include <nn/nifm.h>
#include <nn/ec/ec_ConsumableServiceItemTypes.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/socket.h>
#include <nn/ssl.h>
#include "testEc_ConsumableServiceItemSyncRequestApi.h"

namespace
{
    const int MaxRightCount = 20;
    const nn::ec::ConsumableServiceItemId SsdOcsiTestItem[3] = { {"SsdOcsiTestItem0"}, {"SsdOcsiTestItem1"}, {"SsdOcsiTestItem2"} };

    //! テストで使用するユーザ
    nn::account::Uid g_User = nn::account::InvalidUid;
    nn::account::NetworkServiceAccountId g_NsaId;

    //! ショップアクセッサが使用するワークメモリ
    NN_ALIGNAS(nn::os::MemoryPageSize) nn::Bit8 g_WorkMemory[nn::os::MemoryPageSize * 16];

    //! ライブラリが使用するワークメモリ
    const size_t g_LibraryWorkMemorySize = nn::ec::RequiredWorkMemorySizeForConsumableServiceItem;
    NN_ALIGNAS(nn::os::MemoryPageSize) nn::Bit8 g_LibraryWorkMemory[g_LibraryWorkMemorySize];

    nn::socket::ConfigDefaultWithMemory g_SocketConfigWithMemory;

    //! 所有する全ての権利を消費済みにする
    void ConsumeAllRightData(const nn::account::Uid user) NN_NOEXCEPT
    {
        nn::ec::ShopServiceAccessor accessor;
        NNT_ASSERT_RESULT_SUCCESS(accessor.Initialize(nn::ec::ShopService(nn::ec::ShopService::Type_OwnedConsumableServiceItem)));

        nn::ec::ConsumableServiceItemRightData rights[MaxRightCount];
        int rightCount = 0;

        //! バッファに取るけど使わない
        nn::ec::ConsumableServiceItemRightData consumedRights[MaxRightCount];
        int consumedRightCount = 0;

        nnt::ec::ConsumableServiceItemSyncRequest syncRequest;
        syncRequest.Initialize(g_User, g_LibraryWorkMemory, sizeof(g_LibraryWorkMemory));

        while (NN_STATIC_CONDITION(true))
        {
            //! 権利参照
            NNT_ASSERT_RESULT_SUCCESS(syncRequest.GetConsumableRightData(&rightCount, rights, MaxRightCount, 0, MaxRightCount));

            if (rightCount == 0)
            {
                return;
            }

            //! 権利消費
            int count = std::min(rightCount, nn::ec::ConsumableServiceItemRightDataCountMax);
            NNT_ASSERT_RESULT_SUCCESS(syncRequest.ConsumeRightData(&consumedRightCount, consumedRights, MaxRightCount, rights, count));

            //! 消費頻度に wait をかける
            nn::os::SleepThread(nn::TimeSpan::FromSeconds(10));
        }
    }

    //! オンメモリ消費を評価する
    void EvaluateOnMemoryConsumption(const nn::account::Uid user, const nn::ec::ConsumableServiceItemId Items[], int itemCount) NN_NOEXCEPT
    {
        nnt::ec::ConsumableServiceItemSyncRequest syncRequest;
        syncRequest.Initialize(g_User, g_LibraryWorkMemory, sizeof(g_LibraryWorkMemory));

        //! 権利参照
        int consumableCount = 0;
        nn::ec::ConsumableServiceItemRightData consumable[MaxRightCount];

        //! 購入時刻の降順で権利情報を取得する (後入れ先出し)
        NNT_ASSERT_RESULT_SUCCESS(syncRequest.GetConsumableRightData(&consumableCount, consumable, MaxRightCount, 0, MaxRightCount));

        //! 評価
        ASSERT_EQ(itemCount, consumableCount);
        for (int i = 0; i < consumableCount; i++)
        {
            ASSERT_EQ(0, nn::util::Strncmp(Items[itemCount - i - 1].value, consumable[i].GetConsumableServiceItemId().value, sizeof(consumable[i].GetConsumableServiceItemId().value)));
            ASSERT_EQ(false, consumable[i].IsServiceProvidableOnLocal());
        }

        //! 権利消費
        int consumedRightCount = 0;
        nn::ec::ConsumableServiceItemRightData consumed[MaxRightCount];
        int count = std::min(consumableCount, nn::ec::ConsumableServiceItemRightDataCountMax);
        NNT_ASSERT_RESULT_SUCCESS(syncRequest.ConsumeRightData(&consumedRightCount, consumed, MaxRightCount, consumable, count));

        //! 評価
        ASSERT_EQ(count, consumedRightCount);
        for (int i = 0; i < count; i++)
        {
            ASSERT_EQ(true, consumed[i].IsServiceProvidableOnLocal());
        }

        //! 権利参照
        NNT_ASSERT_RESULT_SUCCESS(syncRequest.GetConsumableRightData(&consumableCount, consumable, MaxRightCount, 0, MaxRightCount));

        //! 評価
        ASSERT_EQ(0, consumableCount);
    }

    //! ユーザセーブデータ管理方式を評価する
    void EvaluateOnSaveDataConsumption(const nn::account::Uid user, const nn::ec::ConsumableServiceItemRightData onSaveData[], int onSaveDataCount, int expectedProvidableCount, int expectedConsumableCount) NN_NOEXCEPT
    {
        nnt::ec::ConsumableServiceItemSyncRequest syncRequest;
        syncRequest.Initialize(g_User, g_LibraryWorkMemory, sizeof(g_LibraryWorkMemory));

        //! セーブデータに記録された権利情報のうち、提供可能な権利情報を取得する
        int providableCount = 0;
        nn::ec::ConsumableServiceItemRightData providable[MaxRightCount];
        NNT_ASSERT_RESULT_SUCCESS(syncRequest.GetProvidableRightData(&providableCount, providable, MaxRightCount, onSaveData, onSaveDataCount));

        //! 評価
        ASSERT_EQ(expectedProvidableCount, providableCount);
        for (int i = 0; i < providableCount; i++)
        {
            ASSERT_EQ(true, providable[i].IsServiceProvidableOnLocal());
        }

        //! 未消費の権利を取得する
        int consumableCount = 0;
        nn::ec::ConsumableServiceItemRightData consumable[MaxRightCount];
        NNT_ASSERT_RESULT_SUCCESS(syncRequest.GetConsumableRightData(&consumableCount, consumable, MaxRightCount, 0, MaxRightCount));

        //! 評価
        ASSERT_EQ(expectedConsumableCount, consumableCount);
        for (int i = 0; i < consumableCount; i++)
        {
            ASSERT_EQ(false, consumable[i].IsServiceProvidableOnLocal());
        }

        //! 提供可能な権利と消費可能な権利を統合する
        int integratedCount = providableCount + consumableCount;
        nn::ec::ConsumableServiceItemRightData integrated[MaxRightCount];
        {
            for (int i = 0; i < providableCount; i++)
            {
                integrated[i] = providable[i];
            }

            for (int i = 0; i < consumableCount; i++)
            {
                integrated[providableCount + i] = consumable[i];
            }
        }

        //! TODO: セーブデータコミット

        //! 権利消費
        int consumedRightCount = 0;
        nn::ec::ConsumableServiceItemRightData consumed[MaxRightCount];
        int count = std::min(integratedCount, nn::ec::ConsumableServiceItemRightDataCountMax);
        NNT_ASSERT_RESULT_SUCCESS(syncRequest.ConsumeRightData(&consumedRightCount, consumed, MaxRightCount, integrated, count));

        //! 評価
        ASSERT_EQ(count, consumedRightCount);
        for (int i = 0; i < count; i++)
        {
            ASSERT_EQ(true, consumed[i].IsServiceProvidableOnLocal());
        }

        //! TODO: セーブデータコミット

        //! 権利参照
        NNT_ASSERT_RESULT_SUCCESS(syncRequest.GetConsumableRightData(&consumableCount, consumable, MaxRightCount, 0, MaxRightCount));

        //! 評価
        ASSERT_EQ(0, consumableCount);
    }
}

//! テスト環境の初期化
TEST(ConsumableServiceItemApi, Initialize)
{
    //! 事前に DevMenuCommand を使ってアカウントを作成する (.testlist.yml に記述)

    NNT_ASSERT_RESULT_SUCCESS(nn::socket::Initialize(g_SocketConfigWithMemory));
    NNT_ASSERT_RESULT_SUCCESS(nn::ssl::Initialize());
    NNT_ASSERT_RESULT_SUCCESS(nn::nifm::Initialize());
    nn::account::Initialize();
    nn::oe::Initialize();

    // ネットワーク利用要求
    nn::nifm::SubmitNetworkRequestAndWait();
    NNT_ASSERT_RESULT_SUCCESS(nn::nifm::HandleNetworkRequestResult());

    //! アカウント選択（1 人目を選択する）
    {
        int count;
        nn::account::Uid users[nn::account::UserCountMax];
        NNT_ASSERT_RESULT_SUCCESS(nn::account::ListAllUsers(&count, users, sizeof(users) / sizeof(users[0])));
        g_User = users[0];

        nn::account::UserHandle userHandle;
        NNT_ASSERT_RESULT_SUCCESS(nn::account::OpenUser(&userHandle, g_User));
        NNT_ASSERT_RESULT_SUCCESS(nn::account::EnsureNetworkServiceAccountAvailable(userHandle));
        NNT_ASSERT_RESULT_SUCCESS(nn::account::GetNetworkServiceAccountId(&g_NsaId, userHandle));
    }
}

//! 1, 5, 20 個のテストをオンメモリで消費する
TEST(ConsumableServiceItemApi, OnMemoryConsumption)
{
    NNT_ASSERT_RESULT_SUCCESS(nn::ec::InitializeForShopServiceAccessors(g_WorkMemory, sizeof(g_WorkMemory)));

    //! 1 個
    {
        ConsumeAllRightData(g_User);

        const nn::ec::ConsumableServiceItemId Items[] =
        {
            SsdOcsiTestItem[0],
        };
        int itemCount = NN_ARRAY_SIZE(Items);

        nnt::ec::PurchaseRight(g_NsaId, Items, itemCount);

        //! テスト
        EvaluateOnMemoryConsumption(g_User, Items, itemCount);
    }

    //! 5 個
    {
        ConsumeAllRightData(g_User);

        const nn::ec::ConsumableServiceItemId Items[] =
        {
            SsdOcsiTestItem[2],
            SsdOcsiTestItem[2],
            SsdOcsiTestItem[2],
            SsdOcsiTestItem[1],
            SsdOcsiTestItem[2],
        };
        int itemCount = NN_ARRAY_SIZE(Items);

        nnt::ec::PurchaseRight(g_NsaId, Items, itemCount);

        //! テスト
        EvaluateOnMemoryConsumption(g_User, Items, itemCount);
    }

    //! 20 個
    {
        ConsumeAllRightData(g_User);

        const nn::ec::ConsumableServiceItemId Items[] =
        {
            SsdOcsiTestItem[2],
            SsdOcsiTestItem[0],
            SsdOcsiTestItem[0],
            SsdOcsiTestItem[1],
            SsdOcsiTestItem[1],
            SsdOcsiTestItem[0],
            SsdOcsiTestItem[0],
            SsdOcsiTestItem[0],
            SsdOcsiTestItem[0],
            SsdOcsiTestItem[1],
            SsdOcsiTestItem[2],
            SsdOcsiTestItem[0],
            SsdOcsiTestItem[1],
            SsdOcsiTestItem[2],
            SsdOcsiTestItem[0],
            SsdOcsiTestItem[1],
            SsdOcsiTestItem[2],
            SsdOcsiTestItem[0],
            SsdOcsiTestItem[1],
            SsdOcsiTestItem[2],
        };
        int itemCount = NN_ARRAY_SIZE(Items);

        nnt::ec::PurchaseRight(g_NsaId, Items, itemCount);

        //! テスト
        EvaluateOnMemoryConsumption(g_User, Items, itemCount);
    }

    NNT_ASSERT_RESULT_SUCCESS(nn::ec::FinalizeForShopServiceAccessors());
}

//! 電源断復帰時テスト
TEST(ConsumableServiceItemApi, ConsumeSuccessResponseLostTest)
{
    NNT_ASSERT_RESULT_SUCCESS(nn::ec::InitializeForShopServiceAccessors(g_WorkMemory, sizeof(g_WorkMemory)));

    //! 18 個
    int itemCount;
    {
        ConsumeAllRightData(g_User);

        const nn::ec::ConsumableServiceItemId Items[] =
        {
            SsdOcsiTestItem[0],
            SsdOcsiTestItem[2],
            SsdOcsiTestItem[0],
            SsdOcsiTestItem[2],
            SsdOcsiTestItem[1],
            SsdOcsiTestItem[1],
            SsdOcsiTestItem[0],
            SsdOcsiTestItem[1],
            SsdOcsiTestItem[2],
            SsdOcsiTestItem[2],
            SsdOcsiTestItem[2],
            SsdOcsiTestItem[0],
            SsdOcsiTestItem[2],
            SsdOcsiTestItem[0],
            SsdOcsiTestItem[1],
            SsdOcsiTestItem[1],
            SsdOcsiTestItem[2],
            SsdOcsiTestItem[2],
        };
        itemCount = NN_ARRAY_SIZE(Items);

        nnt::ec::PurchaseRight(g_NsaId, Items, itemCount);
    }

    nnt::ec::ConsumableServiceItemSyncRequest syncRequest;
    syncRequest.Initialize(g_User, g_LibraryWorkMemory, sizeof(g_LibraryWorkMemory));

    //! 電源断相当の権利状態を作成する

    //! 権利参照
    int rightCount = 0;
    nn::ec::ConsumableServiceItemRightData rights[MaxRightCount];
    NNT_ASSERT_RESULT_SUCCESS(syncRequest.GetConsumableRightData(&rightCount, rights, MaxRightCount, 0, MaxRightCount));

    //! 各権利消費の一次出力領域として、利用する
    int outCount = 0;
    nn::ec::ConsumableServiceItemRightData out[MaxRightCount];

    //! 権利消費（消費成功を受信できた権利）
    const nn::ec::ConsumableServiceItemRightData received[] =
    {
        rights[0],
        rights[1],
        rights[5],
        rights[6],
        rights[7],
    };
    int receivedCount = NN_ARRAY_SIZE(received);
    NNT_ASSERT_RESULT_SUCCESS(syncRequest.ConsumeRightData(&outCount, out, MaxRightCount, received, receivedCount));

    //! 消費成功を受信できた
    rights[0] = out[0];
    rights[1] = out[1];
    rights[5] = out[2];
    rights[6] = out[3];
    rights[7] = out[4];

    //! 権利消費（消費成功を受信できなかった権利）
    const nn::ec::ConsumableServiceItemRightData lost[] =
    {
        rights[2],
        rights[8],
        rights[9],
        rights[10],
        rights[11],
        rights[12],
    };
    int lostCount = NN_ARRAY_SIZE(lost);
    NNT_ASSERT_RESULT_SUCCESS(syncRequest.ConsumeRightData(&outCount, out, MaxRightCount, lost, lostCount));

    //! 消費成功を受信できなかった
    //rights[2] = out[0];
    //rights[8] = out[1];
    //rights[9] = out[2];
    //rights[10] = out[3];
    //rights[11] = out[4];
    //rights[12] = out[5];

    //! 権利消費（他の Consumption Request ID によって消費される）
    const nn::ec::ConsumableServiceItemRightData intercept[] =
    {
        rights[14],
        rights[16],
    };
    int interceptCount = NN_ARRAY_SIZE(intercept);

    //! 他デバイスからの権利消費を想定して、Consumption Request ID を再発行する
    int republishedCount = 0;
    nn::ec::ConsumableServiceItemRightData republished[MaxRightCount];
    nnt::ec::RepublishConsumptionRequestIdOfConsumableServiceItemRightData(&republishedCount, republished, MaxRightCount, intercept, interceptCount);
    NNT_ASSERT_RESULT_SUCCESS(syncRequest.ConsumeRightData(&outCount, out, MaxRightCount, republished, republishedCount));

    //! 権利消費（まだ権利消費してない権利）
    //! rights[3], [4], [13], [15], [17] には権利消費が実行されていない

    //! 評価
    const bool expected[] =
    {
        true,
        true,
        false,
        false,
        false,
        true,
        true,
        true,
        false,
        false,
        false,
        false,
        false,
        false,
        false,
        false,
        false,
        false,
    };
    int expectedCount = NN_ARRAY_SIZE(expected);

    ASSERT_EQ(expectedCount, rightCount);
    for (int i = 0; i < expectedCount; i++)
    {
        ASSERT_EQ(expected[i], rights[i].IsServiceProvidableOnLocal());
    }

    //! TODO: セーブデータコミット

    //! サーバで自分が消費済みの権利が、提供可能になっていることを期待する
    int expectedProvidableCount = receivedCount + lostCount; //! + interceptCount

    //! サーバで未消費の権利が、消費可能になっていることを期待する
    int expectedConsumableCount = itemCount - (expectedProvidableCount + interceptCount);

    EvaluateOnSaveDataConsumption(g_User, rights, rightCount, expectedProvidableCount, expectedConsumableCount);

    NNT_ASSERT_RESULT_SUCCESS(nn::ec::FinalizeForShopServiceAccessors());

} // NOLINT(impl/function_size)

//! セーブデータ・シリアライズテスト
TEST(ConsumableServiceItemApi, SaveDataSerialize)
{
    NNT_ASSERT_RESULT_SUCCESS(nn::ec::InitializeForShopServiceAccessors(g_WorkMemory, sizeof(g_WorkMemory)));
    nn::ec::ShopServiceAccessor accessor;
    NNT_ASSERT_RESULT_SUCCESS(accessor.Initialize(nn::ec::ShopService(nn::ec::ShopService::Type_OwnedConsumableServiceItem)));

    //! サーバ上の権利を全て消費する
    ConsumeAllRightData(g_User);

    //! 4 個権利購入
    const nn::ec::ConsumableServiceItemId Items[] =
    {
        SsdOcsiTestItem[2],
        SsdOcsiTestItem[0],
        SsdOcsiTestItem[0],
        SsdOcsiTestItem[1],
    };
    int itemCount = NN_ARRAY_SIZE(Items);
    nnt::ec::PurchaseRight(g_NsaId, Items, itemCount);

    nnt::ec::ConsumableServiceItemSyncRequest syncRequest;
    syncRequest.Initialize(g_User, g_LibraryWorkMemory, sizeof(g_LibraryWorkMemory));

    //! 権利参照
    int consumableCount = 0;
    nn::ec::ConsumableServiceItemRightData consumable[MaxRightCount];
    NNT_ASSERT_RESULT_SUCCESS(syncRequest.GetConsumableRightData(&consumableCount, consumable, MaxRightCount, 0, MaxRightCount));

    //! 評価
    ASSERT_EQ(itemCount, consumableCount);
    for (int i = 0; i < consumableCount; i++)
    {
        ASSERT_EQ(false, consumable[i].IsServiceProvidableOnLocal());
    }

    //! シリアライズ・デシリアライズテスト

    //! セーブデータマウント
    const char* mountName = "save";
    const char* rightDataFilePath = "save:/ConsumableServiceItemRightData";

    NNT_ASSERT_RESULT_SUCCESS(nn::fs::EnsureSaveData(g_User));
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::MountSaveData(mountName, g_User));
    NN_UTIL_SCOPE_EXIT
    {
        nn::fs::Unmount(mountName);
    };

    //! シリアライズ
    size_t requiredBufferSize = nn::ec::GetRequiredBufferSizeToSerializeConsumableServiceItemRightData(consumable, consumableCount);

    std::unique_ptr<nn::Bit8> buffer(new nn::Bit8[requiredBufferSize]);
    size_t bufferSize = nn::ec::SerializeConsumableServiceItemRightData(buffer.get(), requiredBufferSize, consumable, consumableCount);

    //! セーブデータ作成
    nn::Result result = nn::fs::CreateFile(rightDataFilePath, bufferSize);
    if (nn::fs::ResultUsableSpaceNotEnough::Includes(result) || nn::fs::ResultPathNotFound::Includes(result))
    {
        //! セーブデータを作れなかったので、強制アサート
        ASSERT_TRUE(false);
    }

    //! セーブデータ書き込み
    NNT_ASSERT_RESULT_SUCCESS(nnt::ec::WriteConsumableServiceItemRightData(rightDataFilePath, buffer.get(), bufferSize));

    //! セーブデータコミット
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::Commit(mountName));

    //! セーブデータ読み出し
    NNT_ASSERT_RESULT_SUCCESS(nnt::ec::ReadConsumableServiceItemRightData(buffer.get(), &bufferSize, rightDataFilePath));

    //! デシリアライズ
    nn::ec::SerializedConsumableServiceItemRightDataInfo info = {};
    if (!nn::ec::GetSerializedConsumableServiceItemRightDataInfo(&info, buffer.get(), bufferSize).IsSuccess())
    {
        //! エラービュワーを表示する
        ASSERT_TRUE(false);
    }

    std::unique_ptr<nn::ec::ConsumableServiceItemRightData> deserialized(new nn::ec::ConsumableServiceItemRightData[info.rightDataCount]);
    int deserializedCount = nn::ec::DeserializeConsumableServiceItemRightData(deserialized.get(), info.rightDataCount, buffer.get(), bufferSize);

    //! 評価
    ASSERT_EQ(itemCount, deserializedCount);
    for (int i = 0; i < deserializedCount; i++)
    {
        ASSERT_EQ(false, deserialized.get()[i].IsServiceProvidableOnLocal());
    }

    //! 最大権利数が、デシリアライズされた権利数より大きい
    ASSERT_TRUE(MaxRightCount >= deserializedCount);

    //! 権利消費
    int consumedRightCount = 0;
    nn::ec::ConsumableServiceItemRightData consumed[MaxRightCount];
    int count = std::min(deserializedCount, nn::ec::ConsumableServiceItemRightDataCountMax);
    NNT_ASSERT_RESULT_SUCCESS(syncRequest.ConsumeRightData(&consumedRightCount, consumed, count, consumable, consumableCount));

    //! 評価
    ASSERT_EQ(count, consumedRightCount);
    for (int i = 0; i < count; i++)
    {
        ASSERT_EQ(true, consumed[i].IsServiceProvidableOnLocal());
    }

    NNT_ASSERT_RESULT_SUCCESS(nn::ec::FinalizeForShopServiceAccessors());
}

//! RequiredWorkMemorySizeForConsumableServiceItem で 20 個の権利消費、権利問い合わせ、権利消費をテストする
TEST(ConsumableServiceItemApi, WorkMemoryTest)
{
    NNT_ASSERT_RESULT_SUCCESS(nn::ec::InitializeForShopServiceAccessors(g_WorkMemory, sizeof(g_WorkMemory)));
    NN_UTIL_SCOPE_EXIT
    {
        NNT_ASSERT_RESULT_SUCCESS(nn::ec::FinalizeForShopServiceAccessors());
    };

    //! 全権利消費
    ConsumeAllRightData(g_User);

    //! 権利購入
    const int itemCount = 20;
    {
        std::unique_ptr<nn::ec::ConsumableServiceItemId> Items(new nn::ec::ConsumableServiceItemId[itemCount]);
        for (int i = 0; i < itemCount; i++)
        {
            Items.get()[i] = SsdOcsiTestItem[0];
        }
        nnt::ec::PurchaseRight(g_NsaId, Items.get(), itemCount);
    }

    nnt::ec::ConsumableServiceItemSyncRequest syncRequest;
    syncRequest.Initialize(g_User, g_LibraryWorkMemory, sizeof(g_LibraryWorkMemory));

    //! 権利参照
    int rightDataCount = 0;
    std::unique_ptr<nn::ec::ConsumableServiceItemRightData> rightData(new nn::ec::ConsumableServiceItemRightData[itemCount]);
    NNT_ASSERT_RESULT_SUCCESS(syncRequest.GetConsumableRightData(&rightDataCount, rightData.get(), nn::ec::ConsumableServiceItemRightDataCountMax, 0, itemCount));

    //! 提供可能権利取得
    int rightDataInquiryCount = 0;
    std::unique_ptr<nn::ec::ConsumableServiceItemRightData> rightDataInquiry(new nn::ec::ConsumableServiceItemRightData[itemCount]);
    NNT_ASSERT_RESULT_SUCCESS(syncRequest.GetProvidableRightData(&rightDataInquiryCount, rightDataInquiry.get(), nn::ec::ConsumableServiceItemRightDataCountMax, rightData.get(), rightDataCount));

    //! 権利消費
    int rightDataConsumedCount = 0;
    std::unique_ptr<nn::ec::ConsumableServiceItemRightData> rightDataConsumed(new nn::ec::ConsumableServiceItemRightData[itemCount]);
    NNT_ASSERT_RESULT_SUCCESS(syncRequest.ConsumeRightData(&rightDataConsumedCount, rightDataConsumed.get(), nn::ec::ConsumableServiceItemRightDataCountMax, rightData.get(), rightDataCount));
}
