﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstring>
#include <cstdlib>
#include <cstdio>
#include <getopt.h>
#include <nn/os.h>
#include <nn/init.h>
#include <nn/mem.h>
#include <nn/fs.h>
#include <nn/lbl/lbl.h>
#include <nn/vi/vi_ProxyName.private.h>
#include <nn/nn_Log.h>
#include <nn/nn_Assert.h>

#if !defined( NN_BUILD_CONFIG_OS_SUPPORTS_WIN32 )
#include <unistd.h>
#endif
#if defined( NN_BUILD_CONFIG_OS_SUPPORTS_HORIZON ) && defined( NN_BUILD_CONFIG_SPEC_NX )
    #include <nv/nv_MemoryManagement.h>
    #include <nv/nv_ServiceName.h>
    #include <nvnTool/nvnTool_GlslcInterface.h>
#endif

#include <Common/GfxJpegViewer.h>
#include <Common/Ui2dSimple.h>
#include <Common/SetupGraphics.h>
#include <Common/VideoConfigUtility.h>
#include <Common/ProgramOptionParser.h>
#include <Common/ConfigSwitcher.h>
#include <Common/FilePathViewer.h>
#include "BrightnessListener.h"
#include "LcdConfig.h"

namespace {

const int CharBufferLength = 512;
nn::mem::StandardAllocator g_Allocator;
NN_ALIGNAS(4096) char g_FsBuffer[8 * 1024 * 1024];

const char* PresetImageSource[] =
{
    "Contents:/CB720p.bmp",
    "Contents:/RMP1080p.bmp",
    "Contents:/Image.jpg",
    "Contents:/oxtonguerapids_T1080.bmp",
    "ui2dsimple",
};

void* Allocate(size_t size)
{
    return g_Allocator.Allocate(size);
}

void Deallocate(void* p, size_t size)
{
    NN_UNUSED(size);
    g_Allocator.Free(p);
}

#if defined( NN_BUILD_CONFIG_OS_SUPPORTS_HORIZON ) && defined( NN_BUILD_CONFIG_SPEC_NX )
//------------------------------------------------------------------------------
// グラフィックスシステム用メモリ割り当て・破棄関数
//------------------------------------------------------------------------------
static void* NvAllocateFunction(size_t size, size_t alignment, void* userPtr)
{
    NN_UNUSED(userPtr);
    return aligned_alloc(alignment, size);
}
static void NvFreeFunction(void* addr, void* userPtr)
{
    NN_UNUSED(userPtr);
    free(addr);
}
static void* NvReallocateFunction(void* addr, size_t newSize, void* userPtr)
{
    NN_UNUSED(userPtr);
    return realloc(addr, newSize);
}
#endif

void ShowOptionalSetting(bool isAutoControl, bool isVrrEnabled,
                         float brightness, int vfp, int performanceIntervalSec, uint64_t shiftIntervalMSec,
                         const char* imageName)
{
    NN_LOG("  - brightness     :[%.2f (=%.1f percent)]\n", brightness, 100.0f * brightness);
    NN_LOG("  - auto bright    :[%s]\n", (isAutoControl ? "true" : "false"));
    NN_LOG("  - VRR            :[%s]\n", (isVrrEnabled ? "enabled" : "disabled"));
    if (0 <= vfp)
    {
        NN_LOG("  - VFP            :[%d]\n", vfp);
    }
    else
    {
        NN_LOG("  - VFP            :[presetting]\n");
    }
    NN_LOG("  - perf interval  :[%d]\n", performanceIntervalSec);
    NN_LOG("  - shift interval :[%lld]\n", shiftIntervalMSec);
    NN_LOG("  - image path     :[%s]\n", imageName);
    NN_LOG("=============================================\n");
}

}

#if defined( NN_BUILD_CONFIG_OS_SUPPORTS_HORIZON )
//-----------------------------------------------------------------------------
// nninitStartup() is invoked before calling nnMain().
//
extern "C" void nninitStartup()
{
    const size_t MallocMemorySize = 512 * 1024 * 1024;
    nn::Result result = nn::os::SetMemoryHeapSize( 768 * 1024 * 1024 );
    uintptr_t address;
    result = nn::os::AllocateMemoryBlock( &address, MallocMemorySize );
    NN_ASSERT( result.IsSuccess() );
    nn::init::InitializeAllocator( reinterpret_cast<void*>( address ), MallocMemorySize );
}
#endif



extern "C" void nnMain()
{
    NN_LOG("=============================================\n");
    NN_LOG("  - Build date:[%s][%s]\n", __DATE__, __TIME__);
    NN_LOG("=============================================\n");

    nns::ProgramOptionParser parser;
    parser.AddOption('a', "auto_backlight", required_argument,
        "auto backlight (if is_looping > 0, will control backlight automatically).");
    parser.AddOption('d', "brightness", required_argument,
        "brightness level (0.0 <= brightness).");
    parser.AddOption('e', "vrr", required_argument,
        "enable VRR (if is_vrr_enabled > 0, will be enabled).");
    parser.AddOption('f', "vfp", required_argument, "VFP.");
    parser.AddOption('p', "performance", required_argument,
        "interval time to show performance (0 <= interval_msec).");
    parser.AddOption('s', "bright_shift", required_argument,
        "interval time to set brightness (0 <= interval_msec).");
    parser.AddOption('i', "image", required_argument,
        "showed image path (<param> is .jpeg path from executable file or image type name (\"ui2dsimple\")).");
    parser.AddOption('o', "host", required_argument, "searched host(PC) path.");
    parser.AddOption('q', "font", required_argument, "show setting fonts when booting. (if is_showing > 0, will show setting fonts).");
    if (!parser.ParseOption())
    {
        NN_LOG("=============================================\n");
        parser.ShowHelp();
        NN_ABORT("End of %s\n", __FILE__);
    }

    // Use full duty ratio defaultly.
    float brightness = parser.GetFloatOptionalValue('d', 1.0f);
    bool isVrrEnabled = parser.GetBooleanOptionalValue('e', false);
    bool isFontVisible = parser.GetBooleanOptionalValue('q', false);
    int vfp = parser.GetIntegerOptionalValue('f', -1);
    int performanceIntervalSec = parser.GetIntegerOptionalValue('p', 0);
    uint64_t shiftIntervalMSec = parser.GetIntegerOptionalValue('s', 0);
    std::string targetImageName = parser.GetStringOptionalValue('i', "CB720p.bmp");
    std::string hostPath = parser.GetStringOptionalValue('o', "C:/image");

    // nn::vi と nvnflinger を共存させるための設定
    nn::vi::SetRelayProxyName("dispsys");

    // 現在の輝度設定値の読み込み。
    nn::lbl::Initialize();
    nn::lbl::LoadCurrentSetting();
    bool isAutoControl = parser.GetBooleanOptionalValue('a', nn::lbl::IsAutoBrightnessControlEnabled());

    if (brightness < 0.0f)
    {
        brightness = nn::lbl::GetCurrentBrightnessSetting();
    }
    else
    {
        // 輝度設定値を指定しているときは輝度設定値を変化させない。
        shiftIntervalMSec = 0;
    }
    if (isAutoControl)
    {
        // 自動輝度調整時は輝度設定値を変化させない。
        shiftIntervalMSec = 0;
    }

    // 設定する情報を出力。
    ShowOptionalSetting(isAutoControl, isVrrEnabled,
        brightness, vfp, performanceIntervalSec, shiftIntervalMSec,
        targetImageName.c_str());

#if defined( NN_BUILD_CONFIG_OS_SUPPORTS_HORIZON ) && defined( NN_BUILD_CONFIG_SPEC_NX )
    // グラフィックスシステムのためのメモリ周りの初期化を行います。
    {
        const size_t GraphicsSystemMemorySize = 8 * 1024 * 1024;
        nv::SetGraphicsAllocator(NvAllocateFunction, NvFreeFunction, NvReallocateFunction, NULL);
        nv::SetGraphicsDevtoolsAllocator(NvAllocateFunction, NvFreeFunction, NvReallocateFunction, NULL);
        nv::SetGraphicsServiceName("nvdrv:t");
        nv::InitializeGraphics( malloc( GraphicsSystemMemorySize ), GraphicsSystemMemorySize );
        glslcSetAllocator(NvAllocateFunction, NvFreeFunction, NvReallocateFunction, NULL);
    }
#endif

    g_Allocator.Initialize(g_FsBuffer, sizeof(g_FsBuffer));
    nn::fs::SetAllocator(Allocate, Deallocate);

    // 第一引数が1以上のとき、適用される輝度レベルを定期的に出力する。
    nns::BrightnessListener listener(0, shiftIntervalMSec);

    nns::LcdConfig config(isAutoControl, isVrrEnabled, brightness, vfp, performanceIntervalSec);
    config.ApplySelectedLabel();
    nns::ConfigSwitcher switcher(&config, isFontVisible);
    switcher.CreateInputListener();

    // host PC と SDカード内にある画像ファイルを検索。
    nns::FilePathViewer pathViewer(hostPath.c_str());
    std::vector<std::string> imagePath;
    pathViewer.SearchImageFileFromAttachedDevice(&imagePath);

    // 読めなかったらプリセットの画像を突っ込む。
    if (0 == imagePath.size())
    {
        for (auto preset : PresetImageSource)
        {
            imagePath.push_back(preset);
        }
    }

    // 指定したファイルに最初に出す絵をあわせる。
    auto iter = std::find(imagePath.begin(), imagePath.end(), targetImageName);
    auto targetIndex = std::distance(imagePath.begin(), iter) % imagePath.size();

    // 描画するイメージのパスを表示。
    for (int i1=0; i1<imagePath.size(); ++i1)
    {
        NN_LOG("%s [%2d/%2d]:[%s]\n", (i1 == targetIndex) ? "*" : " ", i1, imagePath.size(), imagePath[i1].c_str());
    }
    NN_LOG("=============================================\n");

    nns::SetupGraphics gfxSetup;

    // 描画。入力に合わせて絵を順次切り替え。
    for (int i1=0; ; ++i1)
    {
        const std::string& image = imagePath[targetIndex];
        if ("ui2dsimple" == image)
        {
            nns::ShowMovingPictures(&gfxSetup, &switcher);
        }
        else
        {
            nns::ShowImage(image.c_str(), &gfxSetup, &switcher);
        }
        // 絵の切り替える先を決定。
        if (config.IsSwitchingSceneAscending())
        {
            targetIndex = (targetIndex + 1) % imagePath.size();
        }
        else
        {
            targetIndex = (targetIndex <= 0) ? imagePath.size() - 1 : targetIndex - 1;
        }
    }

    nn::lbl::Finalize();

    NN_LOG("End of %s\n", __FILE__);
} // NOLINT(impl/function_size)
