﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <memory>
#include <vector>
#include <string>
#include <nn/nn_Macro.h>
#include <Common/DisplayConfig.h>
#include <Common/VideoConfigUtility.h>
#include <Common/AudioRenderer.h>
#include <Common/ConfigSwitcher.h>
#include "HdcpManager.h"
#include "CecManager.h"

namespace nns {

class HdmiConfig : public DisplayConfig
{
private:
    enum HdcpState
    {
        HdcpState_Done,
        HdcpState_Going,
        HdcpState_Failed,
        HdcpState_Unpluged,
    };
    HdcpState m_State;
    bool m_IsHdcpSetEnabled;
    bool m_IsLabelResetting;
    bool m_IsStubEmulationEnabled;
    int m_HdcpPollingCount;
    int m_ConfigMode;
    int m_PreVideoConfigs;
    int m_VideoConfigSeries;
    int m_HdcpAuthenticationTimeout;
    int64_t m_HdcpStartMilliSeconds;
    int64_t m_HdcpElapsedMilliSeconds;
    static const int LabelStringLength = 64;
    char m_LabelString[LabelStringLength];

    typedef void (HdmiConfig::*SettingFunction)(int);
    struct SettingContext
    {
        struct State
        {
            const std::string label;
            const int level;
        };
        SettingFunction switchState;
        typedef const char* (HdmiConfig::*StatusFunction)(void);
        StatusFunction getStatus;
        const char* caption;
        std::vector<State> state;
        int stateIndex;
        bool isStateIndexChanged;
    };
    std::vector<SettingContext> m_Setting;
    VideoConfigUtility m_Video;
    HdcpManager m_Hdcp;
    CecManager m_Cec;
    Color m_BackGroundColor;

    char m_StereoAudioContent[2][CharBufferLength];
    char m_SurroundAudioContent[6][CharBufferLength];

    void SwitchVideoConfig(int configMode) NN_NOEXCEPT;
    void SwitchVideoConfigSeries(int series) NN_NOEXCEPT;
    void SwitchRgbRange(int rgbRange) NN_NOEXCEPT;
    void SwitchContentType(int contentType) NN_NOEXCEPT;
    void SwitchHdcp(int isEnabled) NN_NOEXCEPT;
    void SetHdcpAuthenticationTimeout(int msec) NN_NOEXCEPT;
    void SwitchHdcpStubEmulation(int isEnabled) NN_NOEXCEPT;
    void SwitchAudio(int audioType) NN_NOEXCEPT;
    void SendCecCommand(int cecCommandType) NN_NOEXCEPT;
    void SendCecVolumeCommand(int cecCommandType) NN_NOEXCEPT;
    void SwitchLabelResetting(int isEnabled) NN_NOEXCEPT;
    const char* GetHdcpStatus() NN_NOEXCEPT;
    const char* GetCecStatus() NN_NOEXCEPT;

    std::vector<SettingContext::State> GetVideoConfigState(int configMode, bool isListUpAll) NN_NOEXCEPT;
    void AssignVideoConfigState(std::vector<SettingContext::State>* pSettingState, bool isListUpAll) NN_NOEXCEPT;
    void AssignHdcpState(std::vector<SettingContext::State>* pSettingState, bool isHdcpEnabled) NN_NOEXCEPT;
    void ResetVideoConfigsState(std::vector<SettingContext::State>& settingState, int configMode, bool isInserted) NN_NOEXCEPT;
    int GetSelectedConfigModeStateIndex() NN_NOEXCEPT;
    bool IsSameVideoConfigState(const std::vector<SettingContext::State>& lhs, const std::vector<SettingContext::State>& rhs) NN_NOEXCEPT;
    int FindSettingStateIndexFromLevel(std::vector<SettingContext::State>& settingState, int level) NN_NOEXCEPT;

public:
    NN_IMPLICIT HdmiConfig(bool isHdcpEnabled, int edidVicId, int configMode, int rgbRange,
        int contentType, const std::vector<std::string>& waveFilePath) NN_NOEXCEPT;
    ~HdmiConfig() NN_NOEXCEPT;
    void ApplySelectedLabel() NN_NOEXCEPT;
    int GetConfigCount() NN_NOEXCEPT;
    const char* GetConfigCaption(int index) NN_NOEXCEPT;
    const char* GetConfigStatus(int index) NN_NOEXCEPT;
    const char* GetConfigLabel(int index) NN_NOEXCEPT;
    void SetConfig(int index, bool isAscending) NN_NOEXCEPT;
    void UpdateConfig() NN_NOEXCEPT;
    bool IsSceneSwitched() NN_NOEXCEPT;
    const Color& GetBackGroundColor() NN_NOEXCEPT;
    void SetBackGroundColor(float rRatio, float gRatio, float bRatio, float aRatio) NN_NOEXCEPT;
};

}
