﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nnt.h>
#include <nnt/result/testResult_Assert.h>
#include <nn/vi/vi_Result.h>
#include "master/visrv_Layer.h"
#include "testDisplay_LayerFixture.h"

// Note: These aren't great tests since they're really dependent on behavior from
//       SurfaceFlinger.  However, this should be an ok check to ensure stability.

typedef LayerFixture Layer;

TEST_F(Layer, Ctor_Any)
{
    nn::visrv::master::Layer layer(GetDisplay(), nn::vi::PolicyLevel_Standard);
    EXPECT_EQ(GetDisplay(), layer.GetDisplay());
    EXPECT_EQ(GetDisplay()->GetInfo(nn::vi::PolicyLevel_Standard).GetPolicy().GetZOrderCountMin(), layer.GetZ());
    EXPECT_EQ(0, layer.GetCurrentWidth());
    EXPECT_EQ(0, layer.GetCurrentHeight());
    EXPECT_EQ(0.f, layer.GetX());
    EXPECT_EQ(0.f, layer.GetY());
    EXPECT_TRUE(layer.GetSettings().IsAllOff());
    EXPECT_EQ(1, layer.GetStacks());
}

TEST_F(Layer, Initialize_Any)
{
    nn::visrv::master::Layer layer(GetDisplay(), nn::vi::PolicyLevel_Standard);

    nn::vi::LayerSettings settings;
    settings.Set();

    NNT_ASSERT_RESULT_SUCCESS(layer.Initialize(settings, nn::applet::GetAppletResourceUserId()));

    int width;
    int height;
    NNT_ASSERT_RESULT_SUCCESS(GetDisplay()->GetLogicalResolution(&width, &height));

    EXPECT_EQ(width, layer.GetCurrentWidth());
    EXPECT_EQ(height, layer.GetCurrentHeight());
    EXPECT_TRUE(layer.GetSettings().IsAllOn());
}

TEST_F(Layer, Initialize_NoFullscreen_Any)
{
    nn::visrv::master::Layer layer(GetDisplay(), nn::vi::PolicyLevel_Standard);

    nn::vi::LayerSettings settings;
    settings.Reset();

    NNT_ASSERT_RESULT_SUCCESS(layer.Initialize(settings, nn::applet::GetAppletResourceUserId()));

    EXPECT_EQ(0, layer.GetCurrentWidth());
    EXPECT_EQ(0, layer.GetCurrentHeight());
    EXPECT_TRUE(layer.GetSettings().IsAllOff());
}

TEST_F(Layer, SetPosition_Any)
{
    nn::visrv::master::Layer layer(GetDisplay(), nn::vi::PolicyLevel_Standard);
    NNT_ASSERT_RESULT_SUCCESS(layer.Initialize(nn::vi::LayerSettings().Reset(), nn::applet::GetAppletResourceUserId()));

    int width;
    int height;
    NNT_ASSERT_RESULT_SUCCESS(GetDisplay()->GetLogicalResolution(&width, &height));

    // testing just inside the boundaries
    NNT_EXPECT_RESULT_SUCCESS(layer.SetPosition(0.f, 0.f));
    EXPECT_EQ(0.f, layer.GetX());
    EXPECT_EQ(0.f, layer.GetY());

    NNT_EXPECT_RESULT_SUCCESS(layer.SetPosition(width - 1.f, 0.f));
    EXPECT_EQ(width - 1.f, layer.GetX());
    EXPECT_EQ(0.f, layer.GetY());

    NNT_EXPECT_RESULT_SUCCESS(layer.SetPosition(0.f, height - 2.f));
    EXPECT_EQ(0.f, layer.GetX());
    EXPECT_EQ(height - 2.f, layer.GetY());
}

TEST_F(Layer, DISABLED_SetPosition_InvalidRange_Any)
{
    nn::visrv::master::Layer layer(GetDisplay(), nn::vi::PolicyLevel_Standard);
    NNT_ASSERT_RESULT_SUCCESS(layer.Initialize(nn::vi::LayerSettings().Reset(), nn::applet::GetAppletResourceUserId()));

    nn::vi::DisplayModeInfo mode;
    NNT_ASSERT_RESULT_SUCCESS(GetDisplay()->GetMode(&mode));

    // test boundaries
    NNT_EXPECT_RESULT_FAILURE(nn::vi::ResultInvalidRange, layer.SetPosition(-1.f, 0.f));
    EXPECT_EQ(0.f, layer.GetX());
    EXPECT_EQ(0.f, layer.GetY());

    NNT_EXPECT_RESULT_FAILURE(nn::vi::ResultInvalidRange, layer.SetPosition(0.f, -1.f));
    EXPECT_EQ(0.f, layer.GetX());
    EXPECT_EQ(0.f, layer.GetY());

    NNT_EXPECT_RESULT_FAILURE(nn::vi::ResultInvalidRange, layer.SetPosition(mode.width, 0.f));
    EXPECT_EQ(0.f, layer.GetX());
    EXPECT_EQ(0.f, layer.GetY());

    NNT_EXPECT_RESULT_FAILURE(nn::vi::ResultInvalidRange, layer.SetPosition(0.f, mode.height));
    EXPECT_EQ(0.f, layer.GetX());
    EXPECT_EQ(0.f, layer.GetY());
}

TEST_F(Layer, SetZ_Any)
{
    nn::visrv::master::Layer layer(GetDisplay(), nn::vi::PolicyLevel_Standard);
    NNT_ASSERT_RESULT_SUCCESS(layer.Initialize(nn::vi::LayerSettings().Reset(), nn::applet::GetAppletResourceUserId()));

    int z = layer.GetDisplay()->GetInfo(nn::vi::PolicyLevel_Standard).GetPolicy().GetZOrderCountMin();
    NNT_ASSERT_RESULT_SUCCESS(layer.SetZ(z));
    ASSERT_EQ(z, layer.GetZ());

    z = layer.GetDisplay()->GetInfo(nn::vi::PolicyLevel_Standard).GetPolicy().GetZOrderCountMax();
    NNT_ASSERT_RESULT_SUCCESS(layer.SetZ(z));
    ASSERT_EQ(z, layer.GetZ());
}

TEST_F(Layer, SetZ_InvalidRange_Any)
{
    nn::visrv::master::Layer layer(GetDisplay(), nn::vi::PolicyLevel_Standard);
    NNT_ASSERT_RESULT_SUCCESS(layer.Initialize(nn::vi::LayerSettings().Reset(), nn::applet::GetAppletResourceUserId()));

    // this should stay constant throughout the test
    int z = layer.GetZ();

    // z-order lower than policy allows
    NNT_EXPECT_RESULT_FAILURE(nn::vi::ResultInvalidRange, layer.SetZ(0));
    EXPECT_EQ(z, layer.GetZ());

    // z-order higher than policy allows
    NNT_EXPECT_RESULT_FAILURE(nn::vi::ResultInvalidRange, layer.SetZ(3));
    EXPECT_EQ(z, layer.GetZ());
}

TEST_F(Layer, SetSize_Any)
{
    nn::visrv::master::Layer layer(GetDisplay(), nn::vi::PolicyLevel_Standard);
    NNT_ASSERT_RESULT_SUCCESS(layer.Initialize(nn::vi::LayerSettings().Reset(), nn::applet::GetAppletResourceUserId()));

    int width;
    int height;
    NNT_ASSERT_RESULT_SUCCESS(GetDisplay()->GetLogicalResolution(&width, &height));

    // test just inside the boundaries
    NNT_ASSERT_RESULT_SUCCESS(layer.SetSize(width, 0));
    EXPECT_EQ(width, layer.GetCurrentWidth());
    EXPECT_EQ(0, layer.GetCurrentHeight());

    NNT_ASSERT_RESULT_SUCCESS(layer.SetSize(0, height));
    EXPECT_EQ(0, layer.GetCurrentWidth());
    EXPECT_EQ(height, layer.GetCurrentHeight());

    NNT_ASSERT_RESULT_SUCCESS(layer.SetSize(width, height));
    EXPECT_EQ(width, layer.GetCurrentWidth());
    EXPECT_EQ(height, layer.GetCurrentHeight());

    NNT_ASSERT_RESULT_SUCCESS(layer.SetSize(0, 0));
    EXPECT_EQ(0, layer.GetCurrentWidth());
    EXPECT_EQ(0, layer.GetCurrentHeight());
}

TEST_F(Layer, SetSize_InvalidRange_Any)
{
    nn::visrv::master::Layer layer(GetDisplay(), nn::vi::PolicyLevel_Standard);
    NNT_ASSERT_RESULT_SUCCESS(layer.Initialize(nn::vi::LayerSettings().Reset(), nn::applet::GetAppletResourceUserId()));

    int width;
    int height;
    NNT_ASSERT_RESULT_SUCCESS(GetDisplay()->GetLogicalResolution(&width, &height));

    // test boundaries
    NNT_ASSERT_RESULT_FAILURE(nn::vi::ResultInvalidRange, layer.SetSize(width + 1, 0));
    EXPECT_EQ(0, layer.GetCurrentWidth());
    EXPECT_EQ(0, layer.GetCurrentHeight());

    NNT_ASSERT_RESULT_FAILURE(nn::vi::ResultInvalidRange, layer.SetSize(0, height + 1));
    EXPECT_EQ(0, layer.GetCurrentWidth());
    EXPECT_EQ(0, layer.GetCurrentHeight());

    // move to non-trivial position
    NNT_ASSERT_RESULT_SUCCESS(layer.SetPosition(1, 2));

    NNT_ASSERT_RESULT_FAILURE(nn::vi::ResultInvalidRange, layer.SetSize(width, 1));
    EXPECT_EQ(0, layer.GetCurrentWidth());
    EXPECT_EQ(0, layer.GetCurrentHeight());

    NNT_ASSERT_RESULT_FAILURE(nn::vi::ResultInvalidRange, layer.SetSize(1, height - 1));
    EXPECT_EQ(0, layer.GetCurrentWidth());
    EXPECT_EQ(0, layer.GetCurrentHeight());
}

TEST_F(Layer, SetVisibility_Any)
{
    nn::visrv::master::Layer layer(GetDisplay(), nn::vi::PolicyLevel_Standard);
    NNT_ASSERT_RESULT_SUCCESS(layer.Initialize(nn::vi::LayerSettings().Reset(), nn::applet::GetAppletResourceUserId()));
    NNT_ASSERT_RESULT_SUCCESS(layer.SetVisibility(true));
}

TEST_F(Layer, SetAlpha_Any)
{
    nn::visrv::master::Layer layer(GetDisplay(), nn::vi::PolicyLevel_Standard);
    NNT_ASSERT_RESULT_SUCCESS(layer.Initialize(nn::vi::LayerSettings().Reset(), nn::applet::GetAppletResourceUserId()));
    NNT_ASSERT_RESULT_SUCCESS(layer.SetAlpha(0.f));
}

TEST_F(Layer, GetNativeWindow_Any)
{
    nn::visrv::master::Layer layer(GetDisplay(), nn::vi::PolicyLevel_Standard);
    NNT_ASSERT_RESULT_SUCCESS(layer.Initialize(nn::vi::LayerSettings().Reset(), nn::applet::GetAppletResourceUserId()));
    ASSERT_NE(nullptr, layer.GetNativeWindow());
}

TEST_F(Layer, AddToStack_Any)
{
    nn::visrv::master::Layer layer(GetDisplay(), nn::vi::PolicyLevel_Standard);
    NNT_ASSERT_RESULT_SUCCESS(layer.Initialize(nn::vi::LayerSettings().Reset(), nn::applet::GetAppletResourceUserId()));

    NNT_EXPECT_RESULT_SUCCESS(layer.AddToStack(nn::vi::LayerStack_Lcd));
    EXPECT_EQ(3, layer.GetStacks());
}

TEST_F(Layer, AddToStack_InvalidStack_Any)
{
    nn::visrv::master::Layer layer(GetDisplay(), nn::vi::PolicyLevel_Standard);
    NNT_ASSERT_RESULT_SUCCESS(layer.Initialize(nn::vi::LayerSettings().Reset(), nn::applet::GetAppletResourceUserId()));

    NNT_EXPECT_RESULT_FAILURE(nn::vi::ResultInvalidRange, layer.AddToStack(static_cast<nn::vi::LayerStack>(std::numeric_limits<std::underlying_type<nn::vi::LayerStack>::type>::max())));
    EXPECT_EQ(1, layer.GetStacks());
}

TEST_F(Layer, RemoveFromStack_Any)
{
    nn::visrv::master::Layer layer(GetDisplay(), nn::vi::PolicyLevel_Standard);
    NNT_ASSERT_RESULT_SUCCESS(layer.Initialize(nn::vi::LayerSettings().Reset(), nn::applet::GetAppletResourceUserId()));

    NNT_EXPECT_RESULT_SUCCESS(layer.RemoveFromStack(nn::vi::LayerStack_Default));
    EXPECT_EQ(0, layer.GetStacks());
}

TEST_F(Layer, RemoveFromStack_InvalidStack_Any)
{
    nn::visrv::master::Layer layer(GetDisplay(), nn::vi::PolicyLevel_Standard);
    NNT_ASSERT_RESULT_SUCCESS(layer.Initialize(nn::vi::LayerSettings().Reset(), nn::applet::GetAppletResourceUserId()));

    NNT_EXPECT_RESULT_FAILURE(nn::vi::ResultInvalidRange, layer.RemoveFromStack(static_cast<nn::vi::LayerStack>(std::numeric_limits<std::underlying_type<nn::vi::LayerStack>::type>::max())));
    EXPECT_EQ(1, layer.GetStacks());
}
