﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nnt.h>
#include <nn/vi/vi_DisplayModeInfo.h>
#include <nn/vi/vi_ScalingMode.h>
#include <nn/vi/vi_PixelFormat.h>
#include <nn/vi/vi_PowerState.h>
#include <nn/vi/vi_RgbRange.h>
#include <hardware/hwcomposer_defs.h>
#include "master/detail/visrv_Android.h"

using nn::visrv::master::detail::operator==;

TEST(ConvertScalingMode, None_Any)
{
    EXPECT_EQ(NATIVE_WINDOW_SCALING_MODE_NO_SCALE_CROP, nn::visrv::master::detail::ConvertScalingMode(nn::vi::ScalingMode_None));
}

TEST(ConvertScalingMode, Exact_Any)
{
    EXPECT_EQ(NATIVE_WINDOW_SCALING_MODE_FREEZE, nn::visrv::master::detail::ConvertScalingMode(nn::vi::ScalingMode_Exact));
}

TEST(ConvertScalingMode, FitToLayer_Any)
{
    EXPECT_EQ(NATIVE_WINDOW_SCALING_MODE_SCALE_TO_WINDOW, nn::visrv::master::detail::ConvertScalingMode(nn::vi::ScalingMode_FitToLayer));
}

TEST(ConvertScalingMode, ScaleAndCrop_Any)
{
    EXPECT_EQ(NATIVE_WINDOW_SCALING_MODE_SCALE_CROP, nn::visrv::master::detail::ConvertScalingMode(nn::vi::ScalingMode_ScaleAndCrop));
}

TEST(ConvertScalingMode, PreserveAspectRatio_Any)
{
    EXPECT_EQ(NATIVE_WINDOW_SCALING_MODE_SCALE_NO_CROP, nn::visrv::master::detail::ConvertScalingMode(nn::vi::ScalingMode_PreserveAspectRatio));
}

TEST(DisplayModeInfo, WriteMode_Any)
{
    nn::vi::DisplayModeInfo mode;
    memset(&mode, 0, sizeof(mode));
    android::DisplayInfo info;
    memset(&info, 0, sizeof(info));

    const int Width = 1920;
    const int Height = 1080;
    const float RefreshRate = 59.94;
    const nn::vi::StereoMode Stereo = nn::vi::StereoMode_None;

    info.w = Width;
    info.h = Height;
    info.fps = RefreshRate;

    nn::visrv::master::detail::ConvertDisplayMode(&mode, info);

    EXPECT_EQ(Width, mode.width);
    EXPECT_EQ(Height, mode.height);
    EXPECT_EQ(RefreshRate, mode.refreshRate);
    EXPECT_EQ(Stereo, mode.mode);
}

TEST(DisplayModeInfo, Equal_Any)
{
    nn::vi::DisplayModeInfo mode;
    memset(&mode, 0, sizeof(mode));
    android::DisplayInfo info;
    memset(&info, 0, sizeof(info));

    const int Width = 1920;
    const int Height = 1080;
    const float RefreshRate = 59.94;

    info.w = Width;
    info.h = Height;
    info.fps = RefreshRate;

    mode.width = Width;
    mode.height = Height;
    mode.refreshRate = RefreshRate;
    mode.mode = nn::vi::StereoMode_None;

    EXPECT_TRUE(mode == info);
    EXPECT_TRUE(info == mode);
}

TEST(DisplayModeInfo, NotEqual_Any)
{
    nn::vi::DisplayModeInfo mode;
    memset(&mode, 0, sizeof(mode));
    android::DisplayInfo info;
    memset(&info, 0, sizeof(info));

    const int Width = 1920;
    const int Height = 1080;
    const float RefreshRate = 59.94;

    info.w = Width;
    info.h = Height;
    info.fps = RefreshRate;

    // different refresh rate
    mode.width = Width;
    mode.height = Height;
    mode.refreshRate = 60.f;

    EXPECT_FALSE(mode == info);
    EXPECT_FALSE(info == mode);

    // different height
    mode.height = 720;
    mode.refreshRate = RefreshRate;

    EXPECT_FALSE(mode == info);
    EXPECT_FALSE(info == mode);

    // different width
    mode.width = 1280;
    mode.height = Height;

    EXPECT_FALSE(mode == info);
    EXPECT_FALSE(info == mode);
}

TEST(ConvertPowerState, PowerState_Off_Any)
{
    EXPECT_EQ(HWC_POWER_MODE_OFF, nn::visrv::master::detail::ConvertPowerState(nn::vi::PowerState_Off));
}

TEST(ConvertPowerState, PowerState_On_Any)
{
    EXPECT_EQ(HWC_POWER_MODE_NORMAL, nn::visrv::master::detail::ConvertPowerState(nn::vi::PowerState_On));
}

TEST(ConvertPowerState, PowerState_Blank_Any)
{
    EXPECT_EQ(HWC_POWER_MODE_PREPARE, nn::visrv::master::detail::ConvertPowerState(nn::vi::PowerState_Blank));
}

TEST(ConvertCmuMode, CmuMode_Disabled_Any)
{
    EXPECT_EQ(DISPLAY_CMU_Disabled, nn::visrv::master::detail::ConvertCmuMode(nn::vi::CmuMode_Disabled));
}

TEST(ConvertCmuMode, CmuMode_Default_Any)
{
    EXPECT_EQ(DISPLAY_CMU_Normal, nn::visrv::master::detail::ConvertCmuMode(nn::vi::CmuMode_Default));
}

TEST(ConvertCmuMode, CmuMode_InvertColor_Any)
{
    EXPECT_EQ(DISPLAY_CMU_ColorInvert, nn::visrv::master::detail::ConvertCmuMode(nn::vi::CmuMode_InvertColor));
}

TEST(ConvertCmuMode, CmuMode_HighContrast_Any)
{
    EXPECT_EQ(DISPLAY_CMU_HighContrast, nn::visrv::master::detail::ConvertCmuMode(nn::vi::CmuMode_HighContrast));
}

TEST(ConvertCmuMode, CmuMode_GrayScale_Any)
{
    EXPECT_EQ(DISPLAY_CMU_GrayScale, nn::visrv::master::detail::ConvertCmuMode(nn::vi::CmuMode_Grayscale));
}

TEST(ConvertCmuMode, DISPLAY_CMU_Disabled_Any)
{
    nn::vi::CmuMode mode;
    ASSERT_TRUE(nn::visrv::master::detail::ConvertCmuMode(&mode, DISPLAY_CMU_Disabled));
    EXPECT_EQ(nn::vi::CmuMode_Disabled, mode);
}

TEST(ConvertCmuMode, DISPLAY_CMU_Normal_Any)
{
    nn::vi::CmuMode mode;
    ASSERT_TRUE(nn::visrv::master::detail::ConvertCmuMode(&mode, DISPLAY_CMU_Normal));
    EXPECT_EQ(nn::vi::CmuMode_Default, mode);
}

TEST(ConvertCmuMode, DISPLAY_CMU_ColorInvert_Any)
{
    nn::vi::CmuMode mode;
    ASSERT_TRUE(nn::visrv::master::detail::ConvertCmuMode(&mode, DISPLAY_CMU_ColorInvert));
    EXPECT_EQ(nn::vi::CmuMode_InvertColor, mode);
}

TEST(ConvertCmuMode, DISPLAY_CMU_HighContrast)
{
    nn::vi::CmuMode mode;
    ASSERT_TRUE(nn::visrv::master::detail::ConvertCmuMode(&mode, DISPLAY_CMU_HighContrast));
    EXPECT_EQ(nn::vi::CmuMode_HighContrast, mode);
}

TEST(ConvertCmuMode, DISPLAY_CMU_Grayscale)
{
    nn::vi::CmuMode mode;
    ASSERT_TRUE(nn::visrv::master::detail::ConvertCmuMode(&mode, DISPLAY_CMU_GrayScale));
    EXPECT_EQ(nn::vi::CmuMode_Grayscale, mode);
}

TEST(ConvertRgbRange, RgbRange_Full)
{
    EXPECT_EQ(DISPLAY_RGB_Full, nn::visrv::master::detail::ConvertRgbRangeToAndroid(nn::vi::RgbRange_Full));
}

TEST(ConvertRgbRange, RgbRange_Limited)
{
    EXPECT_EQ(DISPLAY_RGB_Limited, nn::visrv::master::detail::ConvertRgbRangeToAndroid(nn::vi::RgbRange_Limited));
}

TEST(ConvertRgbRange, DISPLAY_RGB_Full)
{
    nn::vi::RgbRange range;
    ASSERT_TRUE(nn::visrv::master::detail::ConvertRgbRangeToVi(&range, DISPLAY_RGB_Full));
    EXPECT_EQ(nn::vi::RgbRange_Full, range);
}

TEST(ConvertRgbRange, DISPLAY_RGB_Limited)
{
    nn::vi::RgbRange range;
    ASSERT_TRUE(nn::visrv::master::detail::ConvertRgbRangeToVi(&range, DISPLAY_RGB_Limited));
    EXPECT_EQ(nn::vi::RgbRange_Limited, range);
}
