﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nnt.h>
#include <nn/nn_TimeSpan.h>
#include <nn/os.h>
#include <nn/vi.private.h>
#include <nn/vi/vi_DisplayConfig.h>
#include <nn/vi/vi_DisplayEvents.h>
#include <nn/vi/vi_CmuLuma.h>
#include <nnt/result/testResult_Assert.h>
#include <nnt/viUtil/testVi_Fixture.h>
#include <nnt/viUtil/testVi_PowerFixture.h>

namespace
{
    class InternalDisplay : public nnt::vi::util::Fixture,
                            public nnt::vi::util::PowerFixture
    {
    };
}

TEST_F(InternalDisplay, SetDisplayUnderscan_IcosaHoag)
{
    nn::vi::Initialize();

    nn::vi::Display* pDisplay;
    NNT_ASSERT_RESULT_SUCCESS(nn::vi::OpenDisplay(&pDisplay, "Internal"));

    NNT_EXPECT_RESULT_FAILURE(nn::vi::ResultNotSupported, nn::vi::SetDisplayUnderscan(pDisplay, 0));

    nn::vi::CloseDisplay(pDisplay);
}

TEST_F(InternalDisplay, GetDisplayUnderscan_IcosaHoag)
{
    nn::vi::Initialize();

    nn::vi::Display* pDisplay;
    NNT_ASSERT_RESULT_SUCCESS(nn::vi::OpenDisplay(&pDisplay, "Internal"));

    int underscan;
    NNT_ASSERT_RESULT_SUCCESS(nn::vi::GetDisplayUnderscan(&underscan, pDisplay));
    EXPECT_EQ(0, underscan);

    nn::vi::CloseDisplay(pDisplay);
}

TEST_F(InternalDisplay, SetDisplayAlpha_IcosaHoag)
{
    nn::vi::Initialize();

    nn::vi::Display* pDisplay;
    NNT_ASSERT_RESULT_SUCCESS(nn::vi::OpenDisplay(&pDisplay, "Internal"));

    NNT_EXPECT_RESULT_SUCCESS(nn::vi::SetDisplayAlpha(pDisplay, 0.f));
    NNT_EXPECT_RESULT_SUCCESS(nn::vi::SetDisplayAlpha(pDisplay, 1.f));

    nn::vi::CloseDisplay(pDisplay);
}

TEST_F(InternalDisplay, SetDisplayAlpha_InvalidRange_IcosaHoag)
{
    nn::vi::Initialize();

    nn::vi::Display* pDisplay;
    NNT_ASSERT_RESULT_SUCCESS(nn::vi::OpenDisplay(&pDisplay, "Internal"));

    NNT_EXPECT_RESULT_FAILURE(nn::vi::ResultInvalidRange, nn::vi::SetDisplayAlpha(pDisplay, -.1f));
    NNT_EXPECT_RESULT_FAILURE(nn::vi::ResultInvalidRange, nn::vi::SetDisplayAlpha(pDisplay, 1.1f));

    nn::vi::CloseDisplay(pDisplay);
}

TEST_F(InternalDisplay, SetDisplayPowerState_IcosaHoag)
{
    nn::vi::Initialize();

    nn::vi::Display* pDisplay;
    NNT_ASSERT_RESULT_SUCCESS(nn::vi::OpenDisplay(&pDisplay, "Internal"));

    NNT_EXPECT_RESULT_SUCCESS(nn::vi::SetDisplayPowerState(pDisplay, nn::vi::PowerState_Off));
    NNT_EXPECT_RESULT_SUCCESS(nn::vi::SetDisplayPowerState(pDisplay, nn::vi::PowerState_On));

    nn::vi::CloseDisplay(pDisplay);
}

TEST_F(InternalDisplay, GetDisplayHotplugEvent_IcosaHoag)
{
    nn::vi::Initialize();

    nn::vi::Display* pDisplay;
    NNT_ASSERT_RESULT_SUCCESS(nn::vi::OpenDisplay(&pDisplay, "Internal"));

    nn::os::SystemEventType hotplugEvent;
    NNT_EXPECT_RESULT_FAILURE(nn::vi::ResultNotSupported, nn::vi::GetDisplayHotplugEvent(&hotplugEvent, pDisplay));

    nn::vi::CloseDisplay(pDisplay);
}

TEST_F(InternalDisplay, SetDisplayLayerStack_IcosaHoag)
{
    nn::vi::Initialize();

    nn::vi::Display* pDisplay;
    NNT_ASSERT_RESULT_SUCCESS(nn::vi::OpenDisplay(&pDisplay, "Internal"));

    NNT_EXPECT_RESULT_SUCCESS(nn::vi::SetDisplayLayerStack(pDisplay, nn::vi::LayerStack_Lcd));
    NNT_EXPECT_RESULT_SUCCESS(nn::vi::SetDisplayLayerStack(pDisplay, nn::vi::LayerStack_Default));

    nn::vi::CloseDisplay(pDisplay);
}

TEST_F(InternalDisplay, SetDisplayCmuLuma_IcosaHoag)
{
    nn::vi::Initialize();

    nn::vi::Display* pExternal;
    NNT_ASSERT_RESULT_SUCCESS(nn::vi::OpenDisplay(&pExternal, "External"));
    nn::vi::Display* pInternal;
    NNT_ASSERT_RESULT_SUCCESS(nn::vi::OpenDisplay(&pInternal, "Internal"));

    float externalLuma;
    NNT_ASSERT_RESULT_SUCCESS(nn::vi::GetDisplayCmuLuma(&externalLuma, pExternal));

    float values[] = { -1.f, 1.f, 0.f };

    for( float value : values )
    {
        NNT_ASSERT_RESULT_SUCCESS(nn::vi::SetDisplayCmuLuma(pInternal, value));
        float luma;
        NNT_ASSERT_RESULT_SUCCESS(nn::vi::GetDisplayCmuLuma(&luma, pInternal));
        EXPECT_EQ(value, luma);

        // make sure only one pipeline is affected
        NNT_ASSERT_RESULT_SUCCESS(nn::vi::GetDisplayCmuLuma(&luma, pExternal));
        EXPECT_EQ(externalLuma, luma);
    }

    nn::vi::CloseDisplay(pExternal);
    nn::vi::CloseDisplay(pInternal);
}

TEST_F(InternalDisplay, SetDisplayCmuLuma_InvalidRange_IcosaHoag)
{
    nn::vi::Initialize();

    nn::vi::Display* pDisplay;
    NNT_ASSERT_RESULT_SUCCESS(nn::vi::OpenDisplay(&pDisplay, "Internal"));

    NNT_EXPECT_RESULT_FAILURE(nn::vi::ResultInvalidRange, nn::vi::SetDisplayCmuLuma(pDisplay, -1.1f));
    NNT_EXPECT_RESULT_FAILURE(nn::vi::ResultInvalidRange, nn::vi::SetDisplayCmuLuma(pDisplay, 1.1f));

    nn::vi::CloseDisplay(pDisplay);
}

TEST_F(InternalDisplay, GetDisplayVsyncEvent_IcosaHoag)
{
    nn::vi::Initialize();

    nn::vi::Display* pDisplay;
    NNT_ASSERT_RESULT_SUCCESS(nn::vi::OpenDefaultDisplay(&pDisplay));

    nn::os::SystemEventType vsync;
    NNT_EXPECT_RESULT_SUCCESS(nn::vi::GetDisplayVsyncEvent(&vsync, pDisplay));
    // best effort to see if event is initialized
    EXPECT_NE(nn::os::SystemEventType::State_NotInitialized, vsync._state);

    EXPECT_TRUE(nn::os::TimedWaitSystemEvent(&vsync, nn::TimeSpan::FromSeconds(1)));

    nn::os::DestroySystemEvent(&vsync);
    nn::vi::CloseDisplay(pDisplay);
}

TEST_F(InternalDisplay, GetDisplayLogicalResolution_IcosaHoag)
{
    nn::vi::Initialize();

    nn::vi::Display* pDisplay;
    NNT_ASSERT_RESULT_SUCCESS(nn::vi::OpenDisplay(&pDisplay, "Internal"));

    int width;
    int height;
    NNT_ASSERT_RESULT_SUCCESS(nn::vi::GetDisplayLogicalResolution(&width, &height, pDisplay));

    EXPECT_EQ(1920, width);
    EXPECT_EQ(1080, height);

    nn::vi::CloseDisplay(pDisplay);
}

TEST_F(InternalDisplay, SetRgbRange_IcosaHoag)
{
    nn::vi::Initialize();

    nn::vi::Display* pDisplay;
    NNT_ASSERT_RESULT_SUCCESS(nn::vi::OpenDisplay(&pDisplay, "Internal"));

    NNT_ASSERT_RESULT_SUCCESS(nn::vi::SetDisplayRgbRange(pDisplay, nn::vi::RgbRange_Auto));
    nn::vi::RgbRange actualRange;
    NNT_ASSERT_RESULT_SUCCESS(nn::vi::GetDisplayRgbRange(&actualRange, pDisplay));
    EXPECT_EQ(nn::vi::RgbRange_Full, actualRange);

    NNT_ASSERT_RESULT_SUCCESS(nn::vi::SetDisplayRgbRange(pDisplay, nn::vi::RgbRange_Full));
    NNT_ASSERT_RESULT_SUCCESS(nn::vi::GetDisplayRgbRange(&actualRange, pDisplay));
    EXPECT_EQ(nn::vi::RgbRange_Full, actualRange);

    NNT_EXPECT_RESULT_FAILURE(nn::vi::ResultNotSupported, nn::vi::SetDisplayRgbRange(pDisplay, nn::vi::RgbRange_Limited));

    nn::vi::CloseDisplay(pDisplay);
}

TEST_F(InternalDisplay, GetDisplayModeChangedEvent_Any)
{
    nn::vi::Initialize();

    nn::vi::Display* pDisplay;
    NNT_ASSERT_RESULT_SUCCESS(nn::vi::OpenDisplay(&pDisplay, "Internal"));

    nn::os::SystemEvent event;
    NNT_ASSERT_RESULT_FAILURE(nn::vi::ResultNotSupported, nn::vi::GetDisplayModeChangedEvent(event.GetBase(), pDisplay));

    nn::vi::CloseDisplay(pDisplay);
}
