<#
    Copyright (C)Nintendo All rights reserved.

    These coded instructions, statements, and computer programs contain proprietary
    information of Nintendo and/or its licensed developers and are protected by
    national and international copyright laws. They may not be disclosed to third
    parties or copied or duplicated in any form, in whole or in part, without the
    prior written consent of Nintendo.

    The content herein is highly confidential and should be handled accordingly.
#>

<#
    .SYNOPSIS
        Common functionality for CEC tests

    .DESCRIPTION
        Provides common functionality for CEC display tests
#>

# TV-specific parameters are in a separate file to make them easier to find
. "${scriptDirectoryPath}\TvParameters.ps1"

function Initialize-Consts
{
    $script:scriptRootPath      = "$env:NINTENDO_SDK_ROOT"
    $script:toolsDirectoryPath  = "${scriptRootPath}\Tools\CommandLineTools"

    if ($scriptRootPath -eq "")
    {
        Write-Error "Error: NINTENDO_SDK_ROOT not set"
        Write-Error "Aborting"
        exit 1
    }

    if ($ArtifactPath -eq "")
    {
        $script:ArtifactPath = ${scriptDirectoryPath}
    }

    $script:CheckToneExe               = "${scriptRootPath}\Tests\Outputs\Win32-v140\Tests\CheckTone\Develop\CheckTone.exe"
    $script:UUExe                      = "${scriptRootPath}\Externals\Binaries\DigitalLoggers\UU.exe"
    $script:ControlTargetExe           = "${toolsDirectoryPath}\ControlTarget.exe"
    $script:ControlTargetPrivateExe    = "${toolsDirectoryPath}\ControlTargetPrivate.exe"
    $script:RunOnTargetExe             = "${toolsDirectoryPath}\RunOnTarget.exe"

    $script:InitLog                    = "${ArtifactPath}\${TestAddress}_init.log"
    $script:CleanUpLog                 = "${ArtifactPath}\${TestAddress}_cleanup.log"

    $script:maxRetryNumber             = 100
    
    $script:PowerUsername              = "admin"
    $script:PowerPassword              = "displaylab"
}

function Initialize([Boolean]$initRemoteControl)
{
    Initialize-Consts
    
    $script:InitRemoteControl = $initRemoteControl

    Write-Host "Iterations: ${Iterations}"
    Write-Host "Artifact Path: ${ArtifactPath}"
    Write-Host "Test Address: ${TestAddress}"
    Write-Host "Test Tone: ${TestTone}"
    if ($InitRemoteControl)
    {
        Write-Host "Remote Control Address: ${RemoteControlAddress}"
        Write-Host "Remote Control Tone: ${RemoteControlTone}"
    }
    Write-Host "Server Address: ${ServerAddress}"
    Write-Host "Power Off TV: ${PowerOffTV}"
    
    # NOTE: Since hard resetting the devices may increase the likelihood of a connection issue,
    # calling it is done as little as possible

    Remove-Item $InitLog -ErrorAction SilentlyContinue

###############################################################################################
    Write-Host "Initiating Serial Logging..."
    $ScriptBlock =
    {
        param($artifactPath, $targetAddress)
        
        $Socket = New-Object System.Net.Sockets.TcpClient($targetAddress, "10023")
        $Stream = $Socket.GetStream()
        $Reader = New-Object System.IO.StreamReader($Stream)
        
        $currentTime = Get-Date -Format "yyyyMMddThhmmss"
        Remove-Item "${artifactPath}\${currentTime}_${targetAddress}_serial.log" -ErrorAction SilentlyContinue
        
        while (1) 
        {
            $line = $Reader.ReadLine()
            
            # Remove the color information
            $line = $line -replace '\e\[\d\d?m', ''
            
            $line >> "${artifactPath}\${currentTime}_${targetAddress}_serial.log"
        }
    }

    $SerialJob1 = Start-Job $ScriptBlock -ArgumentList $ArtifactPath, $TestAddress
    if ($InitRemoteControl)
    {
        $SerialJob2 = Start-Job $ScriptBlock -ArgumentList $ArtifactPath, $RemoteControlAddress
    }
    
###############################################################################################
    Write-Host "Making sure splitter power is on"
    & $UUExe "${HDMIPowerIP}:80" "${PowerUsername}:${PowerPassword}" "${HDMIPowerNum}on" status

###############################################################################################
    Write-Host "Formatting SD cards..."

    & $RunOnTargetExe "${scriptRootPath}/Programs/Eris/Outputs/NX-NXFP2-a64/TargetTools/DevMenuCommandSystem/Develop/DevMenuCommandSystem.nsp" -t $TestAddress --failure-timeout=10 -- sdcard format >> $InitLog

    if ($InitRemoteControl)
    {
        & $RunOnTargetExe "${scriptRootPath}/Programs/Eris/Outputs/NX-NXFP2-a64/TargetTools/DevMenuCommandSystem/Develop/DevMenuCommandSystem.nsp" -t $RemoteControlAddress --failure-timeout=10 -- sdcard format >> $InitLog
    }

    Write-Host "Initiating SD card logging..."

    & $RunOnTargetExe "${scriptRootPath}/Programs/Eris/Outputs/NX-NXFP2-a64/TargetTools/DevMenuCommandSystem/Develop/DevMenuCommandSystem.nsp" -t $TestAddress --failure-timeout=10 -- debug enable-sd-card-logging >> $InitLog

    if ($InitRemoteControl)
    {
        & $RunOnTargetExe "${scriptRootPath}/Programs/Eris/Outputs/NX-NXFP2-a64/TargetTools/DevMenuCommandSystem/Develop/DevMenuCommandSystem.nsp" -t $RemoteControlAddress --failure-timeout=10 -- debug enable-sd-card-logging >> $InitLog
    }

###############################################################################################
    Write-Host "Resetting SDEV..."
    & $ControlTargetExe reset -t $TestAddress >> $InitLog
    Write-Host "Connecting to SDEV..."
    & $ControlTargetExe connect -t $TestAddress >> $InitLog

###############################################################################################
    if ($InitRemoteControl)
    {
        Write-Host "Resetting remote control..."
        & $ControlTargetExe reset -t $RemoteControlAddress >> $InitLog
        Write-Host "Connecting to remote control..."
        & $ControlTargetExe connect -t $RemoteControlAddress >> $InitLog
    }
    
###############################################################################################
    Write-Host "Docking cradles..."
    
    & $ControlTargetExe enable-cradle -t $TestAddress >> $InitLog
    
    if ($InitRemoteControl)
    {
        & $ControlTargetExe enable-cradle -t $RemoteControlAddress >> $InitLog
    }
    
###############################################################################################
    # Note: This needs to be done after resetting and docking the SDEV and turning on the splitter
    $edid = Get-Tv-Edid $TestAddress
    Check-For-Compatibility $edid $PowerOffTV
    Update-Tv-Parameters $edid
    
    $script:TestAmplitude = Apply-Amplitude-Multiplier $script:TestTone $script:Amplitude
    Write-Host "Test Amplitude: ${TestAmplitude}"
    if ($InitRemoteControl)
    {
        $script:RemoteControlAmplitude = Apply-Amplitude-Multiplier $script:RemoteControlTone $script:Amplitude
        Write-Host "Remote Control Amplitude: ${RemoteControlAmplitude}"
    }
}

function Cleanup
{
    Write-Host "Cleaning up..."
    Remove-Item $CleanUpLog -ErrorAction SilentlyContinue

###############################################################################################
    Write-Host "Restarting test SDEV..."

    & $ControlTargetExe reset -t $TestAddress --hard >> $CleanUpLog

    & $ControlTargetExe connect -t $TestAddress >> $CleanUpLog

###############################################################################################
    Write-Host "Disabling SD card logging..."

    & $RunOnTargetExe "${scriptRootPath}/Programs/Eris/Outputs/NX-NXFP2-a64/TargetTools/DevMenuCommandSystem/Develop/DevMenuCommandSystem.nsp" -t $TestAddress --failure-timeout=10 -- debug disable-sd-card-logging > $null

    if ($InitRemoteControl)
    {
        & $RunOnTargetExe "${scriptRootPath}/Programs/Eris/Outputs/NX-NXFP2-a64/TargetTools/DevMenuCommandSystem/Develop/DevMenuCommandSystem.nsp" -t $RemoteControlAddress --failure-timeout=10 -- debug disable-sd-card-logging > $null
    }

###############################################################################################
    Write-Host "Restarting SDEVs..."

    & $ControlTargetExe reset -t $TestAddress --hard >> $CleanUpLog

    & $ControlTargetExe connect -t $TestAddress >> $CleanUpLog

    if ($InitRemoteControl)
    {
        & $ControlTargetExe reset -t $RemoteControlAddress --hard >> $CleanUpLog

        & $ControlTargetExe connect -t $RemoteControlAddress >> $CleanUpLog
    }

###############################################################################################
    Write-Host "Collecting logs..."

    & $RunOnTargetExe "${scriptRootPath}\Tests\Outputs\NX-NXFP2-a64\Tests\GetSdCardFiles\Develop\GetSdCardFiles.nsp" -t $TestAddress --failure-timeout=30 -- --read_dir=sd:/NxBinLogs --write_dir=${ArtifactPath} >> $CleanUpLog

    if ($InitRemoteControl)
    {
        & $RunOnTargetExe "${scriptRootPath}\Tests\Outputs\NX-NXFP2-a64\Tests\GetSdCardFiles\Develop\GetSdCardFiles.nsp" -t $RemoteControlAddress --failure-timeout=30 -- --read_dir=sd:/NxBinLogs --write_dir=${ArtifactPath} >> $CleanUpLog
    }
    
    & "${scriptRootPath}\Tests\Outputs\AnyCPU\Tools\ParseNxBinLogs\Release\ParseNxBinLogs.exe" "${ArtifactPath}" >> $CleanUpLog
}

function Get-TV-Focus-With-Remote
{
    Write-Host "Performing OneTouchPlay with remote control..."
    $attemptCounter = 1
    do
    {
        if ($attemptCounter -ge $script:maxRetryNumber)
        {
            Cleanup
            Write-Error "*** UNABLE TO POWER ON TV WITH REMOTE CONTROL ***"
            Write-Error "Aborting"
            exit 1
        }

        Write-Host "- Attempt #${attemptCounter}..."
        if ($attemptCounter % 20 -eq 1)
        {
            Write-Host "(attempting OneTouchPlay)"
            $ret = & $RunOnTargetExe "${scriptDirectoryPath}\RemoteControl.nsp" -t $RemoteControlAddress -- -OneTouchPlay
            Write-Host $ret
            if ($ret -contains "Operation returned 4")
            {
                Cleanup
                Write-Error "*** HIT LOGICAL ADDRESS ZERO BUG ***"
                Write-Error "Aborting"
                exit 1
            }
            & $RunOnTargetExe "${scriptDirectoryPath}\PlayTone.nsp" --no-wait -t $RemoteControlAddress -- -frequency $RemoteControlTone -amplitude $RemoteControlAmplitude > $null
        }
        
        Start-Sleep -s 1

        & $CheckToneExe -activeTone $RemoteControlTone -silentTone $TestTone -ipAddr $ServerAddress > $null
        
        $attemptCounter = $attemptCounter + 1
    }
    while ($lastexitcode -ne 0)
}

function Power-Off-TV-With-Remote([Boolean]$standbyAll)
{
    if ($standbyAll)
    {
        Write-Host "Turning off all with remote control..."
        $command = "AllGoStandby"
    }
    else
    {
        Write-Host "Turning off TV with remote control..."
        $command = "GoStandby"
    }
    $attemptCounter = 1
    do
    {
        if ($attemptCounter -ge $script:maxRetryNumber)
        {
            Cleanup
            Write-Error "*** UNABLE TO POWER OFF TV WITH REMOTE CONTROL ***"
            Write-Error "Aborting"
            exit 1
        }

        Write-Host "- Attempt #${attemptCounter} -"
        if ($attemptCounter % 20 -eq 1)
        {
            Write-Host "(attempting ${command})"
            $ret = & $RunOnTargetExe "${scriptDirectoryPath}\RemoteControl.nsp" -t $RemoteControlAddress -- "-${command}"
            Write-Host $ret
            if ($ret -contains "Operation returned 4")
            {
                Cleanup
                Write-Error "*** HIT LOGICAL ADDRESS ZERO BUG (see SIGLO-65916) ***"
                Write-Error "Aborting"
                exit 1
            }
            & $RunOnTargetExe "${scriptDirectoryPath}\PlayTone.nsp" --no-wait -t $RemoteControlAddress -- -frequency $RemoteControlTone -amplitude $RemoteControlAmplitude > $null
        }
        
        Start-Sleep -s 1

        & $CheckToneExe -silentTone $RemoteControlTone -ipAddr $ServerAddress > $null
        
        $attemptCounter = $attemptCounter + 1
    }
    while ($lastexitcode -ne 0)
    
    # Make sure the device isn't put to sleep multiple times
    Start-Sleep -s $SleepDelay
}

function Get-Tv-Edid([String]$ipAddress)
{
    $out = & $RunOnTargetExe "${scriptRootPath}\Tests\Outputs\NX-NXFP2-a64\Tests\PrintTVIdentifier\Develop\PrintTVIdentifier.nsp" -t $ipAddress | Out-String
    
    $index = $out.IndexOf('<edid>')
    $index2 = $out.IndexOf('</edid>')
    
    if ($index -eq -1 -or $index2 -eq -1)
    {
        Write-Error "Unable to get EDID information"
        Write-Error "Aborting"
        exit 1
    }
    
    $index = $index + 6
    $edid = $out.SubString($index, $index2 - $index)
    
    Write-Host "Note: TV is $edid"
    return $edid
}

function Apply-Amplitude-Multiplier([Int]$frequency, [Int]$amplitude)
{
    # 450 is used as the base here since the tones are 400~1400
    # This transformation makes sure the detected tone is at the same level regardless of frequency
    return $amplitude * 450 / $frequency
}
