﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstdlib>

#include <nn/nn_Abort.h>
#include <nn/nn_Assert.h>
#include <nn/audio.h>
#include <nn/os.h>
#include <nn/mem.h>
#include <nn/nn_TimeSpan.h>
#include <nn/nn_Log.h>

#include <nn/cec/cec_Api.h>

nn::Result CommandOneTouchPlay()
{
    NN_LOG("Performing One Touch Play...\n");

    return nn::cec::PerformOneTouchPlay();
}

nn::Result CommandAllGoStandby()
{
    NN_LOG("Performing All Go Standby...\n");

    return nn::cec::PerformGoStandby(false);
}

nn::Result CommandGoStandby()
{
    NN_LOG("Performing Go Standby...\n");

    return nn::cec::PerformGoStandby(true);
}

extern "C" int nnMain()
{
    int retryCounter = 0;
    int ret = -1;
    nn::Result result;

    nn::os::SystemEventType systemEvent;
    nn::cec::Initialize(&systemEvent);

    nn::cec::SuspendManager();

    do
    {
        // Retry up to 10 times
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(500));
        ++retryCounter;

        result = nn::cec::RestartManager();

        ret = result.GetDescription();
    }
    while (ret == 4 && retryCounter < 10);

    if (ret)
    {
        NN_LOG("Failed to start CEC manager\n", ret);
        NN_LOG("Operation returned %d\n", ret);

        return ret;
    }

    retryCounter = 0;

    int argc = nn::os::GetHostArgc();
    char** argv = nn::os::GetHostArgv();
    for(int i = 1; i < argc; ++i)
    {
        if(strcmp("-OneTouchPlay", argv[i]) == 0)
        {
            result = CommandOneTouchPlay();
        }
        else if(strcmp("-AllGoStandby", argv[i]) == 0)
        {
            result = CommandAllGoStandby();
        }
        else if(strcmp("-GoStandby", argv[i]) == 0)
        {
            result = CommandGoStandby();
        }
        else
        {
            continue;
        }

        ret = result.GetDescription();

        // Retry up to 5 times
        if (ret == 4 && retryCounter < 5)
        {
            nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(500));
            ++retryCounter;
            --i;
            continue;
        }

        NN_LOG("Operation returned %d\n", ret);

        retryCounter = 0;
    }

    nn::cec::Finalize(&systemEvent);

    return ret;
}
