﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstdio>
#include <cstring>
#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/nn_Log.h>
#include <nn/settings/system/settings_Tv.h>
#include <nn/result/result_HandlingUtility.h>
#include <nnt/result/testResult_Assert.h>
#include <nnt/nntest.h>

#include "Helper.h"

nn::os::SystemEventType s_CecSystemEvent;

#define CECCaseToReturnString(name) \
case nn::cec::name:                 \
    return #name;

const char* BusEventToString(nn::cec::BusEventType busEvent) NN_NOEXCEPT
{
    switch(busEvent)
    {
        CECCaseToReturnString(BusEventType_Ignore)
        CECCaseToReturnString(BusEventType_ActiveSourceChangedToActive)
        CECCaseToReturnString(BusEventType_ActiveSourceChangedToInactive)
        CECCaseToReturnString(BusEventType_GoStandby)
        CECCaseToReturnString(BusEventType_Suspending)
        CECCaseToReturnString(BusEventType_ConnectionChange)
        CECCaseToReturnString(BusEventType_FeatureAbortOneTouchPlay)
        CECCaseToReturnString(BusEventType_FeatureAbortStandby)
        CECCaseToReturnString(BusEventType_FeatureAbortSetOnScreenString)
        CECCaseToReturnString(BusEventType_Started)
        default:
            return "<unknown>";
    }
    return "<unknown>";
}

const char* PowerStateToString(nn::cec::PowerState powerState) NN_NOEXCEPT
{
    switch(powerState)
    {
        CECCaseToReturnString(PowerState_On)
        CECCaseToReturnString(PowerState_Standby)
        CECCaseToReturnString(PowerState_GoingOn)
        CECCaseToReturnString(PowerState_GoingStandby)
        default:
            return "<unknown>";
    }
    return "<unknown>";
}

const char* ConnectionStateToString(nn::cec::ConnectionState connectionState) NN_NOEXCEPT
{
    switch(connectionState)
    {
        CECCaseToReturnString(ConnectionState_NotConnected)
        CECCaseToReturnString(ConnectionState_OnlyCradleConnected)
        CECCaseToReturnString(ConnectionState_CradleAndTvConnected)
        default:
            return "<unknown>";
    }
    return "<unknown>";
}

bool WaitSystemEventWithTimeout(int timeoutMs, nn::cec::BusEventType busEvent, bool fail)
{
    NN_LOG("Waiting %i ms for bus event %s...\n", timeoutMs, BusEventToString(busEvent));
    nn::TimeSpan start = nn::os::GetSystemTick().ToTimeSpan();

    do
    {
        bool result = nn::os::TryWaitSystemEvent(&s_CecSystemEvent);

        while (!result)
        {
            if ((nn::os::GetSystemTick().ToTimeSpan() - start).GetMilliSeconds() >= timeoutMs)
            {
                if (fail)
                {
                    NN_LOG("Timed out after %i ms waiting for system event\n", timeoutMs);
                    ADD_FAILURE();
                    // TODO: Remove this once manual debugging is finished
                    while (1)
                    {
                        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(1000));
                    }
                }
                return false;
            }

            nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(10));
            result = nn::os::TryWaitSystemEvent(&s_CecSystemEvent);
        }

        NN_LOG("Got event %i\n", GetCECBusType());
    } while (busEvent != GetCECBusType());

    return true;
}

void CheckCECBusType(nn::cec::BusEventType expectedEvent)
{
    nn::cec::BusEventType   busEvent;
    NNT_EXPECT_RESULT_SUCCESS(nn::cec::GetBusEventType(&busEvent));
    EXPECT_EQ(expectedEvent, busEvent);
}

nn::cec::BusEventType GetCECBusType()
{
    nn::cec::BusEventType   busEvent;
    NNT_EXPECT_RESULT_SUCCESS(nn::cec::GetBusEventType(&busEvent));
    return busEvent;
}

void SuspendCECManager(bool isStarted)
{
    NN_LOG("Suspending CEC Manager (currently %s)...\n", (isStarted ? "running" : "suspended"));
    nn::cec::SuspendManager();

    // As soon as SuspendManager returns, there should be a system event if there's going to be one
    ASSERT_EQ(isStarted, nn::os::TryWaitSystemEvent(&s_CecSystemEvent));

    if (isStarted)
        CheckCECBusType(nn::cec::BusEventType_Suspending);
}

void ResumeCECManager(bool isStarted)
{
    NN_LOG("Resuming CEC Manager (currently %s)...\n", (isStarted ? "running" : "suspended"));
    nn::cec::RestartManager();

    // As soon as RestartManager returns, there should be a system event
    ASSERT_NE(isStarted, nn::os::TryWaitSystemEvent(&s_CecSystemEvent));

    if (!isStarted)
        CheckCECBusType(nn::cec::BusEventType_Started);
}

bool WaitForPowerState(nn::cec::PowerState targetState, int timeoutMs, bool fail)
{
    NN_LOG("Waiting %i ms for %s...\n", timeoutMs, PowerStateToString(targetState));
    nn::TimeSpan start = nn::os::GetSystemTick().ToTimeSpan();

    nn::cec::PowerState powerState = nn::cec::PowerState_GoingOn;
    nn::Result result = nn::cec::GetTvPowerState(&powerState);
    if (result.IsFailure())
    {
        // A timeout is acceptable, but only for so long
        NNT_EXPECT_RESULT_FAILURE(nn::cec::ResultTimeout, result);
    }

    while (powerState != targetState)
    {
        if ((nn::os::GetSystemTick().ToTimeSpan() - start).GetMilliSeconds() >= timeoutMs)
        {
            if (fail)
            {
                NN_LOG("Timed out after %i ms waiting for %s\n", timeoutMs, PowerStateToString(targetState));
                ADD_FAILURE();
            }
            return false;
        }

        // Note: As the CEC only allows 40 bytes per second, we shouldn't overload it
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(500));
        nn::Result result = nn::cec::GetTvPowerState(&powerState);

        if (result.IsFailure())
        {
            // A timeout is acceptable, but only for so long
            NNT_EXPECT_RESULT_FAILURE(nn::cec::ResultTimeout, result);
        }
    }

    return true;
}

nn::cec::PowerState GetPowerState(int timeoutMs)
{
    NN_LOG("Getting PowerState (timeout %i ms)...\n", timeoutMs);
    nn::TimeSpan start = nn::os::GetSystemTick().ToTimeSpan();

    nn::cec::PowerState powerState = nn::cec::PowerState_Standby;
    nn::Result result = nn::cec::GetTvPowerState(&powerState);
    while (result.IsFailure())
    {
        // A timeout is acceptable, but only for so long
        NNT_EXPECT_RESULT_FAILURE(nn::cec::ResultTimeout, result);

        if ((nn::os::GetSystemTick().ToTimeSpan() - start).GetMilliSeconds() >= timeoutMs)
        {
            NN_LOG("Timed out after %i ms trying to get PowerState\n", timeoutMs);
            ADD_FAILURE();
            break;
        }

        // Note: As the CEC only allows 40 bytes per second, we shouldn't overload it
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(500));
        result = nn::cec::GetTvPowerState(&powerState);
    }

    return powerState;
}

void TurnOffTV(int timeout)
{
    NN_LOG("Turning off TV...\n");

    // Note:  Unlike turning on a TV, turning it off should never be ignored
    nn::cec::PerformGoStandby(true);

    // We should get a system event fairly quickly to go into standby (from a TV like Samsung)
    WaitSystemEventWithTimeout(timeout, nn::cec::BusEventType_GoStandby);

    EXPECT_EQ(nn::cec::PowerState_Standby, GetPowerState());

    NN_LOG("Turned off TV\n");
}

void TurnOnTV(int maxRetries, int retryDelay, int timeout)
{
    NN_LOG("Turning on TV...\n");
    int retries = 0;

    nn::cec::PerformOneTouchPlay();
    bool result = WaitSystemEventWithTimeout(retryDelay, nn::cec::BusEventType_ConnectionChange, false);

    // This may get ignored at first, so we must be ready to retry
    while (!result)
    {
        ++retries;
        if (retries >= maxRetries)
        {
            FAIL();
            return;
        }

        nn::cec::PerformOneTouchPlay();
        result = WaitSystemEventWithTimeout(retryDelay, nn::cec::BusEventType_ConnectionChange, false);
    }

    WaitForPowerState(nn::cec::PowerState_On, timeout, true);

    NN_LOG("Turned on TV after %i retry(s)\n", retries);
}

void CallCancelledRestartManager(void* data) NN_NOEXCEPT
{
    ThreadData* threadData = reinterpret_cast<ThreadData*>(data);

    nn::os::SignalEvent(&threadData->syncEvent);
    threadData->result = nn::cec::RestartManager();
    nn::os::SignalEvent(&threadData->syncEvent);
}

void CallCancelledIsTvResponsive(void* data) NN_NOEXCEPT
{
    ThreadData* threadData = reinterpret_cast<ThreadData*>(data);

    bool isSuccessfullyPinged;
    nn::os::SignalEvent(&threadData->syncEvent);
    threadData->result = nn::cec::IsTvResponsive(&isSuccessfullyPinged);
    nn::os::SignalEvent(&threadData->syncEvent);
}

void CallCancelledPerformOneTouchPlay(void* data) NN_NOEXCEPT
{
    ThreadData* threadData = reinterpret_cast<ThreadData*>(data);

    nn::os::SignalEvent(&threadData->syncEvent);
    threadData->result = nn::cec::PerformOneTouchPlay();
    nn::os::SignalEvent(&threadData->syncEvent);
}

void CallCancelledGetTvPowerState(void* data) NN_NOEXCEPT
{
    ThreadData* threadData = reinterpret_cast<ThreadData*>(data);

    nn::cec::PowerState powerState;
    nn::os::SignalEvent(&threadData->syncEvent);
    threadData->result = nn::cec::GetTvPowerState(&powerState);
    nn::os::SignalEvent(&threadData->syncEvent);
}

void CallAndCancelFunctionFunc(nn::os::ThreadFunction func, int delayUs, int timeoutMs)
{
    static const size_t                             s_CancelTestStackSize = 8 * 1024;
    static NN_ALIGNAS(nn::os::MemoryPageSize) char  s_CancelTestStack[s_CancelTestStackSize];
    nn::os::ThreadType                              thread;
    bool                                            isCancelling;
    ThreadData                                      threadData;

    nn::os::InitializeEvent(&threadData.syncEvent, false, nn::os::EventClearMode_AutoClear);

    nn::os::CreateThread(&thread, func, &threadData,
                         s_CancelTestStack, s_CancelTestStackSize,
                         nn::os::GetThreadPriority(nn::os::GetCurrentThread()) + 1);
    nn::os::StartThread(&thread);

    nn::os::WaitEvent(&threadData.syncEvent);
    nn::os::SleepThread(nn::TimeSpan::FromMicroSeconds(delayUs));
    nn::cec::CancelCurrentApiCall(&isCancelling);

    // Wait for the thread to actually cancel
    nn::TimeSpan start = nn::os::GetSystemTick().ToTimeSpan();

    while (!nn::os::TryWaitEvent(&threadData.syncEvent))
    {
        if ((nn::os::GetSystemTick().ToTimeSpan() - start).GetMilliSeconds() >= timeoutMs)
        {
            NN_LOG("Timed out after %i ms trying to cancel function (may hang)\n", timeoutMs);
            ADD_FAILURE();
            break;
        }

        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(1));
    }

    nn::os::WaitThread(&thread);
    nn::os::DestroyThread(&thread);
    nn::os::FinalizeEvent(&threadData.syncEvent);

    // Make sure it either was successful or cancelled
    if (isCancelling)
    {
        NNT_EXPECT_RESULT_FAILURE(nn::cec::ResultCanceled, threadData.result);
    }
    else
    {
        NNT_ASSERT_RESULT_SUCCESS(threadData.result);
    }
}
