﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <windows.h>
#include <cstdio>

#define clamp(n) (unsigned char)((n) >= 255 ? 255 : ((n) < 0 ? 0 : (n)))

namespace BitmapHelper
{

/// Source from https://msdn.microsoft.com/en-us/library/windows/desktop/dd407288(v=vs.85).aspx

// Writes a bitmap file
//  pszFileName:  Output file name.
//  pBMI:         Bitmap format information (including palette).
//  cbBMI:        Size of the BITMAPINFOHEADER, including palette, if present.
//  pData:        Pointer to the bitmap bits.
//  cbData        Size of the bitmap, in bytes.

HRESULT Write(const char* pszFileName, BITMAPINFOHEADER *pBMI, unsigned long cbBMI, unsigned char *pData, unsigned long cbData)
{
    HRESULT hr;
    HANDLE hFile = CreateFileA(pszFileName, GENERIC_WRITE, 0, nullptr, CREATE_ALWAYS, 0, nullptr);

    if (hFile)
    {
        BITMAPFILEHEADER bmf = {};

        bmf.bfType = 'MB';
        bmf.bfSize = cbBMI + cbData + sizeof(bmf);
        bmf.bfOffBits = sizeof(bmf) + cbBMI;

        unsigned long cbWritten = 0;
        BOOL result = WriteFile(hFile, &bmf, sizeof(bmf), &cbWritten, nullptr);
        if (result)
        {
            result = WriteFile(hFile, pBMI, cbBMI, &cbWritten, nullptr);
        }
        if (result)
        {
            result = WriteFile(hFile, pData, cbData, &cbWritten, nullptr);
        }

        hr = result ? S_OK : HRESULT_FROM_WIN32(GetLastError());

        CloseHandle(hFile);
    }
    else
    {
        hr = HRESULT_FROM_WIN32(GetLastError());
    }

    return hr;
}

// Wrapper for WriteBitmap (24-bit)
HRESULT Write24(const char* pFile, unsigned long width, unsigned long height, unsigned long size, unsigned char* pData)
{
    BITMAPINFOHEADER header;

    ZeroMemory(&header, sizeof(BITMAPINFOHEADER));
    header.biSize = sizeof(BITMAPINFOHEADER);
    header.biWidth = width;
    header.biHeight = height;
    header.biPlanes = 1;
    header.biBitCount = 24;
    header.biCompression = 0;
    header.biSizeImage = header.biWidth * header.biHeight * 3;

    return Write(pFile, &header, header.biSize, pData, size);
}

// Converts an 8 bit YUV bitmap into a 24-bit BGR bitmap
void Convert8BitYUVToBGR(unsigned char* bytes, int width, int height, unsigned char** out, int* outSize)
{
    // 24 bit bitmap
    // +1 for 32 bit -> 24 bit overrun
    unsigned char* pBitmap = new unsigned char[width * height * 3 + 1];
    *outSize = width * height * 3;

    // Note: The source is inverted vertically
    unsigned char* pOut = pBitmap;
    unsigned char* pIn = bytes + ((height - 1) * width) * 2;

    const int yBitmap = 0;
    const int yBytes = -4 * width;
    const int xBitmap = 6;
    const int xBytes = 4;

    for (int y = 0; y < height; ++y)
    {
        for (int x = 0; x < width; x += 2)
        {
            int Y = pIn[1] * 298;
            int U = pIn[0];
            int V = pIn[2];

            int a = 409 * V - 56992;
            int b = 36064 - 100 * U - 208 * V;
            int c = 516 * U - 70688;

            pOut[2] = clamp(((Y + a) >> 8));
            pOut[1] = clamp(((Y + b) >> 8));
            pOut[0] = clamp(((Y + c) >> 8));

            Y = pIn[3] * 298;

            pOut[5] = clamp(((Y + a) >> 8));
            pOut[4] = clamp(((Y + b) >> 8));
            pOut[3] = clamp(((Y + c) >> 8));

            pOut += xBitmap;
            pIn += xBytes;
        }

        pOut += yBitmap;
        pIn += yBytes;
    }

    *out = pBitmap;
}

// Converts an 32-bit BGRA bitmap into a 24-bit BGR bitmap
void ConvertBGRAToBGR(unsigned char* bytes, int width, int height, unsigned char** out, int* outSize)
{
    // 24 bit bitmap
    // +1 for 32 bit -> 24 bit overrun
    unsigned char* pBitmap = new unsigned char[width * height * 3 + 1];
    *outSize = width * height * 3;

    unsigned char* pOut = pBitmap;
    unsigned char* pIn = bytes + ((height - 1) * width) * 4;

    const int yBitmap = 0;
    const int yBytes = -8 * width;

    // 16x-width is common, and can be faster
    if ((width & 0xF) == 0)
    {
        const int xBitmap = 48;
        const int xBytes = 64;

        for (int y = 0; y < height; ++y)
        {
            for (int x = 0; x < width; x += 16)
            {
                *(__int32*)(pOut + 0) = *(__int32*)(pIn + 0);
                *(__int32*)(pOut + 3) = *(__int32*)(pIn + 4);
                *(__int32*)(pOut + 6) = *(__int32*)(pIn + 8);
                *(__int32*)(pOut + 9) = *(__int32*)(pIn + 12);
                *(__int32*)(pOut + 12) = *(__int32*)(pIn + 16);
                *(__int32*)(pOut + 15) = *(__int32*)(pIn + 20);
                *(__int32*)(pOut + 18) = *(__int32*)(pIn + 24);
                *(__int32*)(pOut + 21) = *(__int32*)(pIn + 28);
                *(__int32*)(pOut + 24) = *(__int32*)(pIn + 32);
                *(__int32*)(pOut + 27) = *(__int32*)(pIn + 36);
                *(__int32*)(pOut + 30) = *(__int32*)(pIn + 40);
                *(__int32*)(pOut + 33) = *(__int32*)(pIn + 44);
                *(__int32*)(pOut + 36) = *(__int32*)(pIn + 48);
                *(__int32*)(pOut + 39) = *(__int32*)(pIn + 52);
                *(__int32*)(pOut + 42) = *(__int32*)(pIn + 56);
                *(__int32*)(pOut + 45) = *(__int32*)(pIn + 60);

                pOut += xBitmap;
                pIn += xBytes;
            }

            pOut += yBitmap;
            pIn += yBytes;
        }
    }
    else
    {
        const int xBitmap = 3;
        const int xBytes = 4;

        for (int y = 0; y < height; ++y)
        {
            for (int x = 0; x < width; ++x)
            {
                *(__int32*)(pOut) = *(__int32*)(pIn);

                pOut += xBitmap;
                pIn += xBytes;
            }

            pOut += yBitmap;
            pIn += yBytes;
        }
    }

    *out = pBitmap;
}

}
