﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstdio>
#include <cstdlib>
#include <ctime>
#include <nn/nn_Log.h>
#include <nn/nn_Assert.h>
#include <nn/os.h>
#include <nnt/nntest.h>
#include <nn/util/util_FormatString.h>

namespace {

const size_t    ThreadCount = 16;
const size_t    ThreadStackSize = 8192;
NN_OS_ALIGNAS_THREAD_STACK char  g_ThreadStack[ThreadCount][ThreadStackSize];

}

void ThreadFunction(void *arg)
{
    NN_UNUSED(arg);

    auto thread = nn::os::GetCurrentThread();
    auto name = nn::os::GetThreadNamePointer(thread);
    auto priority = nn::os::GetThreadPriority(thread);

    NN_LOG("%s (priority=%d): Start\n", name, priority);

    for (auto i = 0; i < 16; ++i)
    {
        auto waitTime = nn::TimeSpan::FromMilliSeconds(((std::rand() % 8) + 1) * 10); // 10ms ~ 80ms
        NN_LOG("%s: Wait %lld msec\n", name, waitTime.GetMilliSeconds());
        nn::os::SleepThread(waitTime);
    }

    NN_LOG("%s: Finish\n", name);
}

// マルチスレッドでログ出力しても、デッドロックしないことを確かめる
TEST(LogTest, MultiThread)
{
    std::srand(static_cast<unsigned int>(std::time(0)));

    nn::os::ThreadType  thread[ThreadCount];

    // スレッドを生成する
    for(auto i = 0; i < ThreadCount; i++)
    {
        auto priority = nn::os::DefaultThreadPriority + ((std::rand() % 8) - 3); // -3 ~ +3
        auto result = nn::os::CreateThread(&thread[i], ThreadFunction, nullptr, g_ThreadStack[i], ThreadStackSize, priority);
        NN_ABORT_UNLESS_RESULT_SUCCESS(result);

        char name[nn::os::ThreadNameLengthMax];
        nn::util::SNPrintf(name, nn::os::ThreadNameLengthMax, "Thread%d", i + 1);
        nn::os::SetThreadName(&thread[i], name);
    }

    // スレッドの実行を開始する
    for(auto i = 0; i < ThreadCount; i++)
    {
        nn::os::StartThread(&thread[i]);
    }

    // スレッドが終了するのを待って破棄する
    for(auto i = 0; i < ThreadCount; i++)
    {
        nn::os::WaitThread(&thread[i]);
        nn::os::DestroyThread(&thread[i]);
    }

    // 全スレッドの終了を確認する
    NN_LOG("Ok.\n");
}
