﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstdlib>
#include <memory>
#include <nnt/nntest.h>
#include <nn/nn_Assert.h>
#include <nn/nn_SdkAssert.h>
#include <nn/diag/diag_AssertionFailureHandler.h>
#include "testDiag_AssertTestUtil.h"

// ExpectFailureAssertReferenceNotNull に渡すためのヌルポインタ。
// 最適化でヌルポインタであると判定されにくくするため、non-const にしてグローバルに置いておく。
int* g_NullPtrToInt = NULL;

#if defined(NN_SDK_BUILD_DEBUG) || defined(NN_SDK_BUILD_DEVELOP)

#define EXPECT_NAMED_ASSERTIONS_SUCCESS(name, arguments) \
    do \
    { \
        EXPECT_ASSERTION_SUCCESS_IMPL(NN_ASSERT_##name arguments, "NN_ASSERT_" #name #arguments); \
        EXPECT_ASSERTION_SUCCESS_IMPL(NN_SDK_ASSERT_##name arguments, "NN_SDK_ASSERT_" #name #arguments); \
        EXPECT_ASSERTION_SUCCESS_IMPL(NN_SDK_REQUIRES_##name arguments, "NN_SDK_REQUIRES_" #name #arguments); \
    } while (NN_STATIC_CONDITION(0))

#define EXPECT_NAMED_ASSERTIONS_FAILURE(name, arguments) \
    do \
    { \
        EXPECT_ASSERTION_FAILURE_IMPL(NN_ASSERT_##name arguments, "NN_ASSERT_" #name #arguments); \
        EXPECT_ASSERTION_FAILURE_IMPL(NN_SDK_ASSERT_##name arguments, "NN_SDK_ASSERT_" #name #arguments); \
        EXPECT_ASSERTION_FAILURE_IMPL(NN_SDK_REQUIRES_##name arguments, "NN_SDK_REQUIRES_" #name #arguments); \
    } while (NN_STATIC_CONDITION(0))

#elif defined(NN_SDK_BUILD_RELEASE)

#define EXPECT_NAMED_ASSERTIONS_SUCCESS(name, arguments) \
    do \
    { \
        EXPECT_ASSERTION_SUCCESS_IMPL(NN_ASSERT_##name arguments, "NN_ASSERT_" #name #arguments); \
    } while (NN_STATIC_CONDITION(0))

#define EXPECT_NAMED_ASSERTIONS_FAILURE(name, arguments) \
    do \
    { \
        EXPECT_ASSERTION_FAILURE_IMPL(NN_ASSERT_##name arguments, "NN_ASSERT_" #name #arguments); \
    } while (NN_STATIC_CONDITION(0))

#endif

NN_PRAGMA_PUSH_WARNINGS
NNT_DIAG_DISABLE_WARNING_STATIC_CONDITION
// インターフェースが揃っていることをテスト
TEST(AssertTest, Interface)
{
    NN_ASSERT(true);
    NN_ASSERT(true, "failed");

    NN_SDK_ASSERT(true);
    NN_SDK_ASSERT(true, "failed");

    NN_SDK_REQUIRES(true);
    NN_SDK_REQUIRES(true, "failed");
}

TEST(AssertTest, PointerVariation)
{
    int v;
    int* p1 = &v;
    const int* p2 = &v;
    volatile int* p3 = &v;
    const volatile int* p4 = &v;

    std::unique_ptr<int> u1(new int);

    NN_ASSERT_NOT_NULL(p1);
    NN_ASSERT_NOT_NULL(p2);
    NN_ASSERT_NOT_NULL(p3);
    NN_ASSERT_NOT_NULL(p4);
    NN_ASSERT_NOT_NULL(u1);
}

// ヌルポインタをデリファレンスした参照を渡した場合でも、assertion 失敗することを確認するテスト
//
// ヌルポインタをデリファレンスする操作は未定義動作になるため、
// コンパイラによっては参照がヌルポインタではないことを仮定して最適化を行うものがある。
// その場合は assertion 失敗 しない（テストが失敗する）。
//
// Diag ライブラリでは、*_NOT_NULL の実装をインラインではなくライブラリに分離することで、
// 最適化によって、assertion 失敗の分岐が削除されることを防いでいる。
NN_NOINLINE void ExpectFailureAssertReferenceNotNull(int &n)
{
    EXPECT_NAMED_ASSERTIONS_FAILURE(NOT_NULL, (&n));
}

// 意味づけされた Assert マクロのテスト
TEST(AssertTest, AssertNotNull)
{
    {
        const int n = 0;
        EXPECT_NAMED_ASSERTIONS_SUCCESS(NOT_NULL, (&n));
    }

    EXPECT_NAMED_ASSERTIONS_FAILURE(NOT_NULL, (static_cast<void*>(NULL)));

    ExpectFailureAssertReferenceNotNull(*g_NullPtrToInt);
}

TEST(AssertTest, AssertEqual)
{
    EXPECT_NAMED_ASSERTIONS_SUCCESS(EQUAL, (1, 1));
    EXPECT_NAMED_ASSERTIONS_SUCCESS(EQUAL, (-1, -1));
    EXPECT_NAMED_ASSERTIONS_FAILURE(EQUAL, (1, -1));
    EXPECT_NAMED_ASSERTIONS_FAILURE(EQUAL, (0, -1));
}

TEST(AssertTest, AssertNotEqual)
{
    EXPECT_NAMED_ASSERTIONS_FAILURE(NOT_EQUAL, (1, 1));
    EXPECT_NAMED_ASSERTIONS_FAILURE(NOT_EQUAL, (-1, -1));
    EXPECT_NAMED_ASSERTIONS_SUCCESS(NOT_EQUAL, (1, -1));
    EXPECT_NAMED_ASSERTIONS_SUCCESS(NOT_EQUAL, (0, -1));
}

TEST(AssertTest, AssertLess)
{
    EXPECT_NAMED_ASSERTIONS_SUCCESS(LESS, (0, 1));
    EXPECT_NAMED_ASSERTIONS_SUCCESS(LESS, (-1, 1));
    EXPECT_NAMED_ASSERTIONS_FAILURE(LESS, (1, 1));
    EXPECT_NAMED_ASSERTIONS_FAILURE(LESS, (-1, -1));
    EXPECT_NAMED_ASSERTIONS_FAILURE(LESS, (1, -1));
    EXPECT_NAMED_ASSERTIONS_FAILURE(LESS, (0, -1));
}

TEST(AssertTest, AssertLessEqual)
{
    EXPECT_NAMED_ASSERTIONS_SUCCESS(LESS_EQUAL, (0, 1));
    EXPECT_NAMED_ASSERTIONS_SUCCESS(LESS_EQUAL, (-1, 1));
    EXPECT_NAMED_ASSERTIONS_SUCCESS(LESS_EQUAL, (1, 1));
    EXPECT_NAMED_ASSERTIONS_SUCCESS(LESS_EQUAL, (-1, -1));
    EXPECT_NAMED_ASSERTIONS_FAILURE(LESS_EQUAL, (1, -1));
    EXPECT_NAMED_ASSERTIONS_FAILURE(LESS_EQUAL, (0, -1));
}

TEST(AssertTest, AssertGreater)
{
    EXPECT_NAMED_ASSERTIONS_FAILURE(GREATER, (0, 1));
    EXPECT_NAMED_ASSERTIONS_FAILURE(GREATER, (-1, 1));
    EXPECT_NAMED_ASSERTIONS_FAILURE(GREATER, (1, 1));
    EXPECT_NAMED_ASSERTIONS_FAILURE(GREATER, (-1, -1));
    EXPECT_NAMED_ASSERTIONS_SUCCESS(GREATER, (1, -1));
    EXPECT_NAMED_ASSERTIONS_SUCCESS(GREATER, (0, -1));
}

TEST(AssertTest, AssertGreaterEqual)
{
    EXPECT_NAMED_ASSERTIONS_FAILURE(GREATER_EQUAL, (0, 1));
    EXPECT_NAMED_ASSERTIONS_FAILURE(GREATER_EQUAL, (-1, 1));
    EXPECT_NAMED_ASSERTIONS_SUCCESS(GREATER_EQUAL, (1, 1));
    EXPECT_NAMED_ASSERTIONS_SUCCESS(GREATER_EQUAL, (-1, -1));
    EXPECT_NAMED_ASSERTIONS_SUCCESS(GREATER_EQUAL, (1, -1));
    EXPECT_NAMED_ASSERTIONS_SUCCESS(GREATER_EQUAL, (0, -1));
}

TEST(AssertTest, AssertStringEqual)
{
    EXPECT_NAMED_ASSERTIONS_SUCCESS(STRING_EQUAL, ("a", "a"));
    EXPECT_NAMED_ASSERTIONS_SUCCESS(STRING_EQUAL, ("abc", "abc"));
    EXPECT_NAMED_ASSERTIONS_SUCCESS(STRING_EQUAL, ("abc\n", "abc\n"));
    EXPECT_NAMED_ASSERTIONS_FAILURE(STRING_EQUAL, (nullptr, nullptr));
    EXPECT_NAMED_ASSERTIONS_FAILURE(STRING_EQUAL, (nullptr, "a"));
    EXPECT_NAMED_ASSERTIONS_FAILURE(STRING_EQUAL, ("a", nullptr));
    EXPECT_NAMED_ASSERTIONS_FAILURE(STRING_EQUAL, ("a", "b"));
    EXPECT_NAMED_ASSERTIONS_FAILURE(STRING_EQUAL, ("abc", "xyz\n"));
}

TEST(AssertTest, AssertStringNotEqual)
{
    EXPECT_NAMED_ASSERTIONS_FAILURE(STRING_NOT_EQUAL, ("a", "a"));
    EXPECT_NAMED_ASSERTIONS_FAILURE(STRING_NOT_EQUAL, ("abc", "abc"));
    EXPECT_NAMED_ASSERTIONS_FAILURE(STRING_NOT_EQUAL, ("abc\n", "abc\n"));
    EXPECT_NAMED_ASSERTIONS_FAILURE(STRING_NOT_EQUAL, (nullptr, nullptr));
    EXPECT_NAMED_ASSERTIONS_FAILURE(STRING_NOT_EQUAL, (nullptr, "a"));
    EXPECT_NAMED_ASSERTIONS_FAILURE(STRING_NOT_EQUAL, ("a", nullptr));
    EXPECT_NAMED_ASSERTIONS_SUCCESS(STRING_NOT_EQUAL, ("a", "b"));
    EXPECT_NAMED_ASSERTIONS_SUCCESS(STRING_NOT_EQUAL, ("abc", "xyz\n"));
}

TEST(AssertTest, AssertAligned)
{
    EXPECT_NAMED_ASSERTIONS_SUCCESS(ALIGNED, (1, 1));
    EXPECT_NAMED_ASSERTIONS_SUCCESS(ALIGNED, (2, 1));
    EXPECT_NAMED_ASSERTIONS_SUCCESS(ALIGNED, (12, 4));
    EXPECT_NAMED_ASSERTIONS_FAILURE(ALIGNED, (12, 8));
    EXPECT_NAMED_ASSERTIONS_FAILURE(ALIGNED, (32, 64));
}

TEST(AssertTest, AssertRange)
{
    EXPECT_NAMED_ASSERTIONS_FAILURE(RANGE, ( 0, 1, 16));
    EXPECT_NAMED_ASSERTIONS_SUCCESS(RANGE, ( 1, 1, 16));
    EXPECT_NAMED_ASSERTIONS_SUCCESS(RANGE, ( 2, 1, 16));
    EXPECT_NAMED_ASSERTIONS_SUCCESS(RANGE, (15, 1, 16));
    EXPECT_NAMED_ASSERTIONS_FAILURE(RANGE, (16, 1, 16));
    EXPECT_NAMED_ASSERTIONS_FAILURE(RANGE, (17, 1, 16));
}

TEST(AssertTest, AssertMinMax)
{
    EXPECT_NAMED_ASSERTIONS_FAILURE(MINMAX, ( 0, 1, 16));
    EXPECT_NAMED_ASSERTIONS_SUCCESS(MINMAX, ( 1, 1, 16));
    EXPECT_NAMED_ASSERTIONS_SUCCESS(MINMAX, ( 2, 1, 16));
    EXPECT_NAMED_ASSERTIONS_SUCCESS(MINMAX, (15, 1, 16));
    EXPECT_NAMED_ASSERTIONS_SUCCESS(MINMAX, (16, 1, 16));
    EXPECT_NAMED_ASSERTIONS_FAILURE(MINMAX, (17, 1, 16));
}

#if defined(NN_BUILD_CONFIG_OS_WIN)

// Assertion 失敗で終了することをテスト
// Death テストは別プロセスで実行されるので、Assertion 失敗ハンドラはデフォルトになります。
TEST(AssertDeathTest, AssertionFailure)
{
    EXPECT_DEATH(
        NN_ASSERT(false),
        "");
}

// メッセージを出力して Assertion 失敗で終了することをテスト
TEST(AssertDeathTest, AssertionFailureMessage)
{
    EXPECT_DEATH(
        NN_ASSERT(false, ""),
        "");
}

// Assertion 成功して正常終了（終了コード 0）する
void AssertionSuccessAndExit0()
{
    NN_ASSERT(true);
    NN_ASSERT(true, "");
    std::exit(0);
}

// Assertion 成功の場合に異常終了しないことをテスト
TEST(AssertDeathTest, AssertionSuccess)
{
    EXPECT_EXIT(
        AssertionSuccessAndExit0(),
        [](int exitCode) { return exitCode == 0; },
        "");
}

#endif // defined(NN_BUILD_CONFIG_OS_WIN)

NN_PRAGMA_POP_WARNINGS
