﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

//-----------------------------------------------------------------------------
//  デバイスツリーの API のテスト
//-----------------------------------------------------------------------------

#include <algorithm>

#include <nn/nn_Common.h>
#include <nn/nn_Log.h>

#include <nn/dt.h>
#include <nn/dt/detail/dt_InternalTypes.h>

#include <nnt/nntest.h>

namespace nnt { namespace dt {

TEST(StringList, Basic)
{
    const char stringList[] = "string0\0string1\0string2";
    const char* expected[] = { "string0", "string1", "string2" };

    int index;

    auto pData = reinterpret_cast<const char*>(stringList);
    auto list = nn::dt::StringForwardList(pData, pData + sizeof(stringList));

    ASSERT_STREQ(expected[0], list.front());

    index = 0;
    for (auto iter = list.begin(); iter != list.end(); iter++)
    {
        ASSERT_STREQ(expected[index], *iter);
        index++;
    }
    ASSERT_EQ(3, index);

    index = 0;
    for (auto iter = list.cbegin(); iter != list.cend(); iter++)
    {
        ASSERT_STREQ(expected[index], *iter);
        index++;
    }
    ASSERT_EQ(3, index);

    index = 0;
    for (auto str: list)
    {
        ASSERT_STREQ(expected[index], str);
        index++;
    }
    ASSERT_EQ(3, index);

    index = 0;
    std::for_each(list.begin(), list.end(), [&](const char* str) {
        ASSERT_STREQ(expected[index], str);
        index++;
    });
    ASSERT_EQ(3, index);

    index = 0;
    std::for_each(list.cbegin(), list.cend(), [&](const char* str) {
        ASSERT_STREQ(expected[index], str);
        index++;
    });
    ASSERT_EQ(3, index);

    ASSERT_FALSE(list.empty());
}

TEST(StringList, Empty)
{
    const char stringList[] = "string0\0string1\0string2";
    auto pData = reinterpret_cast<const char*>(stringList);

    auto list = nn::dt::StringForwardList(pData, pData);

    ASSERT_TRUE(list.begin() == list.end());
    ASSERT_TRUE(list.cbegin() == list.cend());
    ASSERT_TRUE(list.empty());
}

TEST(StringList, Assign)
{
    const char stringList0[] = "string0\0string1\0string2";
    const char* expected0[] = { "string0", "string1", "string2" };

    const char stringList1[] = "string3";
    const char* expected1[] = { "string3" };

    int index;

    auto pData0 = reinterpret_cast<const char*>(stringList0);
    auto list = nn::dt::StringForwardList(pData0, pData0 + sizeof(stringList0));

    index = 0;
    for (auto iter = list.begin(); iter != list.end(); iter++)
    {
        ASSERT_STREQ(expected0[index], *iter);
        index++;
    }
    ASSERT_EQ(3, index);

    auto pData1 = reinterpret_cast<const char*>(stringList1);
    list.assign(pData1, pData1 + sizeof(stringList1));

    index = 0;
    for (auto iter = list.begin(); iter != list.end(); iter++)
    {
        ASSERT_STREQ(expected1[index], *iter);
        index++;
    }
    ASSERT_EQ(1, index);
}

TEST(StringList, Swap)
{
    const char stringList0[] = "string0\0string1\0string2";
    const char* expected0[] = { "string0", "string1", "string2" };

    const char stringList1[] = "string3";
    const char* expected1[] = { "string3" };

    int index;

    auto pData0 = reinterpret_cast<const char*>(stringList0);
    auto list0 = nn::dt::StringForwardList(pData0, pData0 + sizeof(stringList0));

    auto pData1 = reinterpret_cast<const char*>(stringList1);
    auto list1 = nn::dt::StringForwardList(pData1, pData1 + sizeof(stringList1));

    swap(list0, list1);

    index = 0;
    for (auto iter = list0.begin(); iter != list0.end(); iter++)
    {
        ASSERT_STREQ(expected1[index], *iter);
        index++;
    }
    ASSERT_EQ(1, index);

    index = 0;
    for (auto iter = list1.begin(); iter != list1.end(); iter++)
    {
        ASSERT_STREQ(expected0[index], *iter);
        index++;
    }
    ASSERT_EQ(3, index);
}

TEST(StringList, Comparison)
{
    const char stringList0[] = "string0\0string1\0string2";
    const char stringList1[] = "string3";

    auto pData0 = reinterpret_cast<const char*>(stringList0);
    auto list0 = nn::dt::StringForwardList(pData0, pData0 + sizeof(stringList0));

    ASSERT_TRUE (list0 == list0);
    ASSERT_FALSE(list0 != list0);
    ASSERT_FALSE(list0 <  list0);
    ASSERT_FALSE(list0 >  list0);
    ASSERT_TRUE (list0 <= list0);
    ASSERT_TRUE (list0 >= list0);

    auto pData1 = reinterpret_cast<const char*>(stringList1);
    auto list1 = nn::dt::StringForwardList(pData1, pData1 + sizeof(stringList1));

    ASSERT_FALSE(list0 == list1);
    ASSERT_TRUE (list0 != list1);
    ASSERT_TRUE (list0 <  list1);
    ASSERT_FALSE(list0 >  list1);
    ASSERT_TRUE (list0 <= list1);
    ASSERT_FALSE(list0 >= list1);

    // 同じ内容の stringList を異なるアドレスに配置して比較
    char bufferA[sizeof(stringList0)];
    char bufferB[sizeof(stringList0)];

    std::memcpy(bufferA, pData0, sizeof(stringList0));
    std::memcpy(bufferB, pData0, sizeof(stringList0));

    auto listA = nn::dt::StringForwardList(bufferA, bufferA + sizeof(stringList0));
    auto listB = nn::dt::StringForwardList(bufferB, bufferB + sizeof(stringList0));

    ASSERT_TRUE (listA == listB);
    ASSERT_FALSE(listA != listB);
    ASSERT_FALSE(listA <  listB);
    ASSERT_FALSE(listA >  listB);
    ASSERT_TRUE (listA <= listB);
    ASSERT_TRUE (listA >= listB);
}

}}
