﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

//-----------------------------------------------------------------------------
//  デバイスツリーの API のテスト
//-----------------------------------------------------------------------------

#include <nn/nn_Common.h>
#include <nn/nn_Log.h>

#include <nn/dt.h>
#include <nn/dt/detail/dt_InternalTypes.h>

#include <nnt/nntest.h>

#include "testDt_Common.h"

namespace nnt { namespace dt {

TEST(NodeApi, FindNodeByPath)
{
    nn::Result result;
    nn::dt::Node node;

    // 正常系
    result = nn::dt::FindNodeByPath(&node, "/cpus/cpu@0");
    NNT_DT_ASSERT_RESULT_SUCCESS(result);
    ASSERT_NE(nn::dt::detail::InvalidNodeOffset, node.GetOffset());

    // 異常系
    result = nn::dt::FindNodeByPath(nullptr, "/cpus/cpu@0");
    NNT_DT_ASSERT_RESULT_INCLUDED(nn::dt::ResultInvalidArgument(), result);
    result = nn::dt::FindNodeByPath(&node, nullptr);
    NNT_DT_ASSERT_RESULT_INCLUDED(nn::dt::ResultInvalidArgument(), result);

    result = nn::dt::FindNodeByPath(&node, "/cpus/cpu@0_not_exist");
    NNT_DT_ASSERT_RESULT_INCLUDED(nn::dt::ResultNodeNotExist(), result);
}

TEST(NodeApi, FindNodeByPHandle)
{
    nn::Result result;
    nn::dt::Node baseNode, node;
    nn::dt::PHandle phandle;
    nn::dt::PHandle invalidPHandle = static_cast<nn::dt::PHandle>(-1);

    // phandle を準備
    result = nn::dt::FindNodeByPath(&baseNode, "/bus/uart");
    NNT_DT_ASSERT_RESULT_SUCCESS(result);

    result = nn::dt::GetProperty<nn::dt::PHandle>(&phandle, &baseNode, "clocks");
    NNT_DT_ASSERT_RESULT_SUCCESS(result);

    // 正常系
    result = nn::dt::FindNodeByPHandle(&node, phandle);
    NNT_DT_ASSERT_RESULT_SUCCESS(result);

    // 異常系
    result = nn::dt::FindNodeByPHandle(nullptr, phandle);
    NNT_DT_ASSERT_RESULT_INCLUDED(nn::dt::ResultInvalidArgument(), result);

    result = nn::dt::FindNodeByPHandle(&node, invalidPHandle);
    NNT_DT_ASSERT_RESULT_INCLUDED(nn::dt::ResultInvalidArgument(), result);
}

TEST(NodeApi, GetCompatibleNodeCount)
{
    nn::Result result;
    int count;

    // 正常系
    result = nn::dt::GetCompatibleNodeCount(&count, "nintendo,cpu");
    NNT_DT_ASSERT_RESULT_SUCCESS(result);
    ASSERT_EQ(4, count);

    result = nn::dt::GetCompatibleNodeCount(&count, "nintendo,clock");
    NNT_DT_ASSERT_RESULT_SUCCESS(result);
    ASSERT_EQ(1, count);

    // 異常系
    result = nn::dt::GetCompatibleNodeCount(&count, "nintendo,not_exist");
    NNT_DT_ASSERT_RESULT_INCLUDED(nn::dt::ResultNodeNotExist(), result);

    result = nn::dt::GetCompatibleNodeCount(nullptr, "nintendo,cpu");
    NNT_DT_ASSERT_RESULT_INCLUDED(nn::dt::ResultInvalidArgument(), result);

    result = nn::dt::GetCompatibleNodeCount(&count, nullptr);
    NNT_DT_ASSERT_RESULT_INCLUDED(nn::dt::ResultInvalidArgument(), result);
}

TEST(NodeApi, ListCompatibleNode)
{
    nn::Result result;
    int count;
    nn::dt::Node bus, cpu0, cpu1, cpu2, cpu3;
    nn::dt::Node nodes[4];

    // 準備
    result = nn::dt::FindNodeByPath(&bus, "/bus");
    NNT_DT_ASSERT_RESULT_SUCCESS(result);

    result = nn::dt::FindNodeByPath(&cpu0, "/cpus/cpu@0");
    NNT_DT_ASSERT_RESULT_SUCCESS(result);

    result = nn::dt::FindNodeByPath(&cpu1, "/cpus/cpu@1");
    NNT_DT_ASSERT_RESULT_SUCCESS(result);

    result = nn::dt::FindNodeByPath(&cpu2, "/cpus/cpu@2");
    NNT_DT_ASSERT_RESULT_SUCCESS(result);

    result = nn::dt::FindNodeByPath(&cpu3, "/cpus/cpu@3");
    NNT_DT_ASSERT_RESULT_SUCCESS(result);

    // 正常系
    result = nn::dt::ListCompatibleNode(nodes, &count, sizeof(nodes) / sizeof(nodes[0]), "nintendo,cpu");
    NNT_DT_ASSERT_RESULT_SUCCESS(result);
    ASSERT_EQ(4, count);
    ASSERT_EQ(nodes[0].GetOffset(), cpu0.GetOffset());
    ASSERT_EQ(nodes[1].GetOffset(), cpu1.GetOffset());
    ASSERT_EQ(nodes[2].GetOffset(), cpu2.GetOffset());
    ASSERT_EQ(nodes[3].GetOffset(), cpu3.GetOffset());

    result = nn::dt::ListCompatibleNode(nodes, &count, sizeof(nodes) / sizeof(nodes[0]), "nintendo,cpu2");
    NNT_DT_ASSERT_RESULT_SUCCESS(result);
    ASSERT_EQ(4, count);
    ASSERT_EQ(nodes[0].GetOffset(), cpu0.GetOffset());
    ASSERT_EQ(nodes[1].GetOffset(), cpu1.GetOffset());
    ASSERT_EQ(nodes[2].GetOffset(), cpu2.GetOffset());
    ASSERT_EQ(nodes[3].GetOffset(), cpu3.GetOffset());

    result = nn::dt::ListCompatibleNode(nodes, &count, sizeof(nodes) / sizeof(nodes[0]), "nintendo,bus");
    NNT_DT_ASSERT_RESULT_SUCCESS(result);
    ASSERT_EQ(1, count);
    ASSERT_EQ(nodes[0].GetOffset(), bus.GetOffset());

    // 異常系
    result = nn::dt::ListCompatibleNode(nullptr, &count, sizeof(nodes) / sizeof(nodes[0]), "nintendo,cpu");
    NNT_DT_ASSERT_RESULT_INCLUDED(nn::dt::ResultInvalidArgument(), result);

    result = nn::dt::ListCompatibleNode(nodes, nullptr, sizeof(nodes) / sizeof(nodes[0]), "nintendo,cpu");
    NNT_DT_ASSERT_RESULT_INCLUDED(nn::dt::ResultInvalidArgument(), result);

    result = nn::dt::ListCompatibleNode(nodes, &count, sizeof(nodes) / sizeof(nodes[0]), nullptr);
    NNT_DT_ASSERT_RESULT_INCLUDED(nn::dt::ResultInvalidArgument(), result);

    result = nn::dt::ListCompatibleNode(nodes, &count, 3, "nintendo,cpu");
    NNT_DT_ASSERT_RESULT_INCLUDED(nn::dt::ResultBufferTooSmall(), result);

    result = nn::dt::ListCompatibleNode(nodes, &count, 0, "nintendo,bus");
    NNT_DT_ASSERT_RESULT_INCLUDED(nn::dt::ResultBufferTooSmall(), result);

    result = nn::dt::ListCompatibleNode(nodes, &count, sizeof(nodes) / sizeof(nodes[0]), "not_exist");
    NNT_DT_ASSERT_RESULT_INCLUDED(nn::dt::ResultNodeNotExist(), result);

    result = nn::dt::ListCompatibleNode(nodes, &count, 0, "not_exist");
    NNT_DT_ASSERT_RESULT_INCLUDED(nn::dt::ResultNodeNotExist(), result);
}

TEST(NodeApi, GetParentNode)
{
    nn::Result result;
    nn::dt::Node baseNode, expectedNode, rootNode, node;

    // 準備
    result = nn::dt::FindNodeByPath(&baseNode, "/bus/uart");
    NNT_DT_ASSERT_RESULT_SUCCESS(result);

    result = nn::dt::FindNodeByPath(&expectedNode, "/bus");
    NNT_DT_ASSERT_RESULT_SUCCESS(result);

    result = nn::dt::FindNodeByPath(&rootNode, "/");
    NNT_DT_ASSERT_RESULT_SUCCESS(result);

    // 正常系
    result = nn::dt::GetParentNode(&node, &baseNode);
    NNT_DT_ASSERT_RESULT_SUCCESS(result);
    ASSERT_EQ(expectedNode.GetOffset(), node.GetOffset());
    // TODO: API に GetName や GetPath を用意して、テスト

    // 異常系
    result = nn::dt::GetParentNode(&node, &rootNode);
    NNT_DT_ASSERT_RESULT_INCLUDED(nn::dt::ResultNodeNotExist(), result);

    result = nn::dt::GetParentNode(nullptr, &rootNode);
    NNT_DT_ASSERT_RESULT_INCLUDED(nn::dt::ResultInvalidArgument(), result);

    result = nn::dt::GetParentNode(&node, nullptr);
    NNT_DT_ASSERT_RESULT_INCLUDED(nn::dt::ResultInvalidArgument(), result);
}

TEST(NodeApi, GetChildByName)
{
    nn::Result result;
    nn::dt::Node baseNode, expectedNode, leafNode, node;

    // 準備
    result = nn::dt::FindNodeByPath(&baseNode, "/cpus");
    NNT_DT_ASSERT_RESULT_SUCCESS(result);

    result = nn::dt::FindNodeByPath(&expectedNode, "/cpus/cpu@0");
    NNT_DT_ASSERT_RESULT_SUCCESS(result);

    result = nn::dt::FindNodeByPath(&leafNode, "/cpus/cpu@3");
    NNT_DT_ASSERT_RESULT_SUCCESS(result);

    // 正常系
    result = nn::dt::GetChildNodeByName(&node, &baseNode, "cpu@0");
    NNT_DT_ASSERT_RESULT_SUCCESS(result);
    ASSERT_EQ(expectedNode.GetOffset(), node.GetOffset());

    // 異常系
    result = nn::dt::GetChildNodeByName(nullptr, &baseNode, "not_exist");
    NNT_DT_ASSERT_RESULT_INCLUDED(nn::dt::ResultInvalidArgument(), result);

    result = nn::dt::GetChildNodeByName(&node, nullptr, "not_exist");
    NNT_DT_ASSERT_RESULT_INCLUDED(nn::dt::ResultInvalidArgument(), result);

    result = nn::dt::GetChildNodeByName(&node, &baseNode, nullptr);
    NNT_DT_ASSERT_RESULT_INCLUDED(nn::dt::ResultInvalidArgument(), result);

    result = nn::dt::GetChildNodeByName(&node, &baseNode, "not_exist");
    NNT_DT_ASSERT_RESULT_INCLUDED(nn::dt::ResultNodeNotExist(), result);
}

TEST(NodeApi, GetChildNodeCount)
{
    nn::Result result;
    nn::dt::Node cpu, bus, uart;
    int count;

    // 準備
    result = nn::dt::FindNodeByPath(&cpu, "/cpus");
    NNT_DT_ASSERT_RESULT_SUCCESS(result);

    result = nn::dt::FindNodeByPath(&bus, "/bus");
    NNT_DT_ASSERT_RESULT_SUCCESS(result);

    result = nn::dt::FindNodeByPath(&uart, "/bus/uart");
    NNT_DT_ASSERT_RESULT_SUCCESS(result);

    // 正常系
    result = nn::dt::GetChildNodeCount(&count, &cpu);
    NNT_DT_ASSERT_RESULT_SUCCESS(result);
    ASSERT_EQ(4, count);

    result = nn::dt::GetChildNodeCount(&count, &bus);
    NNT_DT_ASSERT_RESULT_SUCCESS(result);
    ASSERT_EQ(2, count);

    // 異常系
    result = nn::dt::GetChildNodeCount(&count, &uart);
    NNT_DT_ASSERT_RESULT_INCLUDED(nn::dt::ResultNodeNotExist(), result);

    result = nn::dt::GetChildNodeCount(nullptr, &cpu);
    NNT_DT_ASSERT_RESULT_INCLUDED(nn::dt::ResultInvalidArgument(), result);

    result = nn::dt::GetChildNodeCount(&count, nullptr);
    NNT_DT_ASSERT_RESULT_INCLUDED(nn::dt::ResultInvalidArgument(), result);
}

TEST(NodeApi, ListChildNode)
{
    nn::Result result;
    nn::dt::Node cpu, cpu0, cpu1, cpu2, cpu3;

    nn::dt::Node buffer[4];
    int count;

    // 準備
    result = nn::dt::FindNodeByPath(&cpu, "/cpus");
    NNT_DT_ASSERT_RESULT_SUCCESS(result);

    result = nn::dt::FindNodeByPath(&cpu0, "/cpus/cpu@0");
    NNT_DT_ASSERT_RESULT_SUCCESS(result);

    result = nn::dt::FindNodeByPath(&cpu1, "/cpus/cpu@1");
    NNT_DT_ASSERT_RESULT_SUCCESS(result);

    result = nn::dt::FindNodeByPath(&cpu2, "/cpus/cpu@2");
    NNT_DT_ASSERT_RESULT_SUCCESS(result);

    result = nn::dt::FindNodeByPath(&cpu3, "/cpus/cpu@3");
    NNT_DT_ASSERT_RESULT_SUCCESS(result);

    // 正常系
    result = nn::dt::ListChildNode(buffer, &count, sizeof(buffer) / sizeof(buffer[0]), &cpu);
    NNT_DT_ASSERT_RESULT_SUCCESS(result);
    ASSERT_EQ(cpu0.GetOffset(), buffer[0].GetOffset());
    ASSERT_EQ(cpu1.GetOffset(), buffer[1].GetOffset());
    ASSERT_EQ(cpu2.GetOffset(), buffer[2].GetOffset());
    ASSERT_EQ(cpu3.GetOffset(), buffer[3].GetOffset());

    // 異常系
    result = nn::dt::ListChildNode(buffer, &count, 1, &cpu);
    NNT_DT_ASSERT_RESULT_INCLUDED(nn::dt::ResultBufferTooSmall(), result);

    result = nn::dt::ListChildNode(buffer, &count, 0, &cpu);
    NNT_DT_ASSERT_RESULT_INCLUDED(nn::dt::ResultBufferTooSmall(), result);

    result = nn::dt::ListChildNode(nullptr, &count, sizeof(buffer) / sizeof(buffer[0]), &cpu);
    NNT_DT_ASSERT_RESULT_INCLUDED(nn::dt::ResultInvalidArgument(), result);

    result = nn::dt::ListChildNode(buffer, nullptr, sizeof(buffer) / sizeof(buffer[0]), &cpu);
    NNT_DT_ASSERT_RESULT_INCLUDED(nn::dt::ResultInvalidArgument(), result);

    result = nn::dt::ListChildNode(buffer, &count, sizeof(buffer) / sizeof(buffer[0]), nullptr);
    NNT_DT_ASSERT_RESULT_INCLUDED(nn::dt::ResultInvalidArgument(), result);
}

TEST(NodeApi, GetFirstChildNode)
{
    nn::Result result;
    nn::dt::Node baseNode, expectedNode, leafNode, node;

    // 準備
    result = nn::dt::FindNodeByPath(&baseNode, "/cpus");
    NNT_DT_ASSERT_RESULT_SUCCESS(result);

    result = nn::dt::FindNodeByPath(&expectedNode, "/cpus/cpu@0");
    NNT_DT_ASSERT_RESULT_SUCCESS(result);

    result = nn::dt::FindNodeByPath(&leafNode, "/cpus/cpu@0");
    NNT_DT_ASSERT_RESULT_SUCCESS(result);

    // 正常系
    result = nn::dt::GetFirstChildNode(&node, &baseNode);
    NNT_DT_ASSERT_RESULT_SUCCESS(result);
    ASSERT_EQ(expectedNode.GetOffset(), node.GetOffset());

    // 異常系
    result = nn::dt::GetFirstChildNode(&node, &leafNode);
    NNT_DT_ASSERT_RESULT_INCLUDED(nn::dt::ResultNodeNotExist(), result);

    result = nn::dt::GetFirstChildNode(nullptr, &baseNode);
    NNT_DT_ASSERT_RESULT_INCLUDED(nn::dt::ResultInvalidArgument(), result);

    result = nn::dt::GetFirstChildNode(&node, nullptr);
    NNT_DT_ASSERT_RESULT_INCLUDED(nn::dt::ResultInvalidArgument(), result);
}

TEST(NodeApi, GetNextSiblingNode)
{
    nn::Result result;
    nn::dt::Node baseNode, expectedNode, lastNode, node;

    // 準備
    result = nn::dt::FindNodeByPath(&baseNode, "/cpus/cpu@0");
    NNT_DT_ASSERT_RESULT_SUCCESS(result);

    result = nn::dt::FindNodeByPath(&expectedNode, "/cpus/cpu@1");
    NNT_DT_ASSERT_RESULT_SUCCESS(result);

    result = nn::dt::FindNodeByPath(&lastNode, "/cpus/cpu@3");
    NNT_DT_ASSERT_RESULT_SUCCESS(result);

    // 正常系
    result = nn::dt::GetNextSiblingNode(&node, &baseNode);
    NNT_DT_ASSERT_RESULT_SUCCESS(result);
    ASSERT_EQ(expectedNode.GetOffset(), node.GetOffset());

    // 異常系
    result = nn::dt::GetNextSiblingNode(&node, &lastNode);
    NNT_DT_ASSERT_RESULT_INCLUDED(nn::dt::ResultNodeNotExist(), result);

    result = nn::dt::GetNextSiblingNode(nullptr, &baseNode);
    NNT_DT_ASSERT_RESULT_INCLUDED(nn::dt::ResultInvalidArgument(), result);

    result = nn::dt::GetNextSiblingNode(&node, nullptr);
    NNT_DT_ASSERT_RESULT_INCLUDED(nn::dt::ResultInvalidArgument(), result);
}

TEST(NodeApi, GetValueOfAddressCells)
{
    nn::Result result;
    nn::dt::Node cpu1, uart;
    uint32_t addressCells;

    // 準備
    result = nn::dt::FindNodeByPath(&cpu1, "/cpus/cpu@1");
    NNT_DT_ASSERT_RESULT_SUCCESS(result);

    result = nn::dt::FindNodeByPath(&uart, "/bus/uart");
    NNT_DT_ASSERT_RESULT_SUCCESS(result);

    // 正常系
    result = nn::dt::GetValueOfAddressCells(&addressCells, &cpu1);
    NNT_DT_ASSERT_RESULT_SUCCESS(result);
    ASSERT_EQ(1, addressCells);

    result = nn::dt::GetValueOfAddressCells(&addressCells, &uart);
    NNT_DT_ASSERT_RESULT_SUCCESS(result);
    ASSERT_EQ(2, addressCells);

    // 異常系
    result = nn::dt::GetValueOfAddressCells(nullptr, &uart);
    NNT_DT_ASSERT_RESULT_INCLUDED(nn::dt::ResultInvalidArgument(), result);

    result = nn::dt::GetValueOfAddressCells(&addressCells, nullptr);
    NNT_DT_ASSERT_RESULT_INCLUDED(nn::dt::ResultInvalidArgument(), result);
}

TEST(NodeApi, GetValueOfSizeCells)
{
    nn::Result result;
    nn::dt::Node cpu1, uart;
    uint32_t sizeCells;

    // 準備
    result = nn::dt::FindNodeByPath(&cpu1, "/cpus/cpu@1");
    NNT_DT_ASSERT_RESULT_SUCCESS(result);

    result = nn::dt::FindNodeByPath(&uart, "/bus/uart");
    NNT_DT_ASSERT_RESULT_SUCCESS(result);

    // 正常系
    result = nn::dt::GetValueOfSizeCells(&sizeCells, &cpu1);
    NNT_DT_ASSERT_RESULT_SUCCESS(result);
    ASSERT_EQ(0, sizeCells);

    result = nn::dt::GetValueOfSizeCells(&sizeCells, &uart);
    NNT_DT_ASSERT_RESULT_SUCCESS(result);
    ASSERT_EQ(1, sizeCells);

    // 異常系
    result = nn::dt::GetValueOfSizeCells(nullptr, &uart);
    NNT_DT_ASSERT_RESULT_INCLUDED(nn::dt::ResultInvalidArgument(), result);

    result = nn::dt::GetValueOfSizeCells(&sizeCells, nullptr);
    NNT_DT_ASSERT_RESULT_INCLUDED(nn::dt::ResultInvalidArgument(), result);
}

}}
