﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

//-----------------------------------------------------------------------------
//  デバイスツリーの API のテスト
//-----------------------------------------------------------------------------

#include <cstring>

#include <nn/nn_Common.h>
#include <nn/nn_Log.h>

#include <nn/dt.h>
#include <nn/dt/dt_DebugApi.h>
#include <nn/dt/detail/dt_InternalTypes.h>

#include <nnt/nntest.h>

#include "testDt_Common.h"

namespace nnt { namespace dt {

TEST(DebugApi, GetNodePath)
{
    nn::Result result;
    nn::dt::Node node;

    const char* path = "/cpus/cpu@0";
    const size_t bufferSize = 16;
    char buffer[bufferSize];

    // 準備
    result = nn::dt::FindNodeByPath(&node, path);
    NNT_DT_ASSERT_RESULT_SUCCESS(result);

    // 正常系
    result = nn::dt::GetNodePath(buffer, bufferSize, &node);
    NNT_DT_ASSERT_RESULT_SUCCESS(result);
    ASSERT_STREQ(path, buffer);

    // 異常系
    result = nn::dt::GetNodePath(buffer, std::strlen(path), &node);
    NNT_DT_ASSERT_RESULT_INCLUDED(nn::dt::ResultBufferTooSmall(), result);

    result = nn::dt::GetNodePath(nullptr, bufferSize, &node);
    NNT_DT_ASSERT_RESULT_INCLUDED(nn::dt::ResultInvalidArgument(), result);

    result = nn::dt::GetNodePath(buffer, bufferSize, nullptr);
    NNT_DT_ASSERT_RESULT_INCLUDED(nn::dt::ResultInvalidArgument(), result);
}

// ダンプ結果を目視で確認すること
TEST(DebugApi, Dump)
{
    nn::Result result;
    nn::dt::Node cpus;
    nn::dt::Node bus;
    nn::dt::Node root;

    // 準備
    result = nn::dt::FindNodeByPath(&cpus, "/cpus");
    NNT_DT_ASSERT_RESULT_SUCCESS(result);

    result = nn::dt::FindNodeByPath(&bus, "/bus");
    NNT_DT_ASSERT_RESULT_SUCCESS(result);

    result = nn::dt::FindNodeByPath(&root, "/");
    NNT_DT_ASSERT_RESULT_SUCCESS(result);

    // 正常系
    NN_LOG("// DumpNode '/cpus'\n");
    result = nn::dt::DumpNode(&cpus);
    NNT_DT_ASSERT_RESULT_SUCCESS(result);

    NN_LOG("// DumpNode '/cpus' (depth=0)\n");
    result = nn::dt::DumpNode(&cpus, 0);
    NNT_DT_ASSERT_RESULT_SUCCESS(result);

    NN_LOG("// DumpNode '/cpus' (depth=1)\n");
    result = nn::dt::DumpNode(&cpus, 1);
    NNT_DT_ASSERT_RESULT_SUCCESS(result);

    NN_LOG("// DumpNode '/bus' (depth=5)\n");
    result = nn::dt::DumpNode(&bus, 5);
    NNT_DT_ASSERT_RESULT_SUCCESS(result);

    NN_LOG("// DumpNode '/' (depth=-1)\n");
    result = nn::dt::DumpNode(&root, -1);
    NNT_DT_ASSERT_RESULT_SUCCESS(result);
}

}}
