﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <array>
#include <vector>

#include <nn/nn_Common.h>
#include <nn/nn_Abort.h>
#include <nn/nn_DeviceCode.h>
#include <nn/ddsf/ddsf_IDriver.h>
#include <nn/ddsf/ddsf_IDevice.h>
#include <nn/ddsf/ddsf_ISession.h>

#include <nn/nn_Log.h>
#include <nnt/nntest.h>
#include <nnt/result/testResult_Assert.h>

// UART サブシステム（コンセプト検証用スタブ）の名前空間
namespace nn { namespace uart_t {

    using namespace ::nn;

    // UART サブシステムのみが参照する名前空間
    namespace detail {

        struct DriverEntity
        {
            ddsf::IDriver::List driverList;
        };
        DriverEntity g_DriverEntity;
    }

    // ドライバ（実装）が参照する名前空間
    namespace driver {

        class UartPort : public ddsf::IDevice
        {
            NN_DDSF_CAST_SAFE_DECL;
        };
        NN_DDSF_CAST_SAFE_DEFINE(UartPort, ddsf::IDevice);

        class IUartDriver : public ddsf::IDriver
        {
            NN_DDSF_CAST_SAFE_DECL;
        public:
            virtual Result GetBaudRate(ddsf::IDevice& device, int* pOutBaudRate) NN_NOEXCEPT = 0;
            virtual Result SetBaudRate(ddsf::IDevice& device, int baudRate) NN_NOEXCEPT = 0;
            virtual Result Send(ddsf::IDevice& device, const char* pBuffer, size_t bufferSize) NN_NOEXCEPT = 0;
        };
        NN_DDSF_CAST_SAFE_DEFINE(IUartDriver, ddsf::IDriver);

        Result RegisterDriver(ddsf::IDriver& driver) NN_NOEXCEPT
        {
            driver.AddTo(&uart_t::detail::g_DriverEntity.driverList);
            NN_RESULT_SUCCESS;
        }
    }

    class UartSession : public ddsf::ISession
    {
        NN_DDSF_CAST_SAFE_DECL;
    };
    NN_DDSF_CAST_SAFE_DEFINE(UartSession, ddsf::ISession);

    void Initialize() NN_NOEXCEPT
    {
        detail::g_DriverEntity.driverList.clear();
    }
    void Finalize() NN_NOEXCEPT
    {
        detail::g_DriverEntity.driverList.clear();
    }

    Result Open(UartSession* pOutSession, DeviceCode devicecode, ddsf::AccessMode accessMode) NN_NOEXCEPT
    {
        NN_UNUSED(devicecode);

        bool needsRollback = true;

        // とりあえず特定のドライバと特定のポート。実際は device code から対応するポートを引っ張ってくる
        ddsf::IDriver& driver = detail::g_DriverEntity.driverList.front();
        driver::UartPort* pOutPort = nullptr;
        driver.ForEachDevice(
            [&pOutPort](ddsf::IDevice* pDevice) -> bool
            {
                if ( pOutPort == nullptr )
                {
                    pOutPort = pDevice->SafeCastToPointer<driver::UartPort>();
                    return false; // Break ForEach loop
                }
                return true;
            }
        );
        NN_SDK_ASSERT_NOT_NULL(pOutPort);

        NN_RESULT_DO(ddsf::OpenSession(pOutPort, pOutSession, accessMode));
        NN_UTIL_SCOPE_EXIT
        {
            if ( needsRollback )
            {
                ddsf::CloseSession(pOutSession);
            }
        };

        needsRollback = false;
        NN_RESULT_SUCCESS;
    }
    void Close(UartSession* pSession) NN_NOEXCEPT
    {
        ddsf::CloseSession(pSession);
    }

    Result GetBaudRate(UartSession* pSession, int* pOutBaudRate) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(pSession->IsOpen());
        NN_SDK_REQUIRES_NOT_NULL(pOutBaudRate);
        auto& device = pSession->GetDevice();
        NN_RESULT_DO(device.GetDriver().SafeCastTo<driver::IUartDriver>().GetBaudRate(device, pOutBaudRate));
        NN_RESULT_SUCCESS;
    }

    Result SetBaudRate(UartSession* pSession, int baudRate) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(pSession->IsOpen());
        auto& device = pSession->GetDevice();
        NN_RESULT_DO(device.GetDriver().SafeCastTo<driver::IUartDriver>().SetBaudRate(device, baudRate));
        NN_RESULT_SUCCESS;
    }

    Result Send(UartSession* pSession, const char* pBuffer, size_t bufferSize) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(pSession->IsOpen());
        auto& device = pSession->GetDevice();
        NN_RESULT_DO(device.GetDriver().SafeCastTo<driver::IUartDriver>().Send(device, pBuffer, bufferSize));
        NN_RESULT_SUCCESS;
    }

}} // nn::uart_t

// UART サブシステム向けの特定ハードウェア向けドライバ実装（コンセプト検証用スタブ）の名前空間
namespace nnd { namespace uart_t { namespace specificchip {

    class SpecificUartPort : public nn::uart_t::driver::UartPort
    {
        NN_DDSF_CAST_SAFE_DECL;
    public:
        void Initialize(int specificId) NN_NOEXCEPT
        {
            m_SpecificId = specificId;
        }

        int GetBaudRate() NN_NOEXCEPT
        {
            // Some register access

            NN_LOG("SpecificUartPort: Get baud rate %d\n", m_BaudRate);
            return m_BaudRate;
        }
        void SetBaudRate(int baudRate) NN_NOEXCEPT
        {
            // Some register access

            NN_LOG("SpecificUartPort: Set baud rate %d\n", baudRate);
            m_BaudRate = baudRate;
        }
        void Send(const char* pBuffer, size_t bufferSize) NN_NOEXCEPT
        {
            // Some register access

            NN_LOG("SpecificUartPort: Sending pBuffer 0x%x size %d\n", pBuffer, bufferSize);
        }

    private:
        int m_SpecificId{ 0 };
        int m_BaudRate{ 0 };
    };
    NN_DDSF_CAST_SAFE_DEFINE(SpecificUartPort, nn::uart_t::driver::UartPort);

    class SpecificUartDriverImpl : public nn::uart_t::driver::IUartDriver
    {
        NN_DDSF_CAST_SAFE_DECL;

    public:
        void Initialize() NN_NOEXCEPT
        {
            // DeviceTree からポート構成を構築する場合もあるが、ここでは静的に設定
            int index = 0;
            for ( auto&& port : m_Ports )
            {
                port.Initialize(index);
                RegisterDevice(&port);
                ++index;
            }
            nn::uart_t::driver::RegisterDriver(*this);
        }

        virtual nn::Result GetBaudRate(nn::ddsf::IDevice& device, int* pOutBaudRate) NN_NOEXCEPT NN_OVERRIDE
        {
            NN_LOG("BusConfigInterfaceImpl %s\n", __FUNCTION__);
            *pOutBaudRate = device.SafeCastTo<SpecificUartPort>().GetBaudRate();
            NN_RESULT_SUCCESS;
        }
        virtual nn::Result SetBaudRate(nn::ddsf::IDevice& device, int baudRate) NN_NOEXCEPT NN_OVERRIDE
        {
            NN_LOG("BusConfigInterfaceImpl %s\n", __FUNCTION__);
            device.SafeCastTo<SpecificUartPort>().SetBaudRate(baudRate);
            NN_RESULT_SUCCESS;
        }
        virtual nn::Result Send(nn::ddsf::IDevice& device, const char* pBuffer, size_t bufferSize) NN_NOEXCEPT NN_OVERRIDE
        {
            NN_LOG("StreamInterfaceImpl %s\n", __FUNCTION__);
            device.SafeCastTo<SpecificUartPort>().Send(pBuffer, bufferSize);
            NN_RESULT_SUCCESS;
        }

        SpecificUartPort& GetPort(int index) NN_NOEXCEPT
        {
            return m_Ports.at(index);
        }

        const SpecificUartPort& GetPort(int index) const NN_NOEXCEPT
        {
            return m_Ports.at(index);
        }

        int GetPortCount() const NN_NOEXCEPT
        {
            return 4;
        }

    private:
        std::array<SpecificUartPort, 4> m_Ports;
    };
    NN_DDSF_CAST_SAFE_DEFINE(SpecificUartDriverImpl, nn::uart_t::driver::IUartDriver);

}}} // nnd::uart_t::specificchip

TEST(RegisterDriver, RegisterDriver)
{
    nn::uart_t::Initialize();
    NN_UTIL_SCOPE_EXIT
    {
        nn::uart_t::Finalize();
    };
    nnd::uart_t::specificchip::SpecificUartDriverImpl impl;
    nnd::uart_t::specificchip::SpecificUartDriverImpl impl2;
    impl.Initialize();
    impl2.Initialize();
}

TEST(Session, OpenClose)
{
    nn::uart_t::Initialize();
    NN_UTIL_SCOPE_EXIT
    {
        nn::uart_t::Finalize();
    };
    nnd::uart_t::specificchip::SpecificUartDriverImpl impl;
    impl.Initialize();

    nn::uart_t::UartSession session;
    nn::uart_t::Open(&session, nn::DeviceCode::GetInvalidCode(), nn::ddsf::AccessMode_ReadAndWrite);

    const char Text[] = "FOOBAR";
    int baudRate = 115200;
    nn::uart_t::SetBaudRate(&session, baudRate);
    nn::uart_t::GetBaudRate(&session, &baudRate);
    nn::uart_t::Send(&session, Text, sizeof(Text));

    nn::uart_t::Close(&session);
}

TEST(IDriver, ForEach)
{
    nn::uart_t::Initialize();
    NN_UTIL_SCOPE_EXIT
    {
        nn::uart_t::Finalize();
    };
    nnd::uart_t::specificchip::SpecificUartDriverImpl impl;
    impl.Initialize();

    NNT_EXPECT_RESULT_SUCCESS(
        impl.ForEachDevice(
            [](nn::ddsf::IDevice*) NN_NOEXCEPT->nn::Result
            {
                NN_RESULT_SUCCESS;
            },
            true)
    );
    // const IDevice* を取るラムダならば const オブジェクトで呼べることをテスト
    NNT_EXPECT_RESULT_SUCCESS(
        const_cast<const nnd::uart_t::specificchip::SpecificUartDriverImpl&>(impl).ForEachDevice(
            [](const nn::ddsf::IDevice*) NN_NOEXCEPT->nn::Result
            {
                NN_RESULT_SUCCESS;
            },
            true)
    );
    EXPECT_EQ(impl.GetPortCount(),
        impl.ForEachDevice(
            [](nn::ddsf::IDevice*) NN_NOEXCEPT -> bool
            {
                return true;
            }
        )
    );
    // const IDevice* を取るラムダならば const オブジェクトで呼べることをテスト
    EXPECT_EQ(impl.GetPortCount(),
        const_cast<const nnd::uart_t::specificchip::SpecificUartDriverImpl&>(impl).ForEachDevice(
            [](const nn::ddsf::IDevice*) NN_NOEXCEPT -> bool
            {
                return true;
            }
        )
    );
}

TEST(IDevice, ForEach)
{
    nn::uart_t::Initialize();
    NN_UTIL_SCOPE_EXIT
    {
        nn::uart_t::Finalize();
    };
    nnd::uart_t::specificchip::SpecificUartDriverImpl impl;
    impl.Initialize();

    nn::uart_t::UartSession session;
    nn::uart_t::Open(&session, nn::DeviceCode::GetInvalidCode(), nn::ddsf::AccessMode_ReadAndWrite);
    NN_UTIL_SCOPE_EXIT
    {
        nn::uart_t::Close(&session);
    };

    NNT_EXPECT_RESULT_SUCCESS(
        impl.GetPort(0).ForEachSession(
        [](nn::ddsf::ISession*) NN_NOEXCEPT -> nn::Result
        {
            NN_RESULT_SUCCESS;
        },
        true)
    );
    // const ISession* を取るラムダならば const オブジェクトで呼べることをテスト
    NNT_EXPECT_RESULT_SUCCESS(
        const_cast<const nnd::uart_t::specificchip::SpecificUartDriverImpl&>(impl).GetPort(0).ForEachSession(
            [](const nn::ddsf::ISession*) NN_NOEXCEPT -> nn::Result
            {
                NN_RESULT_SUCCESS;
            },
        true)
    );
    EXPECT_EQ(1,
        impl.GetPort(0).ForEachSession(
            [](nn::ddsf::ISession*) NN_NOEXCEPT -> bool
            {
                return true;
            }
        )
    );
    // const ISession* を取るラムダならば const オブジェクトで呼べることをテスト
    EXPECT_EQ(1,
        const_cast<const nnd::uart_t::specificchip::SpecificUartDriverImpl&>(impl).GetPort(0).ForEachSession(
            [](const nn::ddsf::ISession*) NN_NOEXCEPT -> bool
            {
                return true;
            }
        )
    );
}
