﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

class TimeCheck
{
public:
    typedef long long usecs_t;

    static inline usecs_t GetSystemTime();

    usecs_t m_Start;
    int m_Max;
    int m_Duration;

    explicit TimeCheck( int maxTime )
    {
        m_Max = maxTime;
        m_Start = GetSystemTime();
    }

    void End()
    {
        usecs_t end = GetSystemTime();
        m_Duration = (int)(end - m_Start);
    }

    void Check( int repeat )
    {
        LocalLog( CSV_SEP "%d", (m_Duration + (repeat / 2)) / repeat );
        if ( m_Max >= 0 )
        {
            //ASSERT_LE( m_Duration, m_Max*repeat );
        }
    }
};

#ifdef ANDROID
    #include <sys/time.h>
inline TimeCheck::usecs_t TimeCheck::GetSystemTime()
{
    struct timespec now = {
        0, 0
    };
    clock_gettime(CLOCK_THREAD_CPUTIME_ID, &now);
    usecs_t t = (usecs_t)(now.tv_sec) * 1000000LL + (usecs_t)(now.tv_nsec / 1000);
    return t;
}

#else
    #if 0
        #ifdef WIN32
            #define NTDDI_WIN7SP1 0 // NOLINT(readability/define)
            #include <windows.h>
inline TimeCheck::usecs_t TimeCheck::GetSystemTime()
{
    static LARGE_INTEGER freq;
    if ( !freq.QuadPart )
    {
        QueryPerformanceFrequency( &freq );
    }

    LARGE_INTEGER now;
    QueryPerformanceCounter( &now );

    usecs_t t = now.QuadPart * 1000000LL / freq.QuadPart;
    return t;
}
        #endif
    #endif
    #include <nn/os/os_Tick.h>
    #include <nn/nn_TimeSpan.h>
inline TimeCheck::usecs_t TimeCheck::GetSystemTime()
{
    usecs_t t = nn::os::GetSystemTick().ToTimeSpan().GetMicroSeconds();
    return t;
}
#endif
