﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nnt/nntest.h>
#include "CavsParser.h"
#include <nn/crypto/crypto_Sha1Generator.h>
#include <nn/crypto/crypto_RsaOaepEncryptor.h>
#include <nn/crypto/crypto_RsaOaepDecryptor.h>

class RsaOaepParser : public CavsParser
{
public:
    RsaOaepParser()
    {
        m_AllowedTokens[N] = AllowedToken("n = ", true);
        m_AllowedTokens[E] = AllowedToken("e = ", true);
        m_AllowedTokens[D] = AllowedToken("d = ", true);
        m_AllowedTokens[Msg] = AllowedToken("Msg = ", true);
        m_AllowedTokens[Seed] = AllowedToken("Seed = ", true);
        m_AllowedTokens[Cipher] = AllowedToken("Cipher = ", true);
        m_AllowedTokens[Comment] = AllowedToken("RSAES-OAEP", false);
        m_TokenForUnitComplete = Msg;
    }

protected:
    enum
    {
        N,
        E,
        D,
        Msg,
        Seed,
        Cipher,
        Comment
    };

    virtual void TestUnit() override
    {
        ASSERT_EQ(1u, m_UnitTokens.count(Msg));
        ASSERT_EQ(1u, m_UnitTokens.count(Seed));
        ASSERT_EQ(1u, m_UnitTokens.count(Cipher));
        ASSERT_EQ(1u, m_UnitTokens.count(N));
        ASSERT_EQ(1u, m_UnitTokens.count(E));
        ASSERT_EQ(1u, m_UnitTokens.count(D));

        const std::string& nStr = m_UnitTokens[N];
        const std::string& eStr = m_UnitTokens[E];
        const std::string& dStr = m_UnitTokens[D];

        const std::string& msgStr = m_UnitTokens[Msg];
        const std::string& seedStr = m_UnitTokens[Seed];
        const std::string& cipherStr = m_UnitTokens[Cipher];

        {
            nn::crypto::RsaOaepEncryptor<2048 / 8, nn::crypto::Sha1Generator> enc;
            enc.Initialize(nStr.c_str(), nStr.size(), eStr.c_str(), eStr.size());
            char cipher[enc.BlockSize];
            std::memset(cipher, 0xBD, sizeof(cipher));
            enc.Encrypt(cipher, sizeof(cipher), msgStr.c_str(), msgStr.size(),
                        seedStr.c_str(), seedStr.size());
            EXPECT_EQ(std::string(cipher, sizeof(cipher)), cipherStr);
        }

        {
            nn::crypto::RsaOaepDecryptor<2048 / 8, nn::crypto::Sha1Generator> dec;
            dec.Initialize(nStr.c_str(), nStr.size(), dStr.c_str(), dStr.size());
            char plain[dec.BlockSize];
            std::memset(plain, 0xBD, sizeof(plain));
            size_t sizeOut = dec.Decrypt(plain, sizeof(plain), cipherStr.c_str(), cipherStr.size());
            EXPECT_EQ(std::string(plain, sizeOut), msgStr);
        }

        m_UnitTokens.erase(Msg);
        m_UnitTokens.erase(Seed);
        m_UnitTokens.erase(Cipher);
    }

};

TEST(RsaTest, OaepFile)
{
    RsaOaepParser tester;
    tester.TestFile("pkcs1-v21/oaep-vectors.txt", 12);
}
