﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/lmem/lmem_ExpHeap.h>

#define LogEvery(fmt,...) //Log(fmt,__VA_ARGS__)

class InstrumentedHeap
{
public:
    InstrumentedHeap()
    {
        m_HeapMemory = 0;
        m_Heap = nullptr;
        m_Capacity = 0;
        Reset();
    }

    void Initialize( size_t sizeInBytes )
    {
        ASSERT_TRUE( !m_Capacity );
        m_Capacity = sizeInBytes;
        //nn::Result err = nn::os::AllocateMemoryBlock(&m_HeapMemory, m_HeapCapacity);
        //ASSERT_TRUE(err.IsSuccess());
        m_HeapMemory = (uintptr_t)std::malloc(m_Capacity);
        ASSERT_TRUE(m_HeapMemory != 0);

        m_Heap = nn::lmem::CreateExpHeap(reinterpret_cast<void*>(m_HeapMemory), m_Capacity,
                                         nn::lmem::CreationOption_NoOption);
        ASSERT_TRUE(m_Heap != nullptr);

        Reset();
    }

    void Finalize()
    {
        if ( m_Heap )
        {
            nn::lmem::DestroyExpHeap(m_Heap);
            m_Heap = nullptr;
        }
        if ( m_HeapMemory )
        {
            std::free((void*)m_HeapMemory);
            //nn::os::FreeMemoryBlock(m_HeapMemory, m_HeapCapacity);
            m_HeapMemory = 0;
        }
    }

    void Reset()
    {
        m_FreeStartMark = m_Heap ? nn::lmem::GetExpHeapTotalFreeSize( m_Heap ) : 0;
        m_FreeLowestMark = m_FreeStartMark;
        m_AllocationsCount = 0;
        m_AllocationsMissed = 0;
    }

    void Check( int maxHeap, int maxCount, int repeat )
    {
        //size_t freeNow = nn::lmem::GetExpHeapTotalFreeSize( m_Heap );
        //ASSERT_EQ( m_FreeStartMark, freeNow );
        LocalLog( CSV_SEP "%d" CSV_SEP "%d", GetMaxUsage(), m_AllocationsCount / repeat );
        if ( maxHeap >= 0 )
        {
            EXPECT_LE( GetMaxUsage(), maxHeap );
        }
        if ( maxCount >= 0 )
        {
            EXPECT_LE( m_AllocationsCount, maxCount * repeat );
        }
    }

    size_t GetMaxUsage()
    {
        return m_FreeStartMark - m_FreeLowestMark;
    }

    void* Allocate( size_t size, int alignment = nn::lmem::DefaultAlignment )
    {
        void* p = nn::lmem::AllocateFromExpHeap( m_Heap, size, alignment );
        size_t freeNow = nn::lmem::GetExpHeapTotalFreeSize( m_Heap );
        if ( freeNow < m_FreeLowestMark )
        {
            m_FreeLowestMark = freeNow;
        }
        ++m_AllocationsCount;
        LogEvery("Alloc_%d %d @%p free %d\n", alignment, size, p, freeNow );
        if ( p == 0 )
        {
            ++m_AllocationsMissed;
            LocalLog("Out of memory for %ld ~%ld\n", size, alignment);
        }
        return p;
    }

    void Deallocate(void* p)
    {
        LogEvery("Free @%p\n", p );
        nn::lmem::FreeToExpHeap( m_Heap, p );
    }

    nn::lmem::HeapHandle m_Heap;
    uintptr_t m_HeapMemory;
    size_t m_Capacity;
    size_t m_FreeStartMark;
    size_t m_FreeLowestMark;
    int m_AllocationsCount;
    int m_AllocationsMissed;
};

extern InstrumentedHeap g_CryptoHeap;

class HeapCheck
{
public:
    int m_Max;
    int m_MaxCount;

    HeapCheck( int maxHeap, int maxCount )
    {
        m_Max = maxHeap;
        m_MaxCount = maxCount;
        g_CryptoHeap.Reset();
    }

    void Check( int repeat )
    {
        g_CryptoHeap.Check( m_Max, m_MaxCount, repeat );
    }
};
