﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include "CavsParser.h"


/**
 * @brief Base class for parsing NIST's test vectors from GCM
 */
class GcmParser : public CavsParser
{
public:
    enum
    {
        COUNT,
        Key,
        IV,
        Plain,
        Auth,
        Cipher,
        Tag,
        Fail,
    };

    GcmParser()
    {
        m_AllowedTokens[COUNT] = AllowedToken("Count = ", false);

        m_AllowedTokens[Auth] = AllowedToken("AAD = ", true);
        m_AllowedTokens[Cipher] = AllowedToken("CT = ", true);
        m_AllowedTokens[Fail] = AllowedToken("FAIL", false);
        m_AllowedTokens[IV] = AllowedToken("IV = ", true);
        m_AllowedTokens[Key] = AllowedToken("Key = ", true);
        m_AllowedTokens[Plain] = AllowedToken("PT = ", true);
        m_AllowedTokens[Tag] = AllowedToken("Tag = ", true);
        m_TokenForUnitComplete = Key;
    }
};


/**
 * @brief Runs NIST's test vectors for GCM encryption
 */
template<typename BlockCipher>
class GcmEncryptParser : public GcmParser
{
protected:
    virtual void TestUnit() override
    {
        EXPECT_EQ(m_UnitTokens.count(Auth), 1);
        EXPECT_EQ(m_UnitTokens.count(Key), 1);
        EXPECT_EQ(m_UnitTokens.count(Plain), 1);
        EXPECT_EQ(m_UnitTokens.count(Cipher), 1);
        EXPECT_EQ(m_UnitTokens.count(Tag), 1);
        EXPECT_EQ(m_UnitTokens.count(IV), 1);

        const std::string& authStr   = m_UnitTokens[Auth];
        const std::string& keyStr    = m_UnitTokens[Key];
        const std::string& plainStr  = m_UnitTokens[Plain];
        const std::string& cipherStr = m_UnitTokens[Cipher];
        const std::string& tagStr    = m_UnitTokens[Tag];
        const std::string& ivStr     = m_UnitTokens[IV];

        {
            std::string resCipher(plainStr.size(), char(0));
            std::string resTag(nn::crypto::GcmEncryptor<BlockCipher>::MacSize, char(0));

            BlockCipher cipher;
            cipher.Initialize(keyStr.c_str(), keyStr.size());

            nn::crypto::GcmEncryptor<BlockCipher> gcm;
            gcm.Initialize(&cipher, ivStr.c_str(), ivStr.size());
            gcm.UpdateAad(authStr.c_str(), authStr.size());
            gcm.Update(const_cast<char*>(resCipher.c_str()), resCipher.size(), plainStr.c_str(), plainStr.size());
            gcm.GetMac(const_cast<char*>(resTag.c_str()), resTag.size());

            // truncate the mac in order to compare by '==' operator
            resTag.resize(tagStr.size());

            EXPECT_TRUE(resCipher == cipherStr);
            EXPECT_TRUE(resTag == tagStr);
        }

        // Same as above, but perform in-place encryption
        {
            std::string msg(plainStr.begin(), plainStr.end());
            std::string resTag(nn::crypto::GcmEncryptor<BlockCipher>::MacSize, char(0));

            BlockCipher cipher;
            cipher.Initialize(keyStr.c_str(), keyStr.size());

            nn::crypto::GcmEncryptor<BlockCipher> gcm;
            gcm.Initialize(&cipher, ivStr.c_str(), ivStr.size());
            gcm.UpdateAad(authStr.c_str(), authStr.size());
            gcm.Update(const_cast<char*>(msg.c_str()), msg.size(), msg.c_str(), msg.size());
            gcm.GetMac(const_cast<char*>(resTag.c_str()), resTag.size());

            // truncate the mac in order to compare by '==' operator
            resTag.resize(tagStr.size());

            EXPECT_TRUE(msg == cipherStr);
            EXPECT_TRUE(resTag == tagStr);
        }

        m_UnitTokens.erase( Key );
    }
};


/**
 * @brief Runs NIST's test vectors for GCM decryption
 */
template<typename BlockCipher>
class GcmDecryptParser : public GcmParser
{
protected:
    virtual void TestUnit() override
    {
        const std::string& authStr   = m_UnitTokens[Auth];
        const std::string& keyStr    = m_UnitTokens[Key];
        const std::string& plainStr  = m_UnitTokens[Plain];
        const std::string& cipherStr = m_UnitTokens[Cipher];
        const std::string& tagStr    = m_UnitTokens[Tag];
        const std::string& ivStr     = m_UnitTokens[IV];
        bool  expectedOk             = !Has( Fail );

        {
            std::string resPlain(cipherStr.size(), char(0));
            char tag[nn::crypto::Aes128GcmDecryptor::MacSize];

            BlockCipher cipher;
            cipher.Initialize(keyStr.c_str(), keyStr.size());

            nn::crypto::GcmDecryptor<BlockCipher> gcm;
            gcm.Initialize(&cipher, ivStr.c_str(), ivStr.size());
            gcm.UpdateAad(authStr.c_str(), authStr.size());
            gcm.Update(const_cast<char*>(resPlain.c_str()), resPlain.size(), cipherStr.c_str(), cipherStr.size());
            gcm.GetMac(tag, sizeof(tag));

            EXPECT_EQ(expectedOk, std::memcmp(tag, tagStr.c_str(), tagStr.size()) == 0);
            if (expectedOk)
            {
                EXPECT_TRUE(resPlain == plainStr);
            }
        }


        // Same as above, but perform in-place decryption
        {
            std::string msg(cipherStr.begin(), cipherStr.end());
            char tag[nn::crypto::Aes128GcmDecryptor::MacSize];

            BlockCipher cipher;
            cipher.Initialize(keyStr.c_str(), keyStr.size());

            nn::crypto::GcmDecryptor<BlockCipher> gcm;
            gcm.Initialize(&cipher, ivStr.c_str(), ivStr.size());
            gcm.UpdateAad(authStr.c_str(), authStr.size());
            gcm.Update(const_cast<char*>(msg.c_str()), msg.size(), msg.c_str(), msg.size());
            gcm.GetMac(tag, sizeof(tag));

            EXPECT_EQ(expectedOk, std::memcmp(tag, tagStr.c_str(), tagStr.size()) == 0);
            if (expectedOk)
            {
                EXPECT_TRUE(msg == plainStr);
            }
        }


        m_UnitTokens.erase(Fail);
        m_UnitTokens.erase(Key);
        m_UnitTokens.erase(Auth);
    }
};
