﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#include <cstring>
#include <iostream>
#include "CavsParser.h"
#include "OctetString.h"

#include <nn/nn_Log.h>
#include <nn/fs.h>
#include <nn/nn_Result.h>

CavsParser::CavsParser()
{
    m_SkipUntilLine = 0;
    m_Completed = 0;
    m_Skipped = false;
    m_FileName = nullptr;
    m_LineNumber = 0;
    m_TokenForUnitComplete = -1;
}

void CavsParser::ClearUnit()
{
    m_StackedTokens.clear();
}

bool CavsParser::ExtractToken(const std::string& line)
{
    for (auto allowed : m_AllowedTokens)
    {
        const AllowedToken& def = allowed.second;
        const std::string& prefix = def.m_Prefix;
        if (std::strstr(line.c_str(), prefix.c_str()) == line.c_str())
        {
            std::string value( line.c_str() + prefix.size(), line.size() - prefix.size() );
            if (def.m_BytesFromHex)
            {
                value = OctetString::Convert(value);
            }
            if (def.m_StackAll)
            {
                m_StackedTokens[ allowed.first ].push_back(value);
            }
            else
            {
                m_UnitTokens[ allowed.first ] = value;
            }
            return true;
        }
    }
    return false;
}

static void CutNextLine( std::string& line, const std::string& text, size_t& start )
{
    size_t next = text.find( '\n', start );
    size_t len = std::string::npos;
    if ( next != std::string::npos )
    {
        len = next - start;
        if ((next > 0) && (text[next - 1] == '\r'))
        {
            --len;
        }
        ++next;
    }
    line = text.substr( start, len );
    start = next;
}

void CavsParser::TestString(const std::string& text, int knownGoodUnits )
{
    m_LineNumber = 0;
    size_t i = 0;
    while ( i != std::string::npos )
    {
        std::string line;
        ++m_LineNumber;
        CutNextLine( line, text, i );
        if ( m_LineNumber < m_SkipUntilLine )
        {
            continue;
        }
        if (ExtractToken(line))
        {
            continue;
        }
        // Blank line or end of string
        if ( line.empty() || (i == std::string::npos))
        {
            if ( Has(m_TokenForUnitComplete) )
            {
                TestUnit();
                if ( !m_Skipped )
                {
                    ++m_Completed;
                }
                m_Skipped = false;
                if (::testing::Test::HasFatalFailure())
                {
                    if ( m_FileName )
                    {
                        ::std::cerr << "Aborting " << m_FileName << "#" << m_LineNumber <<
                            " Unit " << m_Completed << "/" << knownGoodUnits << ::std::endl;
                    }
                    //::testing::internal::GTestLog log( ::testing::internal::GTEST_ERROR, m_FileName ? m_FileName : "(string)", m_LineNumber );
                    //::testing::internal::ColoredPrintf(testing::internal::COLOR_GREEN, "[          ] ");
                    return;
                }
            }
        }
    }
    EXPECT_EQ( knownGoodUnits, m_Completed );
}

void CavsParser::TestFile(const char* filename, int expectCompletedUnits )
{
    std::string path;
    std::string text;
    const int buflen = 2048;
    char buf[buflen];
    text.clear();

    //#define NNT_CAVS_SIGLO_ROOT set by nact script or vcxporj to the root path of Siglo.
    path = std::string("rom:/") + filename;

    nn::fs::FileHandle file;
    nn::Result result;
    result = nn::fs::OpenFile(&file, path.c_str(), nn::fs::OpenMode_Read);
    if ( result.IsFailure() )
    {
        NN_LOG("error %d %d",result.GetDescription(), result.GetModule());
        //char pwd[260]; _getcwd(pwd,sizeof(pwd));
        GTEST_FATAL_FAILURE_(filename);
    }

    int64_t offset = 0;
    for (;; )
    {
        size_t readSize = 0;
        result = nn::fs::ReadFile(&readSize, file, offset, buf, buflen - 1, nn::fs::ReadOption());
        ASSERT_TRUE(readSize >= 0);
        if (readSize == 0)
        {
            break;
        }
        offset += readSize;
        buf[readSize] = 0;
        text += buf;
    }

    CloseFile(file);

    m_FileName = filename;
    TestString(text, expectCompletedUnits );
}
