﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstdlib> // std::[malloc,free]
#include <limits>  // std::numeric_limits::[min,max]
#include <random>  // std::mt19937

#include <nnt.h>
#include <nn/fs.h>
#include <nn/os.h>
#include <nn/nn_Assert.h>

#include <nn/codec.h>
#include <nn/codec/codec_OpusEncoder.h>
#include <nn/codec/detail/codec_OpusPacketInternal.h>

#include <nnt/codecUtil/testCodec_Util.h>
#include <nnt/codecUtil/testCodec_FixtureBase.h>

#include <nn/nn_Log.h>

namespace {

const char* SimpleTestParameterListStrings[] =
{
    "SampleRate=48000, ChannelCount=1",
    "SampleRate=48000, ChannelCount=2",
    "SampleRate=24000, ChannelCount=1",
    "SampleRate=24000, ChannelCount=2",
    "SampleRate=16000, ChannelCount=1",
    "SampleRate=16000, ChannelCount=2",
    "SampleRate=12000, ChannelCount=1",
    "SampleRate=12000, ChannelCount=2",
    "SampleRate=8000,  ChannelCount=1",
    "SampleRate=8000,  ChannelCount=2",
};

// 以下のテストパラメータでは、列挙型の値を決め打ちしているので注意。
NN_STATIC_ASSERT(nn::codec::OpusCodingMode_Celt == 2);
NN_STATIC_ASSERT(nn::codec::OpusCodingMode_Silk == 3);

const char* ComplexTestParameterListStrings[] =
{
    "SampleRate=48000, ChannelCount=1, Frame=2500,  BitRate=6000,   CodingMode=2",
    "SampleRate=48000, ChannelCount=2, Frame=2500,  BitRate=6000,   CodingMode=2",
    "SampleRate=24000, ChannelCount=1, Frame=2500,  BitRate=6000,   CodingMode=2",
    "SampleRate=24000, ChannelCount=2, Frame=2500,  BitRate=6000,   CodingMode=2",
    "SampleRate=16000, ChannelCount=1, Frame=2500,  BitRate=6000,   CodingMode=2",
    "SampleRate=16000, ChannelCount=2, Frame=2500,  BitRate=6000,   CodingMode=2",
    "SampleRate=12000, ChannelCount=1, Frame=2500,  BitRate=6000,   CodingMode=2",
    "SampleRate=12000, ChannelCount=2, Frame=2500,  BitRate=6000,   CodingMode=2",
    "SampleRate=8000,  ChannelCount=1, Frame=2500,  BitRate=6000,   CodingMode=2",
    "SampleRate=8000,  ChannelCount=2, Frame=2500,  BitRate=6000,   CodingMode=2",
    "SampleRate=48000, ChannelCount=1, Frame=2500,  BitRate=300000, CodingMode=2",
    "SampleRate=48000, ChannelCount=2, Frame=2500,  BitRate=510000, CodingMode=2",
    "SampleRate=24000, ChannelCount=1, Frame=2500,  BitRate=300000, CodingMode=2",
    "SampleRate=24000, ChannelCount=2, Frame=2500,  BitRate=510000, CodingMode=2",
    "SampleRate=16000, ChannelCount=1, Frame=2500,  BitRate=300000, CodingMode=2",
    "SampleRate=16000, ChannelCount=2, Frame=2500,  BitRate=510000, CodingMode=2",
    "SampleRate=12000, ChannelCount=1, Frame=2500,  BitRate=300000, CodingMode=2",
    "SampleRate=12000, ChannelCount=2, Frame=2500,  BitRate=510000, CodingMode=2",
    "SampleRate=8000,  ChannelCount=1, Frame=2500,  BitRate=300000, CodingMode=2",
    "SampleRate=8000,  ChannelCount=2, Frame=2500,  BitRate=510000, CodingMode=2",
    "SampleRate=48000, ChannelCount=1, Frame=5000,  BitRate=6000,   CodingMode=2",
    "SampleRate=48000, ChannelCount=2, Frame=5000,  BitRate=6000,   CodingMode=2",
    "SampleRate=24000, ChannelCount=1, Frame=5000,  BitRate=6000,   CodingMode=2",
    "SampleRate=24000, ChannelCount=2, Frame=5000,  BitRate=6000,   CodingMode=2",
    "SampleRate=16000, ChannelCount=1, Frame=5000,  BitRate=6000,   CodingMode=2",
    "SampleRate=16000, ChannelCount=2, Frame=5000,  BitRate=6000,   CodingMode=2",
    "SampleRate=12000, ChannelCount=1, Frame=5000,  BitRate=6000,   CodingMode=2",
    "SampleRate=12000, ChannelCount=2, Frame=5000,  BitRate=6000,   CodingMode=2",
    "SampleRate=8000,  ChannelCount=1, Frame=5000,  BitRate=6000,   CodingMode=2",
    "SampleRate=8000,  ChannelCount=2, Frame=5000,  BitRate=6000,   CodingMode=2",
    "SampleRate=48000, ChannelCount=1, Frame=5000,  BitRate=300000, CodingMode=2",
    "SampleRate=48000, ChannelCount=2, Frame=5000,  BitRate=510000, CodingMode=2",
    "SampleRate=24000, ChannelCount=1, Frame=5000,  BitRate=300000, CodingMode=2",
    "SampleRate=24000, ChannelCount=2, Frame=5000,  BitRate=510000, CodingMode=2",
    "SampleRate=16000, ChannelCount=1, Frame=5000,  BitRate=300000, CodingMode=2",
    "SampleRate=16000, ChannelCount=2, Frame=5000,  BitRate=510000, CodingMode=2",
    "SampleRate=12000, ChannelCount=1, Frame=5000,  BitRate=300000, CodingMode=2",
    "SampleRate=12000, ChannelCount=2, Frame=5000,  BitRate=510000, CodingMode=2",
    "SampleRate=8000,  ChannelCount=1, Frame=5000,  BitRate=300000, CodingMode=2",
    "SampleRate=8000,  ChannelCount=2, Frame=5000,  BitRate=510000, CodingMode=2",
    "SampleRate=48000, ChannelCount=1, Frame=10000, BitRate=6000,   CodingMode=2",
    "SampleRate=48000, ChannelCount=2, Frame=10000, BitRate=6000,   CodingMode=2",
    "SampleRate=24000, ChannelCount=1, Frame=10000, BitRate=6000,   CodingMode=2",
    "SampleRate=24000, ChannelCount=2, Frame=10000, BitRate=6000,   CodingMode=2",
    "SampleRate=16000, ChannelCount=1, Frame=10000, BitRate=6000,   CodingMode=2",
    "SampleRate=16000, ChannelCount=2, Frame=10000, BitRate=6000,   CodingMode=2",
    "SampleRate=12000, ChannelCount=1, Frame=10000, BitRate=6000,   CodingMode=2",
    "SampleRate=12000, ChannelCount=2, Frame=10000, BitRate=6000,   CodingMode=2",
    "SampleRate=8000,  ChannelCount=1, Frame=10000, BitRate=6000,   CodingMode=2",
    "SampleRate=8000,  ChannelCount=2, Frame=10000, BitRate=6000,   CodingMode=2",
    "SampleRate=48000, ChannelCount=1, Frame=10000, BitRate=300000, CodingMode=2",
    "SampleRate=48000, ChannelCount=2, Frame=10000, BitRate=510000, CodingMode=2",
    "SampleRate=24000, ChannelCount=1, Frame=10000, BitRate=300000, CodingMode=2",
    "SampleRate=24000, ChannelCount=2, Frame=10000, BitRate=510000, CodingMode=2",
    "SampleRate=16000, ChannelCount=1, Frame=10000, BitRate=300000, CodingMode=2",
    "SampleRate=16000, ChannelCount=2, Frame=10000, BitRate=510000, CodingMode=2",
    "SampleRate=12000, ChannelCount=1, Frame=10000, BitRate=300000, CodingMode=2",
    "SampleRate=12000, ChannelCount=2, Frame=10000, BitRate=510000, CodingMode=2",
    "SampleRate=8000,  ChannelCount=1, Frame=10000, BitRate=300000, CodingMode=2",
    "SampleRate=8000,  ChannelCount=2, Frame=10000, BitRate=510000, CodingMode=2",
    "SampleRate=48000, ChannelCount=1, Frame=20000, BitRate=6000,   CodingMode=2",
    "SampleRate=48000, ChannelCount=2, Frame=20000, BitRate=6000,   CodingMode=2",
    "SampleRate=24000, ChannelCount=1, Frame=20000, BitRate=6000,   CodingMode=2",
    "SampleRate=24000, ChannelCount=2, Frame=20000, BitRate=6000,   CodingMode=2",
    "SampleRate=16000, ChannelCount=1, Frame=20000, BitRate=6000,   CodingMode=2",
    "SampleRate=16000, ChannelCount=2, Frame=20000, BitRate=6000,   CodingMode=2",
    "SampleRate=12000, ChannelCount=1, Frame=20000, BitRate=6000,   CodingMode=2",
    "SampleRate=12000, ChannelCount=2, Frame=20000, BitRate=6000,   CodingMode=2",
    "SampleRate=8000,  ChannelCount=1, Frame=20000, BitRate=6000,   CodingMode=2",
    "SampleRate=8000,  ChannelCount=2, Frame=20000, BitRate=6000,   CodingMode=2",
    "SampleRate=48000, ChannelCount=1, Frame=20000, BitRate=300000, CodingMode=2",
    "SampleRate=48000, ChannelCount=2, Frame=20000, BitRate=510000, CodingMode=2",
    "SampleRate=24000, ChannelCount=1, Frame=20000, BitRate=300000, CodingMode=2",
    "SampleRate=24000, ChannelCount=2, Frame=20000, BitRate=510000, CodingMode=2",
    "SampleRate=16000, ChannelCount=1, Frame=20000, BitRate=300000, CodingMode=2",
    "SampleRate=16000, ChannelCount=2, Frame=20000, BitRate=510000, CodingMode=2",
    "SampleRate=12000, ChannelCount=1, Frame=20000, BitRate=300000, CodingMode=2",
    "SampleRate=12000, ChannelCount=2, Frame=20000, BitRate=510000, CodingMode=2",
    "SampleRate=8000,  ChannelCount=1, Frame=20000, BitRate=300000, CodingMode=2",
    "SampleRate=8000,  ChannelCount=2, Frame=20000, BitRate=510000, CodingMode=2",
    "SampleRate=48000, ChannelCount=1, Frame=2500,  BitRate=6000,   CodingMode=3",
    "SampleRate=48000, ChannelCount=2, Frame=2500,  BitRate=6000,   CodingMode=3",
    "SampleRate=24000, ChannelCount=1, Frame=2500,  BitRate=6000,   CodingMode=3",
    "SampleRate=24000, ChannelCount=2, Frame=2500,  BitRate=6000,   CodingMode=3",
    "SampleRate=16000, ChannelCount=1, Frame=2500,  BitRate=6000,   CodingMode=3",
    "SampleRate=16000, ChannelCount=2, Frame=2500,  BitRate=6000,   CodingMode=3",
    "SampleRate=12000, ChannelCount=1, Frame=2500,  BitRate=6000,   CodingMode=3",
    "SampleRate=12000, ChannelCount=2, Frame=2500,  BitRate=6000,   CodingMode=3",
    "SampleRate=8000,  ChannelCount=1, Frame=2500,  BitRate=6000,   CodingMode=3",
    "SampleRate=8000,  ChannelCount=2, Frame=2500,  BitRate=6000,   CodingMode=3",
    "SampleRate=48000, ChannelCount=1, Frame=2500,  BitRate=300000, CodingMode=3",
    "SampleRate=48000, ChannelCount=2, Frame=2500,  BitRate=510000, CodingMode=3",
    "SampleRate=24000, ChannelCount=1, Frame=2500,  BitRate=300000, CodingMode=3",
    "SampleRate=24000, ChannelCount=2, Frame=2500,  BitRate=510000, CodingMode=3",
    "SampleRate=16000, ChannelCount=1, Frame=2500,  BitRate=300000, CodingMode=3",
    "SampleRate=16000, ChannelCount=2, Frame=2500,  BitRate=510000, CodingMode=3",
    "SampleRate=12000, ChannelCount=1, Frame=2500,  BitRate=300000, CodingMode=3",
    "SampleRate=12000, ChannelCount=2, Frame=2500,  BitRate=510000, CodingMode=3",
    "SampleRate=8000,  ChannelCount=1, Frame=2500,  BitRate=300000, CodingMode=3",
    "SampleRate=8000,  ChannelCount=2, Frame=2500,  BitRate=510000, CodingMode=3",
    "SampleRate=48000, ChannelCount=1, Frame=5000,  BitRate=6000,   CodingMode=3",
    "SampleRate=48000, ChannelCount=2, Frame=5000,  BitRate=6000,   CodingMode=3",
    "SampleRate=24000, ChannelCount=1, Frame=5000,  BitRate=6000,   CodingMode=3",
    "SampleRate=24000, ChannelCount=2, Frame=5000,  BitRate=6000,   CodingMode=3",
    "SampleRate=16000, ChannelCount=1, Frame=5000,  BitRate=6000,   CodingMode=3",
    "SampleRate=16000, ChannelCount=2, Frame=5000,  BitRate=6000,   CodingMode=3",
    "SampleRate=12000, ChannelCount=1, Frame=5000,  BitRate=6000,   CodingMode=3",
    "SampleRate=12000, ChannelCount=2, Frame=5000,  BitRate=6000,   CodingMode=3",
    "SampleRate=8000,  ChannelCount=1, Frame=5000,  BitRate=6000,   CodingMode=3",
    "SampleRate=8000,  ChannelCount=2, Frame=5000,  BitRate=6000,   CodingMode=3",
    "SampleRate=48000, ChannelCount=1, Frame=5000,  BitRate=300000, CodingMode=3",
    "SampleRate=48000, ChannelCount=2, Frame=5000,  BitRate=510000, CodingMode=3",
    "SampleRate=24000, ChannelCount=1, Frame=5000,  BitRate=300000, CodingMode=3",
    "SampleRate=24000, ChannelCount=2, Frame=5000,  BitRate=510000, CodingMode=3",
    "SampleRate=16000, ChannelCount=1, Frame=5000,  BitRate=300000, CodingMode=3",
    "SampleRate=16000, ChannelCount=2, Frame=5000,  BitRate=510000, CodingMode=3",
    "SampleRate=12000, ChannelCount=1, Frame=5000,  BitRate=300000, CodingMode=3",
    "SampleRate=12000, ChannelCount=2, Frame=5000,  BitRate=510000, CodingMode=3",
    "SampleRate=8000,  ChannelCount=1, Frame=5000,  BitRate=300000, CodingMode=3",
    "SampleRate=8000,  ChannelCount=2, Frame=5000,  BitRate=510000, CodingMode=3",
    "SampleRate=48000, ChannelCount=1, Frame=10000, BitRate=6000,   CodingMode=3",
    "SampleRate=48000, ChannelCount=2, Frame=10000, BitRate=6000,   CodingMode=3",
    "SampleRate=24000, ChannelCount=1, Frame=10000, BitRate=6000,   CodingMode=3",
    "SampleRate=24000, ChannelCount=2, Frame=10000, BitRate=6000,   CodingMode=3",
    "SampleRate=16000, ChannelCount=1, Frame=10000, BitRate=6000,   CodingMode=3",
    "SampleRate=16000, ChannelCount=2, Frame=10000, BitRate=6000,   CodingMode=3",
    "SampleRate=12000, ChannelCount=1, Frame=10000, BitRate=6000,   CodingMode=3",
    "SampleRate=12000, ChannelCount=2, Frame=10000, BitRate=6000,   CodingMode=3",
    "SampleRate=8000,  ChannelCount=1, Frame=10000, BitRate=6000,   CodingMode=3",
    "SampleRate=8000,  ChannelCount=2, Frame=10000, BitRate=6000,   CodingMode=3",
    "SampleRate=48000, ChannelCount=1, Frame=10000, BitRate=300000, CodingMode=3",
    "SampleRate=48000, ChannelCount=2, Frame=10000, BitRate=510000, CodingMode=3",
    "SampleRate=24000, ChannelCount=1, Frame=10000, BitRate=300000, CodingMode=3",
    "SampleRate=24000, ChannelCount=2, Frame=10000, BitRate=510000, CodingMode=3",
    "SampleRate=16000, ChannelCount=1, Frame=10000, BitRate=300000, CodingMode=3",
    "SampleRate=16000, ChannelCount=2, Frame=10000, BitRate=510000, CodingMode=3",
    "SampleRate=12000, ChannelCount=1, Frame=10000, BitRate=300000, CodingMode=3",
    "SampleRate=12000, ChannelCount=2, Frame=10000, BitRate=510000, CodingMode=3",
    "SampleRate=8000,  ChannelCount=1, Frame=10000, BitRate=300000, CodingMode=3",
    "SampleRate=8000,  ChannelCount=2, Frame=10000, BitRate=510000, CodingMode=3",
    "SampleRate=48000, ChannelCount=1, Frame=20000, BitRate=6000,   CodingMode=3",
    "SampleRate=48000, ChannelCount=2, Frame=20000, BitRate=6000,   CodingMode=3",
    "SampleRate=24000, ChannelCount=1, Frame=20000, BitRate=6000,   CodingMode=3",
    "SampleRate=24000, ChannelCount=2, Frame=20000, BitRate=6000,   CodingMode=3",
    "SampleRate=16000, ChannelCount=1, Frame=20000, BitRate=6000,   CodingMode=3",
    "SampleRate=16000, ChannelCount=2, Frame=20000, BitRate=6000,   CodingMode=3",
    "SampleRate=12000, ChannelCount=1, Frame=20000, BitRate=6000,   CodingMode=3",
    "SampleRate=12000, ChannelCount=2, Frame=20000, BitRate=6000,   CodingMode=3",
    "SampleRate=8000,  ChannelCount=1, Frame=20000, BitRate=6000,   CodingMode=3",
    "SampleRate=8000,  ChannelCount=2, Frame=20000, BitRate=6000,   CodingMode=3",
    "SampleRate=48000, ChannelCount=1, Frame=20000, BitRate=300000, CodingMode=3",
    "SampleRate=48000, ChannelCount=2, Frame=20000, BitRate=510000, CodingMode=3",
    "SampleRate=24000, ChannelCount=1, Frame=20000, BitRate=300000, CodingMode=3",
    "SampleRate=24000, ChannelCount=2, Frame=20000, BitRate=510000, CodingMode=3",
    "SampleRate=16000, ChannelCount=1, Frame=20000, BitRate=300000, CodingMode=3",
    "SampleRate=16000, ChannelCount=2, Frame=20000, BitRate=510000, CodingMode=3",
    "SampleRate=12000, ChannelCount=1, Frame=20000, BitRate=300000, CodingMode=3",
    "SampleRate=12000, ChannelCount=2, Frame=20000, BitRate=510000, CodingMode=3",
    "SampleRate=8000,  ChannelCount=1, Frame=20000, BitRate=300000, CodingMode=3",
    "SampleRate=8000,  ChannelCount=2, Frame=20000, BitRate=510000, CodingMode=3",
};

#if !defined(NN_SDK_BUILD_RELEASE)
const char* ComplexDeathTestParameterListStrings[] =
{
    "SampleRate=48000, ChannelCount=1, Frame=2500,  CodingMode=2",
    "SampleRate=48000, ChannelCount=2, Frame=2500,  CodingMode=2",
    "SampleRate=24000, ChannelCount=1, Frame=2500,  CodingMode=2",
    "SampleRate=24000, ChannelCount=2, Frame=2500,  CodingMode=2",
    "SampleRate=16000, ChannelCount=1, Frame=2500,  CodingMode=2",
    "SampleRate=16000, ChannelCount=2, Frame=2500,  CodingMode=2",
    "SampleRate=12000, ChannelCount=1, Frame=2500,  CodingMode=2",
    "SampleRate=12000, ChannelCount=2, Frame=2500,  CodingMode=2",
    "SampleRate=8000,  ChannelCount=1, Frame=2500,  CodingMode=2",
    "SampleRate=8000,  ChannelCount=2, Frame=2500,  CodingMode=2",
    "SampleRate=48000, ChannelCount=1, Frame=5000,  CodingMode=2",
    "SampleRate=48000, ChannelCount=2, Frame=5000,  CodingMode=2",
    "SampleRate=24000, ChannelCount=1, Frame=5000,  CodingMode=2",
    "SampleRate=24000, ChannelCount=2, Frame=5000,  CodingMode=2",
    "SampleRate=16000, ChannelCount=1, Frame=5000,  CodingMode=2",
    "SampleRate=16000, ChannelCount=2, Frame=5000,  CodingMode=2",
    "SampleRate=12000, ChannelCount=1, Frame=5000,  CodingMode=2",
    "SampleRate=12000, ChannelCount=2, Frame=5000,  CodingMode=2",
    "SampleRate=8000,  ChannelCount=1, Frame=5000,  CodingMode=2",
    "SampleRate=8000,  ChannelCount=2, Frame=5000,  CodingMode=2",
    "SampleRate=48000, ChannelCount=1, Frame=10000, CodingMode=2",
    "SampleRate=48000, ChannelCount=2, Frame=10000, CodingMode=2",
    "SampleRate=24000, ChannelCount=1, Frame=10000, CodingMode=2",
    "SampleRate=24000, ChannelCount=2, Frame=10000, CodingMode=2",
    "SampleRate=16000, ChannelCount=1, Frame=10000, CodingMode=2",
    "SampleRate=16000, ChannelCount=2, Frame=10000, CodingMode=2",
    "SampleRate=12000, ChannelCount=1, Frame=10000, CodingMode=2",
    "SampleRate=12000, ChannelCount=2, Frame=10000, CodingMode=2",
    "SampleRate=8000,  ChannelCount=1, Frame=10000, CodingMode=2",
    "SampleRate=8000,  ChannelCount=2, Frame=10000, CodingMode=2",
    "SampleRate=48000, ChannelCount=1, Frame=20000, CodingMode=2",
    "SampleRate=48000, ChannelCount=2, Frame=20000, CodingMode=2",
    "SampleRate=24000, ChannelCount=1, Frame=20000, CodingMode=2",
    "SampleRate=24000, ChannelCount=2, Frame=20000, CodingMode=2",
    "SampleRate=16000, ChannelCount=1, Frame=20000, CodingMode=2",
    "SampleRate=16000, ChannelCount=2, Frame=20000, CodingMode=2",
    "SampleRate=12000, ChannelCount=1, Frame=20000, CodingMode=2",
    "SampleRate=12000, ChannelCount=2, Frame=20000, CodingMode=2",
    "SampleRate=8000,  ChannelCount=1, Frame=20000, CodingMode=2",
    "SampleRate=8000,  ChannelCount=2, Frame=20000, CodingMode=2",
    "SampleRate=48000, ChannelCount=1, Frame=2500,  CodingMode=3",
    "SampleRate=48000, ChannelCount=2, Frame=2500,  CodingMode=3",
    "SampleRate=24000, ChannelCount=1, Frame=2500,  CodingMode=3",
    "SampleRate=24000, ChannelCount=2, Frame=2500,  CodingMode=3",
    "SampleRate=16000, ChannelCount=1, Frame=2500,  CodingMode=3",
    "SampleRate=16000, ChannelCount=2, Frame=2500,  CodingMode=3",
    "SampleRate=12000, ChannelCount=1, Frame=2500,  CodingMode=3",
    "SampleRate=12000, ChannelCount=2, Frame=2500,  CodingMode=3",
    "SampleRate=8000,  ChannelCount=1, Frame=2500,  CodingMode=3",
    "SampleRate=8000,  ChannelCount=2, Frame=2500,  CodingMode=3",
    "SampleRate=48000, ChannelCount=1, Frame=5000,  CodingMode=3",
    "SampleRate=48000, ChannelCount=2, Frame=5000,  CodingMode=3",
    "SampleRate=24000, ChannelCount=1, Frame=5000,  CodingMode=3",
    "SampleRate=24000, ChannelCount=2, Frame=5000,  CodingMode=3",
    "SampleRate=16000, ChannelCount=1, Frame=5000,  CodingMode=3",
    "SampleRate=16000, ChannelCount=2, Frame=5000,  CodingMode=3",
    "SampleRate=12000, ChannelCount=1, Frame=5000,  CodingMode=3",
    "SampleRate=12000, ChannelCount=2, Frame=5000,  CodingMode=3",
    "SampleRate=8000,  ChannelCount=1, Frame=5000,  CodingMode=3",
    "SampleRate=8000,  ChannelCount=2, Frame=5000,  CodingMode=3",
    "SampleRate=48000, ChannelCount=1, Frame=10000, CodingMode=3",
    "SampleRate=48000, ChannelCount=2, Frame=10000, CodingMode=3",
    "SampleRate=24000, ChannelCount=1, Frame=10000, CodingMode=3",
    "SampleRate=24000, ChannelCount=2, Frame=10000, CodingMode=3",
    "SampleRate=16000, ChannelCount=1, Frame=10000, CodingMode=3",
    "SampleRate=16000, ChannelCount=2, Frame=10000, CodingMode=3",
    "SampleRate=12000, ChannelCount=1, Frame=10000, CodingMode=3",
    "SampleRate=12000, ChannelCount=2, Frame=10000, CodingMode=3",
    "SampleRate=8000,  ChannelCount=1, Frame=10000, CodingMode=3",
    "SampleRate=8000,  ChannelCount=2, Frame=10000, CodingMode=3",
    "SampleRate=48000, ChannelCount=1, Frame=20000, CodingMode=3",
    "SampleRate=48000, ChannelCount=2, Frame=20000, CodingMode=3",
    "SampleRate=24000, ChannelCount=1, Frame=20000, CodingMode=3",
    "SampleRate=24000, ChannelCount=2, Frame=20000, CodingMode=3",
    "SampleRate=16000, ChannelCount=1, Frame=20000, CodingMode=3",
    "SampleRate=16000, ChannelCount=2, Frame=20000, CodingMode=3",
    "SampleRate=12000, ChannelCount=1, Frame=20000, CodingMode=3",
    "SampleRate=12000, ChannelCount=2, Frame=20000, CodingMode=3",
    "SampleRate=8000,  ChannelCount=1, Frame=20000, CodingMode=3",
    "SampleRate=8000,  ChannelCount=2, Frame=20000, CodingMode=3",
};
#endif // !defined(NN_SDK_BUILD_RELEASE)

class OpusEncoderApiTestBase
    : public nnt::codec::OpusEncoderBase<nn::codec::OpusEncoder>
    , public ::testing::WithParamInterface<const char*>
{
protected:

    OpusEncoderApiTestBase() NN_NOEXCEPT
        : sampleRate(0)
        , channelCount(0)
    {}

    nnt::codec::TestParameterSet GetTestParameterSet(const char* valueString) NN_NOEXCEPT
    {
        const int _sampleRate = nnt::codec::util::GetIntegerValueLabeledWith(valueString, "SampleRate");
        const int _channelCount = nnt::codec::util::GetIntegerValueLabeledWith(valueString, "ChannelCount");
        const int _frame = nnt::codec::util::GetIntegerValueLabeledWith(valueString, "Frame");
        const int _bitRate = nnt::codec::util::GetIntegerValueLabeledWith(valueString, "BitRate");
        const int _codingMode = nnt::codec::util::GetIntegerValueLabeledWith(valueString, "CodingMode");
        nnt::codec::TestParameterSet set =
        {
            _sampleRate,
            _channelCount,
            _frame,
            _bitRate,
            static_cast<nn::codec::OpusCodingMode>(_codingMode)
        };
        return set;
    }

    virtual void Initialize(const nnt::codec::TestParameterSet& parameterSet) NN_NOEXCEPT
    {
        sampleRate = parameterSet.sampleRate;
        channelCount = parameterSet.channelCount;
        OpusEncoderBase::Initialize(parameterSet);
        ASSERT_EQ(
            nn::codec::OpusResult_Success,
            encoder.Initialize(sampleRate, channelCount, encoderWorkBufferAddress, encoderWorkBufferSize)
        );
    }

    virtual void Finalize() NN_NOEXCEPT
    {
        OpusEncoderBase::Finalize();
    }

    virtual void SetUp() NN_NOEXCEPT
    {
        const auto parameterSet = GetTestParameterSet(GetParam());
        Initialize(parameterSet);
    }
    virtual void TearDown() NN_NOEXCEPT
    {
        Finalize();
    }
    int sampleRate;
    int channelCount;
};

class OpusEncoderApiSimpleTest
    : public OpusEncoderApiTestBase
    , public ::testing::Test
{
protected:
    virtual void SetUp() NN_NOEXCEPT
    {
        OpusEncoderApiTestBase::SetUp();
    }
    virtual void TeaDown() NN_NOEXCEPT
    {
        OpusEncoderApiTestBase::TearDown();
    }
};

class OpusEncoderApiComplexTest
    : public OpusEncoderApiTestBase
    , public nnt::codec::PcmDataBufferBase
    , public nnt::codec::EncodedDataBufferBase
    , public ::testing::Test
{
protected:
    virtual void Initialize(const nnt::codec::TestParameterSet& parameterSet) NN_NOEXCEPT
    {
        OpusEncoderApiTestBase::Initialize(parameterSet);

        frame = parameterSet.frame;
        bitRate = parameterSet.bitRate;
        codingMode = parameterSet.codingMode;
        NN_ASSERT(
            codingMode == nn::codec::OpusCodingMode_Silk
            // || codingMode == nn::codec::OpusCodingMode_Auto
            || codingMode == nn::codec::OpusCodingMode_Hybrid // Should be removed
            || codingMode == nn::codec::OpusCodingMode_Celt
        );

        PcmDataBufferBase::Initialize(parameterSet);
        EncodedDataBufferBase::Initialize(parameterSet);

        std::mt19937 mt(static_cast<int>(nn::os::GetSystemTick().GetInt64Value()));
        const auto a = std::numeric_limits<uint16_t>::max();
        const auto b = std::numeric_limits<int16_t>::min();
        for (auto k = 0; k < pcmDataBufferSampleCount * channelCount; ++k)
        {
            pcmDataBuffer[k] = (mt() % a) + b;
        }
    }
    virtual void Finalize() NN_NOEXCEPT
    {
        PcmDataBufferBase::Finalize();
        OpusEncoderApiTestBase::Finalize();
    }
    virtual void SetUp() NN_NOEXCEPT
    {
        const auto parameterSet = GetTestParameterSet(GetParam());
        Initialize(parameterSet);
    }
    virtual void TearDown() NN_NOEXCEPT
    {
        Finalize();
    }
    int frame;
    nn::codec::OpusCodingMode codingMode;
    int bitRate;
};

}

////////////////////////////////////////////////////////////////////////
// Simple
////////////////////////////////////////////////////////////////////////
/**
 * @brief       OpusEncoder::SetBitRate() の正常系テストです。
 */
TEST_P(OpusEncoderApiSimpleTest, SetBitRate)
{
    SCOPED_TRACE(GetParam());

    const int bitRateMaximum = nn::codec::GetOpusBitRateMax(encoder.GetChannelCount());
    const int bitRateMinimum = nn::codec::GetOpusBitRateMin(encoder.GetChannelCount());

    // 境界を確認しておく
    const int bitRates[] = {
        bitRateMinimum ,
        bitRateMinimum + 1,
        bitRateMaximum - 1,
        bitRateMaximum
    };
    for (auto bitRate : bitRates)
    {
        encoder.SetBitRate(bitRate);
        EXPECT_EQ(bitRate, encoder.GetBitRate());
    }
}

TEST_P(OpusEncoderApiSimpleTest, WaveformType)
{
    const nn::codec::OpusWaveformType waveformTypes[] =
    {
        nn::codec::OpusWaveformType_Voice,
        nn::codec::OpusWaveformType_Music
    };
    for (auto waveformType : waveformTypes)
    {
        encoder.SetWaveformType(waveformType);
        EXPECT_EQ(waveformType, encoder.GetWaveformType());
    }
}

TEST_P(OpusEncoderApiSimpleTest, GetPreSkipSampleCount)
{
    const int restrictedLowDelay = sampleRate / 400;
    const int nonRestrictedLowDelay = restrictedLowDelay + sampleRate / 250;

    // Check right after Initialize()
    EXPECT_EQ(restrictedLowDelay, encoder.GetPreSkipSampleCount());

    // Check after calling BingCodingMode()
    encoder.BindCodingMode(nn::codec::OpusCodingMode_Celt); // ToDo
    EXPECT_EQ(restrictedLowDelay, encoder.GetPreSkipSampleCount());
    encoder.BindCodingMode(nn::codec::OpusCodingMode_Silk);
    EXPECT_EQ(nonRestrictedLowDelay, encoder.GetPreSkipSampleCount());
}

INSTANTIATE_TEST_CASE_P(
    RoundRobin,
    OpusEncoderApiSimpleTest,
    ::testing::ValuesIn(SimpleTestParameterListStrings)
);

////////////////////////////////////////////////////////////////////////
// Complex
////////////////////////////////////////////////////////////////////////
/**
 * @brief       OpusEncoder::EncodeInterleaved、OpusEncoder::BindBitRate()、GetBitRate() の正常系テストです。
 */
TEST_P(OpusEncoderApiComplexTest, EncodeAndCodingMode)
{
    SCOPED_TRACE(GetParam());
    encoder.BindCodingMode(codingMode);
    EXPECT_EQ(codingMode, encoder.GetCodingMode());

    // Silk は、2.5 ms、5.0 ms は非対応。
    if (codingMode == nn::codec::OpusCodingMode_Silk && frame < 10000)
    {
        ASSERT_EQ(
            nn::codec::OpusResult_UnsupportedSampleCount,
            encoder.EncodeInterleaved(
                &encodedDataSize,
                encodedDataBuffer,
                encodedDataBufferSize,
                pcmDataBuffer,
                pcmDataBufferSampleCount)
        );
    }
    else
    {
        ASSERT_EQ(
            nn::codec::OpusResult_Success,
            encoder.EncodeInterleaved(
                &encodedDataSize,
                encodedDataBuffer,
                encodedDataBufferSize,
                pcmDataBuffer,
                pcmDataBufferSampleCount)
        );
        ASSERT_GT(encodedDataSize, 0u);
        nn::codec::OpusCodingMode value;
        ASSERT_EQ(
            nn::codec::OpusResult_Success,
            nn::codec::detail::OpusPacketInternal::GetOpusPacketCodingMode(
                &value,
                encodedDataBuffer,
                encodedDataSize)
        );
        EXPECT_EQ(codingMode, value);
    }
}

INSTANTIATE_TEST_CASE_P(
    RoundRobin,
    OpusEncoderApiComplexTest,
    ::testing::ValuesIn(ComplexTestParameterListStrings));

/**
 * @brief       OpusEncoder::SetBitRate() の異常系テストです。
 */
#if !defined(NN_SDK_BUILD_RELEASE)
////////////////////////////////////////////////////////////////////////
// Simple
////////////////////////////////////////////////////////////////////////
typedef OpusEncoderApiSimpleTest OpusEncoderApiSimpleDeathTest;

TEST_P(OpusEncoderApiSimpleDeathTest, SetBitRate)
{
    SCOPED_TRACE(GetParam());

    const int bitRateMaximum = nn::codec::GetOpusBitRateMax(encoder.GetChannelCount());
    const int bitRateMinimum = nn::codec::GetOpusBitRateMin(encoder.GetChannelCount());
    // 境界を確認しておく
    const int bitRates[] = {
        std::numeric_limits<int>::min(),
        -1,
        0,
        1,
        bitRateMinimum - 1,
        bitRateMaximum + 1,
        std::numeric_limits<int>::max(),
    };
    for (auto bitRate : bitRates)
    {
        EXPECT_DEATH_IF_SUPPORTED(encoder.SetBitRate(bitRate), "");
    }
}

TEST_P(OpusEncoderApiSimpleDeathTest, CodingMode)
{
    SCOPED_TRACE(GetParam());

    // 境界を確認しておく
    const nn::codec::OpusCodingMode codingModes[] = {
        static_cast<nn::codec::OpusCodingMode>(0),
        nn::codec::OpusCodingMode_Invalid
    };
    for (auto codingMode : codingModes)
    {
        EXPECT_DEATH_IF_SUPPORTED(encoder.BindCodingMode(codingMode), "");
    }
}

TEST_P(OpusEncoderApiSimpleDeathTest, WaveformType)
{
    const nn::codec::OpusWaveformType waveformTypes[] =
    {
        static_cast<nn::codec::OpusWaveformType>(0),
        nn::codec::OpusWaveformType_Invalid
    };
    for (auto waveformType : waveformTypes)
    {
        EXPECT_DEATH_IF_SUPPORTED(encoder.SetWaveformType(waveformType), "");
    }
}

INSTANTIATE_TEST_CASE_P(
    RoundRobin,
    OpusEncoderApiSimpleDeathTest,
    ::testing::ValuesIn(SimpleTestParameterListStrings)
);

////////////////////////////////////////////////////////////////////////
// Complex
////////////////////////////////////////////////////////////////////////
typedef OpusEncoderApiComplexTest OpusEncoderApiComplexDeathTest;

/**
 * @brief       OpusEncoder::EncodeInterleaved() の異常系テストです。
 */
TEST_P(OpusEncoderApiComplexDeathTest, Encode)
{
    SCOPED_TRACE(GetParam());

    const int frameSizeInUsec = 10000;
    const int supportedSampleCount = encoder.CalculateFrameSampleCount(frameSizeInUsec);

    // デコードサイズ格納先が nullptr
    EXPECT_DEATH_IF_SUPPORTED(
        encoder.EncodeInterleaved(
            nullptr,
            encodedDataBuffer,
            encodedDataBufferSize,
            pcmDataBuffer,
            supportedSampleCount), "");

    // デコード結果格納先が nullptr
    EXPECT_DEATH_IF_SUPPORTED(
        encoder.EncodeInterleaved(
            &encodedDataSize,
            nullptr,
            encodedDataBufferSize,
            pcmDataBuffer,
            supportedSampleCount), "");

    // デコード結果格納先が不十分
    /* libopus に任せる
    EXPECT_DEATH_IF_SUPPORTED(
        encoder.EncodeInterleaved(
            &encodedDataSize,
            encodedDataBuffer,
            static_cast<std::size_t>(encodedDataBufferSize - 1),
            pcmDataBuffer,
            supportedSampleCount), "");
    */

    // Pcm バッファが nullptr
    EXPECT_DEATH_IF_SUPPORTED(
        encoder.EncodeInterleaved(
            &encodedDataSize,
            encodedDataBuffer,
            encodedDataBufferSize,
            nullptr,
            supportedSampleCount), "");

    // 未サポートのサンプル数指定 (端数)
    EXPECT_DEATH_IF_SUPPORTED(
        encoder.EncodeInterleaved(
            &encodedDataSize,
            encodedDataBuffer,
            encodedDataBufferSize,
            pcmDataBuffer,
            pcmDataBufferSampleCount - 1), "");

    EXPECT_DEATH_IF_SUPPORTED(
        encoder.EncodeInterleaved(
            &encodedDataSize,
            encodedDataBuffer,
            encodedDataBufferSize,
            pcmDataBuffer,
            pcmDataBufferSampleCount + 1), "");
}

INSTANTIATE_TEST_CASE_P(
    RoundRobin,
    OpusEncoderApiComplexDeathTest,
    ::testing::ValuesIn(ComplexDeathTestParameterListStrings)
);


#endif // !defined(NN_SDK_BUILD_RELEASE)
