﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstdlib> // std::[malloc,free]

#include <nn/nn_Assert.h>
#include <nn/codec.h>

#include <nnt.h>
#include <nnt/codecUtil/testCodec_Util.h>
#include <nnt/codecUtil/testCodec_FixtureBase.h>

#include "testCodec_AacDecoderApiTestParameters.h"

namespace {

template <typename DecoderType>
class AacDecoderApiTestBase
{
private:
    const int DefaultIterationCount = 10;
    const int32_t DefaultOption = 0;

protected:
    AacDecoderApiTestBase() NN_NOEXCEPT
        : sampleRate(0)
        , channelCount(0)
        , type(nn::codec::Mpeg4AudioObjectType_Invalid)
        , iteration(DefaultIterationCount)
    {}
    virtual ~AacDecoderApiTestBase() NN_NOEXCEPT
    {
        Finalize();
    }
    nnt::codec::TestParameterSet GetTestParameterSet(const std::string& valueString) NN_NOEXCEPT
    {
        return GetTestParameterSet(valueString.c_str());
    }
    nnt::codec::TestParameterSet GetTestParameterSet(const char* valueString) NN_NOEXCEPT
    {
        nnt::codec::TestParameterSet parameterSet;
        parameterSet.sampleRate = nnt::codec::util::GetIntegerValueLabeledWith(valueString, "SampleRate");
        parameterSet.channelCount = nnt::codec::util::GetIntegerValueLabeledWith(valueString, "ChannelCount");
        parameterSet.type = static_cast<nn::codec::Mpeg4AudioObjectType>(nnt::codec::util::GetIntegerValueLabeledWith(valueString, "Type"));
        parameterSet.frame = nnt::codec::util::GetIntegerValueLabeledWith(valueString, "Frame");
        return parameterSet;
    }
    void Initialize(const nnt::codec::TestParameterSet& parameterSet, bool executeInitialization) NN_NOEXCEPT
    {
        sampleRate = parameterSet.sampleRate;
        channelCount = parameterSet.channelCount;
        type = parameterSet.type;
        NN_ASSERT(sampleRate != 0);
        NN_ASSERT(channelCount != 0);
        NN_ASSERT(type != nn::codec::Mpeg4AudioObjectType_Invalid);
        // Initializing
        pDecoder = nnt::codec::CreateAacDecoder<DecoderType>(DefaultOption);
        NN_ASSERT_NOT_NULL(this->pDecoder);
        decoderWorkBufferSize = pDecoder->GetWorkBufferSize(this->sampleRate, this->channelCount, this->type);
        decoderWorkBufferAddress = std::malloc(decoderWorkBufferSize);
        ASSERT_NE(nullptr, this->decoderWorkBufferAddress);
        if (executeInitialization)
        {
            ASSERT_EQ(
                nn::codec::AacDecoderResult_Success,
                pDecoder->Initialize(this->sampleRate, this->channelCount, this->type, this->decoderWorkBufferAddress, this->decoderWorkBufferSize)
            );
        }
    }
    virtual void Finalize() NN_NOEXCEPT
    {
        if (nullptr != pDecoder)
        {
            if (this->pDecoder->IsInitialized())
            {
                pDecoder->Finalize();
            }
            nnt::codec::DeleteAacDecoder(this->pDecoder);
        }
        this->pDecoder = nullptr;
        if (nullptr != decoderWorkBufferAddress)
        {
            std::free(decoderWorkBufferAddress);
        }
        decoderWorkBufferAddress = nullptr;
        decoderWorkBufferSize = 0;
    }
    DecoderType* pDecoder;
    void* decoderWorkBufferAddress;
    std::size_t decoderWorkBufferSize;
    int sampleRate;
    int channelCount;
    nn::codec::Mpeg4AudioObjectType type;
    int iteration;
};

template <typename T>
class AacDecoderApiSimpleTest
    : public AacDecoderApiTestBase<T>
    , public nnt::codec::DecodedDataBufferBase
    , public nnt::codec::EncodedDataBufferBase
    , public ::testing::Test
{
protected:
    typedef T DecoderType;
    ~AacDecoderApiSimpleTest() NN_NOEXCEPT
    {
        Finalize();
    }
    virtual void Initialize(const char* parameterString, bool executeInitialization) NN_NOEXCEPT
    {
        auto parameterSet = AacDecoderApiTestBase<T>::GetTestParameterSet(std::string(parameterString));
        // Simple では Frame を指定していないので "0" になっている。
        // Frame を変えて実施するテストは含まれていない。
        // 固定で 20 ms を与えておく。
        parameterSet.frame = 20000;
        AacDecoderApiTestBase<T>::Initialize(parameterSet, executeInitialization);
        EncodedDataBufferBase::Initialize(parameterSet);
        DecodedDataBufferBase::Initialize(parameterSet);
    }
    virtual void Finalize() NN_NOEXCEPT
    {
        EncodedDataBufferBase::Finalize();
        DecodedDataBufferBase::Finalize();
        AacDecoderApiTestBase<T>::Finalize();
    }
};

template <typename T>
class AacDecoderApiSimpleDeathTest
    : public AacDecoderApiSimpleTest<T>
{
};

}  // namespace

/**
 * @brief       インスタンス化する型を列挙します。
 */
typedef ::testing::Types<nn::codec::AacDecoder> CommonTestImplementations;

TYPED_TEST_CASE(AacDecoderApiSimpleTest, CommonTestImplementations);

TYPED_TEST(AacDecoderApiSimpleTest, InitializeFinalize)
{
    for (const auto parameterString : SimpleTestParameterListStrings)
    {
        this->Initialize(parameterString, false);
        SCOPED_TRACE(parameterString);
        for (auto k = 0; k < this->iteration; ++k)
        {
            // 戻値を確認する。
            EXPECT_EQ(
                nn::codec::AacDecoderResult_Success,
                this->pDecoder->Initialize(this->sampleRate, this->channelCount, this->type, this->decoderWorkBufferAddress, this->decoderWorkBufferSize)
            );
            // 戻値はないので、停止しないことを確認する。
            this->pDecoder->Finalize();
        }
        this->Finalize();
    }
}

TYPED_TEST(AacDecoderApiSimpleTest, GetWorkBufferSize)
{
    for (const auto parameterString : SimpleTestParameterListStrings)
    {
        this->Initialize(parameterString, false);
        SCOPED_TRACE(parameterString);
        for (auto k = 0; k < this->iteration; ++k)
        {
            // GetWorkBufferSize() はいつでも有効
            const auto workBufferSize = this->pDecoder->GetWorkBufferSize(this->sampleRate, this->channelCount, this->type);
            EXPECT_GT(workBufferSize, 0u);
            ASSERT_EQ(
                nn::codec::AacDecoderResult_Success,
                this->pDecoder->Initialize(this->sampleRate, this->channelCount, this->type, this->decoderWorkBufferAddress, this->decoderWorkBufferSize)
            );
            // 取得したサイズをリファレンスとして照合する。
            EXPECT_EQ(workBufferSize, this->pDecoder->GetWorkBufferSize(this->sampleRate, this->channelCount, this->type));
            this->pDecoder->Finalize();
            EXPECT_EQ(workBufferSize, this->pDecoder->GetWorkBufferSize(this->sampleRate, this->channelCount, this->type));
        }
        this->Finalize();
    }
}

TYPED_TEST(AacDecoderApiSimpleTest, GetSampleRate)
{
    for (const auto parameterString : SimpleTestParameterListStrings)
    {
        this->Initialize(parameterString, false);
        SCOPED_TRACE(parameterString);
        ASSERT_EQ(
            nn::codec::AacDecoderResult_Success,
            this->pDecoder->Initialize(this->sampleRate, this->channelCount, this->type, this->decoderWorkBufferAddress, this->decoderWorkBufferSize)
        );
        EXPECT_EQ(this->sampleRate, this->pDecoder->GetSampleRate());
        this->Finalize();
    }
}

TYPED_TEST(AacDecoderApiSimpleTest, GetChannelCount)
{
    for (const auto parameterString : SimpleTestParameterListStrings)
    {
        this->Initialize(parameterString, false);
        SCOPED_TRACE(parameterString);
        ASSERT_EQ(
            nn::codec::AacDecoderResult_Success,
            this->pDecoder->Initialize(this->sampleRate, this->channelCount, this->type, this->decoderWorkBufferAddress, this->decoderWorkBufferSize)
        );
        EXPECT_EQ(this->channelCount, this->pDecoder->GetChannelCount());
        this->Finalize();
    }
}

TYPED_TEST(AacDecoderApiSimpleTest, GetMpeg4AudioObjectType)
{
    for (const auto parameterString : SimpleTestParameterListStrings)
    {
        this->Initialize(parameterString, false);
        SCOPED_TRACE(parameterString);
        ASSERT_EQ(
            nn::codec::AacDecoderResult_Success,
            this->pDecoder->Initialize(this->sampleRate, this->channelCount, this->type, this->decoderWorkBufferAddress, this->decoderWorkBufferSize)
        );
        EXPECT_EQ(this->type, this->pDecoder->GetMpeg4AudioObjectType());
        this->Finalize();
    }
}

TYPED_TEST(AacDecoderApiSimpleTest, Initialized)
{
    for (const auto parameterString : SimpleTestParameterListStrings)
    {
        this->Initialize(parameterString, false);
        SCOPED_TRACE(parameterString);
        for (auto k = 0; k < this->iteration; ++k)
        {
            EXPECT_EQ(false, this->pDecoder->IsInitialized());
            ASSERT_EQ(
                nn::codec::AacDecoderResult_Success,
                this->pDecoder->Initialize(this->sampleRate, this->channelCount, this->type, this->decoderWorkBufferAddress, this->decoderWorkBufferSize)
            );
            EXPECT_EQ(true, this->pDecoder->IsInitialized());
            this->pDecoder->Finalize();
            EXPECT_EQ(false, this->pDecoder->IsInitialized());
        }
        this->Finalize();
    }
}

TYPED_TEST(AacDecoderApiSimpleTest, DoubleInitializing)
{
    for (const auto parameterString : SimpleTestParameterListStrings)
    {
        this->Initialize(parameterString, true);
        SCOPED_TRACE(parameterString);
        auto pAnotherDecoder = nnt::codec::CreateAacDecoder<typename TestFixture::DecoderType>(0);
#if defined(NN_CODEC_HAVE_HARDWARE_ACCELERATOR) // ハードウェアアクセラレータを持っている場合
        // TransferMemory で失敗する (既にプロセスに権利を委譲している領域に対して再度 TransferMemory することになる)
        EXPECT_DEATH_IF_SUPPORTED(
            pAnotherDecoder->Initialize(
                this->pDecoder->GetSampleRate(),
                this->pDecoder->GetChannelCount(),
                this->decoderWorkBufferAddress,
                this->decoderWorkBufferSize), "");
#else  // defined(NN_CODEC_HAVE_HARDWARE_ACCELERATOR)
        // ワークバッファとインスタンスの紐付けはないので、使用中のワークバッファを別インスタンスの Initialize() に指定しても通る。
        EXPECT_EQ(
            nn::codec::AacDecoderResult_Success,
            pAnotherDecoder->Initialize(
                this->pDecoder->GetSampleRate(),
                this->pDecoder->GetChannelCount(),
                this->pDecoder->GetMpeg4AudioObjectType(),
                this->decoderWorkBufferAddress,
                this->decoderWorkBufferSize)
        );
#endif // defined(NN_CODEC_HAVE_HARDWARE_ACCELERATOR)
        nnt::codec::DeleteAacDecoder(pAnotherDecoder);
        this->Finalize();
    }
}

#if !defined(NN_SDK_BUILD_RELEASE)

TYPED_TEST_CASE(AacDecoderApiSimpleDeathTest, CommonTestImplementations);

TYPED_TEST(AacDecoderApiSimpleDeathTest, GetWorkBufferSize)
{
    for (const auto parameterString : SimpleTestParameterListStrings)
    {
        this->Initialize(parameterString, false);
        SCOPED_TRACE(parameterString);
        // Unsupported channel count.
        EXPECT_DEATH_IF_SUPPORTED(this->pDecoder->GetWorkBufferSize(this->sampleRate, 0, this->type), "");
        EXPECT_DEATH_IF_SUPPORTED(this->pDecoder->GetWorkBufferSize(this->sampleRate, 7, this->type), "");

        // Unsupported sample rate.
        EXPECT_DEATH_IF_SUPPORTED(this->pDecoder->GetWorkBufferSize(this->sampleRate - 1, this->channelCount, this->type), "");
        EXPECT_DEATH_IF_SUPPORTED(this->pDecoder->GetWorkBufferSize(this->sampleRate + 1, this->channelCount, this->type), "");

        // Unsupported Mpeg4AudioObjectType
        EXPECT_DEATH_IF_SUPPORTED(this->pDecoder->GetWorkBufferSize(this->sampleRate, this->channelCount, nn::codec::Mpeg4AudioObjectType_Invalid), "");
        this->Finalize();
    }
}

TYPED_TEST(AacDecoderApiSimpleDeathTest, FailedInInitializing)
{
    for (const auto parameterString : SimpleTestParameterListStrings)
    {
        this->Initialize(parameterString, false);
        SCOPED_TRACE(parameterString);
        // Unsupported channel count.
        EXPECT_DEATH_IF_SUPPORTED(this->pDecoder->Initialize(this->sampleRate, 0, this->type, this->decoderWorkBufferAddress, this->decoderWorkBufferSize), "");
        EXPECT_DEATH_IF_SUPPORTED(this->pDecoder->Initialize(this->sampleRate, 7, this->type, this->decoderWorkBufferAddress, this->decoderWorkBufferSize), "");

        // Unsupported sample rate.
        EXPECT_DEATH_IF_SUPPORTED(this->pDecoder->Initialize(this->sampleRate - 1, this->channelCount, this->type, this->decoderWorkBufferAddress, this->decoderWorkBufferSize), "");
        EXPECT_DEATH_IF_SUPPORTED(this->pDecoder->Initialize(this->sampleRate + 1, this->channelCount, this->type, this->decoderWorkBufferAddress, this->decoderWorkBufferSize), "");

        // Unsupported Mpeg4AudioObjectType
        EXPECT_DEATH_IF_SUPPORTED(this->pDecoder->Initialize(this->sampleRate, this->channelCount, nn::codec::Mpeg4AudioObjectType_Invalid, this->decoderWorkBufferAddress, this->decoderWorkBufferSize), "");

        // nullptr
        EXPECT_DEATH_IF_SUPPORTED(this->pDecoder->Initialize(this->sampleRate, this->channelCount, this->type, nullptr, this->decoderWorkBufferSize), "");

        // Smaller work buffer size
        EXPECT_DEATH_IF_SUPPORTED(this->pDecoder->Initialize(this->sampleRate, this->channelCount, this->type, this->decoderWorkBufferAddress, this->decoderWorkBufferSize - 1), "");
        this->Finalize();
    }
}

TYPED_TEST(AacDecoderApiSimpleDeathTest, InitializeUnderInitializedState)
{
    for (const auto parameterString : SimpleTestParameterListStrings)
    {
        this->Initialize(parameterString, true);
        SCOPED_TRACE(parameterString);
        void* tmp = std::malloc(this->decoderWorkBufferSize);
        // Initialization under "Initialized" state.
        EXPECT_DEATH_IF_SUPPORTED(this->pDecoder->Initialize(this->sampleRate, this->channelCount, this->type, tmp, this->decoderWorkBufferSize), "");
        std::free(tmp);
        this->Finalize();
    }
}

TYPED_TEST(AacDecoderApiSimpleDeathTest, CallingUnderUninitializedState)
{
    for (const auto parameterString : SimpleTestParameterListStrings)
    {
        this->Initialize(parameterString, false);
        SCOPED_TRACE(parameterString);
        // Calling some APIs requiring "Initialized" state.
        EXPECT_DEATH_IF_SUPPORTED(this->pDecoder->GetSampleRate(), "");
        EXPECT_DEATH_IF_SUPPORTED(this->pDecoder->GetChannelCount(), "");
        EXPECT_DEATH_IF_SUPPORTED(this->pDecoder->GetMpeg4AudioObjectType(), "");
        std::size_t consumed;
        int sampleCount;
        EXPECT_DEATH_IF_SUPPORTED(
            this->pDecoder->DecodeInterleaved(
                &consumed, &sampleCount, this->decodedDataBuffer, this->decodedDataBufferSize, this->encodedDataBuffer, this->encodedDataSize), "");
        EXPECT_DEATH_IF_SUPPORTED(this->pDecoder->Finalize(), "");
        this->Finalize();
    }
}

TYPED_TEST(AacDecoderApiSimpleDeathTest, CallingUnderFinalizedState)
{
    for (const auto parameterString : SimpleTestParameterListStrings)
    {
        this->Initialize(parameterString, true);
        SCOPED_TRACE(parameterString);
        this->pDecoder->Finalize();

        // Calling some APIs requiring "Initialized" state.
        EXPECT_DEATH_IF_SUPPORTED(this->pDecoder->GetSampleRate(), "");
        EXPECT_DEATH_IF_SUPPORTED(this->pDecoder->GetChannelCount(), "");
        EXPECT_DEATH_IF_SUPPORTED(this->pDecoder->GetMpeg4AudioObjectType(), "");
        std::size_t consumed;
        int sampleCount;
        EXPECT_DEATH_IF_SUPPORTED(
            this->pDecoder->DecodeInterleaved(
                &consumed, &sampleCount, this->decodedDataBuffer, this->decodedDataBufferSize, this->encodedDataBuffer, this->encodedDataSize), "");
        EXPECT_DEATH_IF_SUPPORTED(this->pDecoder->Finalize(), "");
        this->Finalize();
    }
}

#endif // !defined(NN_SDK_BUILD_RELEASE)
