﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <random>
#include <vector>

#include <nn/nn_Common.h>
#include <nn/nn_Log.h>
#include <nn/fs.h>
#include <nn/os.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/image/image_JpegEncoder.h>

#include <nnt.h>

#include <nn/capsrv/capsrv_Result.h>
#include <nn/capsrv/capsrv_AlbumFileSizeLimit.h>

#include "../../../../../Programs/Iris/Sources/Libraries/capsrv/server/capsrvServer_Config.h"
#include "../../../../../Programs/Iris/Sources/Libraries/capsrv/server/capsrvServer_ResultPrivate.h"
#include "../../../../../Programs/Iris/Sources/Libraries/capsrv/server/album/capsrvServer_VerifyScreenShotFileData.h"
#include "../../../../../Programs/Iris/Sources/Libraries/capsrv/server/detail/capsrvServer_CalculateJpegMac.h"
#include "../../../../../Programs/Iris/Sources/Libraries/capsrv/server/detail/capsrvServer_ExifWorkStorage.h"

#include "../../Common/testCapsrv_TestFileUtility.h"

namespace {

    std::vector<uint8_t> EncodeJpeg(const nnt::capsrv::TestRawImage& rawImage) NN_NOEXCEPT
    {
        nn::image::JpegEncoder encoder;
        encoder.SetPixelData(rawImage.data.data(), rawImage.GetDimension(), 4);
        encoder.SetQuality(50);
        NN_ABORT_UNLESS(
            encoder.Analyze() == nn::image::JpegStatus_Ok
        );
        std::vector<char> workBuffer;
        std::vector<uint8_t> jpeg;
        jpeg.resize(nn::capsrv::AlbumFileSizeLimit_ScreenShot);
        size_t size;
        workBuffer.resize(encoder.GetAnalyzedWorkBufferSize());
        NN_ABORT_UNLESS(
            encoder.Encode(&size, jpeg.data(), jpeg.size(), workBuffer.data(), workBuffer.size()) == nn::image::JpegStatus_Ok
        );
        jpeg.resize(size);
        return jpeg;
    }

    std::vector<uint8_t> EncodeJpeg(const nnt::capsrv::TestRawImage& rawImage, nn::image::ExifBuilder& exifBuilder) NN_NOEXCEPT
    {
        nn::image::JpegEncoder encoder;
        encoder.SetPixelData(rawImage.data.data(), rawImage.GetDimension(), 4);
        encoder.SetQuality(50);
        NN_ABORT_UNLESS(
            encoder.Analyze() == nn::image::JpegStatus_Ok
        );
        std::vector<char> workBuffer;
        std::vector<uint8_t> jpeg;
        jpeg.resize(nn::capsrv::AlbumFileSizeLimit_ScreenShot);
        size_t size;
        workBuffer.resize(encoder.GetAnalyzedWorkBufferSize());
        NN_ABORT_UNLESS(
            encoder.Encode(&size, jpeg.data(), jpeg.size(), workBuffer.data(), workBuffer.size(), &exifBuilder) == nn::image::JpegStatus_Ok
        );
        jpeg.resize(size);
        return jpeg;
    }

}

TEST(UnitTest, VerifyScreenShotFileData)
{
    const unsigned long DefaultRandomSeed = 0x95C52A56;
    const nn::ncm::ApplicationId DefaultApplicationId = { 0x97A94A67 };
    const nn::capsrv::AlbumFileDateTime DefaultDateTime = { 2016, 4, 22, 8, 54, 28, 9 };
    const char DefaultDateTimeString[] = "2016:04:22 08:54:28";
    const nn::capsrv::AlbumFileId DefaultFileId = { DefaultApplicationId, DefaultDateTime, nn::capsrv::AlbumStorage_Nand, nn::capsrv::AlbumFileContents_ScreenShot };

    // テスト用の signer を挿しておく
    nn::capsrv::server::detail::SetScreenShotFileSignFunction(
        nn::capsrv::server::detail::ScreenShotFileSignatureVersion_Version1,
        [](nn::capsrv::server::detail::Signature* pOutValue, const void* data, size_t dataSize) -> nn::Result
        {
            std::memcpy(pOutValue, data, std::min<size_t>(sizeof(*pOutValue), dataSize));
            NN_RESULT_SUCCESS;
        }
    );

    nn::capsrv::server::detail::ExifBuildWorkStorage exifBuildWork;
    nn::capsrv::server::EnvironmentInfo env;
    env.Initialize();
    NN_UTIL_SCOPE_EXIT{ env.Finalize(); };

    nnt::capsrv::TestRawImage rawImage;
    {
        std::mt19937 engine;
        engine.seed(DefaultRandomSeed);
        rawImage = nnt::capsrv::TestFileUtility::CreateRawImageDataRgba32(nn::capsrv::server::ScreenShotWidth, nn::capsrv::server::ScreenShotHeight, 16, engine);
    }

    // 正常な JPEG が読めることの確認
    nnt::capsrv::TestScreenShotFileMetaData goodMeta;
    goodMeta.applicationId = DefaultApplicationId;
    goodMeta.time = DefaultDateTime;
    goodMeta.orientation = nn::capsrv::ScreenShotOrientation_Default;
    auto goodJpeg = nnt::capsrv::TestFileUtility::CreateScreenShotFileData(rawImage, goodMeta, nnt::capsrv::TestFileSignaturePattern_Valid);
    NN_LOG("Checking valid jpeg data\n");
    NNT_EXPECT_RESULT_SUCCESS(
        nn::capsrv::server::album::VerifyScreenShotFileData(&DefaultFileId, goodJpeg.data(), goodJpeg.size(), env)
    );

    NN_LOG("Checking empty data\n");
    NNT_EXPECT_RESULT_FAILURE(nn::capsrv::server::ResultInternalFileDataVerificationEmptyFileData,
        nn::capsrv::server::album::VerifyScreenShotFileData(&DefaultFileId, nullptr, 0, env)
    );

    NN_LOG("Checking ExtractExifInfo()...\n");
    {
        // JPEG ですらないバイト列を弾くことの確認
        NN_LOG("  Checking non-JPEG data\n");
        {
            // ランダムに生成したのできっと jpeg として解釈できないデータになっている
            NNT_EXPECT_RESULT_FAILURE(nn::capsrv::server::ResultInternalSignatureExifExtractionFailed,
                nn::capsrv::server::album::VerifyScreenShotFileData(&DefaultFileId, rawImage.data.data(), rawImage.data.size(), env)
            );
        }

        // Exif がない JPEG を弾くことの確認
        NN_LOG("  Checking no Exif\n");
        {
            auto jpeg = EncodeJpeg(rawImage);
            NNT_EXPECT_RESULT_FAILURE(nn::capsrv::server::ResultInternalSignatureExifExtractionFailed,
                nn::capsrv::server::album::VerifyScreenShotFileData(&DefaultFileId, jpeg.data(), jpeg.size(), env)
            );
        }

        // 不正な Exif の JPEG を弾くことの確認
        {
            // TODO:
            //   Exif の解析に失敗するパターンを作るのが大変なので未実装
        }

        // MakerNote のない JPEG を弾くことの確認
        NN_LOG("  Checking no MakerNote\n");
        {
            nn::image::ExifBuilder exifBuilder(&exifBuildWork, sizeof(exifBuildWork));
            exifBuilder.SetDateTime(DefaultDateTimeString, std::strlen(DefaultDateTimeString) + 1);
            exifBuilder.Analyze();
            auto jpeg = EncodeJpeg(rawImage, exifBuilder);
            NNT_EXPECT_RESULT_FAILURE(nn::capsrv::server::ResultInternalSignatureMakerNoteExtractionFailed,
                nn::capsrv::server::album::VerifyScreenShotFileData(&DefaultFileId, jpeg.data(), jpeg.size(), env)
            );
        }

        // 不正な MakerNote の JPEG を弾くことの確認
        NN_LOG("  Checking ill-formed MakerNote\n");
        {
            const char InvalidMakerNote[] = "hogefuga";
            nn::image::ExifBuilder exifBuilder(&exifBuildWork, sizeof(exifBuildWork));
            exifBuilder.SetDateTime(DefaultDateTimeString, std::strlen(DefaultDateTimeString) + 1);
            exifBuilder.SetMakerNote(InvalidMakerNote, sizeof(InvalidMakerNote));
            exifBuilder.Analyze();
            auto jpeg = EncodeJpeg(rawImage, exifBuilder);
            // MakerNoteParser のエラーが返る
            NNT_EXPECT_RESULT_FAILURE(nn::capsrv::ResultAlbumInvalidFileData,
                nn::capsrv::server::album::VerifyScreenShotFileData(&DefaultFileId, jpeg.data(), jpeg.size(), env)
            );
        }

        // 撮影日時のない JPEG を弾くことの確認
    }

    NN_LOG("Checking VerifyDateTime...\n");
    {
        //NN_LOG("  Checking no DateTime\n");
        //{
        //    nn::image::ExifBuilder exifBuilder(&exifBuildWork, sizeof(exifBuildWork));
        //    exifBuilder.Analyze();
        //    auto jpeg = EncodeJpeg(rawImage, exifBuilder);
        //    NNT_EXPECT_RESULT_FAILURE(nn::capsrv::server::ResultInternalFileDataVerificationDateTimeExtractionFailed,
        //        nn::capsrv::server::album::VerifyScreenShotFileData(&DefaultFileId, jpeg.data(), jpeg.size(), env)
        //    );
        //}

        // 不正な長さの DateTime を弾くことの確認
        {
            // TODO:
            //   nn::image::ExifBuilder で長さの間違った DateTime が作れないので未実装
        }

        // ファイル ID と矛盾した DateTime を弾くことの確認
        NN_LOG("  Checking inconsistent DateTime\n");
        {
            nn::capsrv::AlbumFileId badFileId = {
                DefaultApplicationId,
                { 2016, 3, 11, 1 ,2 ,3, 4 },
                nn::capsrv::AlbumStorage_Nand,
                nn::capsrv::AlbumFileContents_ScreenShot,
            };
            NNT_EXPECT_RESULT_FAILURE(nn::capsrv::server::ResultInternalFileDataVerificationInconsistentDateTime,
                nn::capsrv::server::album::VerifyScreenShotFileData(&badFileId, goodJpeg.data(), goodJpeg.size(), env)
            );
        }
    }

    NN_LOG("Checking VerifyApplicationId...\n");
    {
        // 不正な暗号化された ApplicationId を弾くことの確認
        {
            // やや面倒なので省略。
            // どの道一致判定で弾かれるので大丈夫なハズ。
        }

        NN_LOG("  Checking inconsistent ApplicationID\n");
        {
            nn::capsrv::AlbumFileId badFileId = {
                { 0x4577DF72 },
                DefaultDateTime,
                nn::capsrv::AlbumStorage_Nand,
                nn::capsrv::AlbumFileContents_ScreenShot,
            };
            NNT_EXPECT_RESULT_FAILURE(nn::capsrv::server::ResultInternalFileDataVerificationInconsistentApplicationId,
                nn::capsrv::server::album::VerifyScreenShotFileData(&badFileId, goodJpeg.data(), goodJpeg.size(), env)
            );
        }
    }

    NN_LOG("Checking VerifySignature...\n");
    {
        NN_LOG("  Checking inconsistent Signature\n");
        {
            std::vector<uint8_t> badJpeg;
            nn::Result result;
            // Zero
            badJpeg = nnt::capsrv::TestFileUtility::CreateScreenShotFileData(rawImage, goodMeta, nnt::capsrv::TestFileSignaturePattern_Zero);
            NNT_EXPECT_RESULT_FAILURE(nn::capsrv::server::ResultInternalFileDataVerificationInconsistentSignature,
                nn::capsrv::server::album::VerifyScreenShotFileData(&DefaultFileId, badJpeg.data(), badJpeg.size(), env)
            );
            // HeadBitFlip
            badJpeg = nnt::capsrv::TestFileUtility::CreateScreenShotFileData(rawImage, goodMeta, nnt::capsrv::TestFileSignaturePattern_HeadBitFlip);
            NNT_EXPECT_RESULT_FAILURE(nn::capsrv::server::ResultInternalFileDataVerificationInconsistentSignature,
                nn::capsrv::server::album::VerifyScreenShotFileData(&DefaultFileId, badJpeg.data(), badJpeg.size(), env)
            );
            // TailBitFlip
            badJpeg = nnt::capsrv::TestFileUtility::CreateScreenShotFileData(rawImage, goodMeta, nnt::capsrv::TestFileSignaturePattern_TailBitFlip);
            result = nn::capsrv::server::album::VerifyScreenShotFileData(&DefaultFileId, badJpeg.data(), badJpeg.size(), env);
            EXPECT_TRUE(nn::capsrv::server::ResultInternalFileDataVerificationInconsistentSignature::Includes(result));
        }
    }

    SUCCEED();
}// NOLINT(impl/function_size)
