﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <vector>
#include <atomic>
#include <limits>

#include <nn/nn_Common.h>
#include <nn/nn_Log.h>
#include <nn/nn_Assert.h>

#include <nn/os.h>
#include <nn/util/util_StringUtil.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/image/image_JpegEncoder.h>
#include <nn/image/image_JpegDecoder.h>
#include <nn/image/image_ExifExtractor.h>

#include <nn/capsrv/capsrv_AlbumControl.h>
#include <nn/capsrv/capsrv_ScreenShotControl.h>
#include <nn/capsrv/capsrv_AlbumAccess.h>
#include "../../Common/testCapsrv_TestFileUtility.h"
#include "../../Common/testCapsrv_DirectAlbumAccessor.h"

#include <nnt.h>

extern std::atomic_int g_ScreenShotCount;
extern nn::os::EventType g_TestEndEvent;

namespace {

    void SaveOverlayThumbnailToFile(const nnt::capsrv::TestRawImage& thumb, int trial) NN_NOEXCEPT
    {
        NN_LOG("getting overlay-thumb\n");
        nn::image::Dimension dim;
        dim.width = nn::capsrv::ThumbnailImageSize_Width;
        dim.height = nn::capsrv::ThumbnailImageSize_Height;
        nn::image::JpegEncoder encoder;

        encoder.SetPixelData(thumb.data.data(), dim, 1);
        NN_ABORT_UNLESS(encoder.Analyze() == nn::image::JpegStatus_Ok);

        std::vector<uint8_t> jpeg;
        std::vector<uint8_t> work;
        jpeg.resize(thumb.data.size());
        work.resize(encoder.GetAnalyzedWorkBufferSize());

        size_t encodedSize = 0;
        NN_ABORT_UNLESS(encoder.Encode(&encodedSize, jpeg.data(), jpeg.size(), work.data(), work.size()) == nn::image::JpegStatus_Ok);

        char path[512] = {};
        nn::util::SNPrintf(path, sizeof(path), "tNA:/thumb%d.jpg", trial);
        nnt::capsrv::DirectAlbumAccessor::SaveFile(jpeg, path);
    }

    void LoadFileAndSaveAsEdited(const nn::capsrv::AlbumFileId& fileId, int trial) NN_NOEXCEPT
    {
        NN_LOG("loading full-file and save-as-edited\n");
        std::vector<uint8_t> data;
        data.resize(nn::capsrv::AlbumFileSizeLimit_ScreenShot);
        size_t dataSize = 0;
        std::vector<uint8_t> work;
        work.resize(nn::image::ExifExtractor::GetWorkBufferSize());

        NN_ABORT_UNLESS_RESULT_SUCCESS(
            nn::capsrv::LoadAlbumScreenShotFile(&dataSize, data.data(), data.size(), &fileId)
        );

        // サムネイル領域を取得
        const void* pExifData = nullptr;
        size_t exifDataSize = 0;
        NN_ABORT_UNLESS(nn::image::JpegDecoder::GetExifData(&pExifData, &exifDataSize, data.data(), dataSize) == nn::image::JpegStatus_Ok);
        nn::image::ExifExtractor exifExtractor(work.data(), work.size());
        exifExtractor.SetExifData(pExifData, exifDataSize);
        NN_ABORT_UNLESS(exifExtractor.Analyze() == nn::image::JpegStatus_Ok);

        const uint8_t* pThumb = nullptr;
        size_t thumbSize = 0;
        pThumb = reinterpret_cast<const uint8_t*>(exifExtractor.ExtractThumbnail(&thumbSize));
        NN_ABORT_UNLESS(pThumb != nullptr && thumbSize > 0);

        std::vector<uint8_t> thumb;
        thumb.assign(pThumb, pThumb + thumbSize);

        nn::image::Dimension dim;
        NN_ABORT_UNLESS(exifExtractor.ExtractEffectiveDimension(&dim));

        char path[512] = {};
        nn::util::SNPrintf(path, sizeof(path), "tNA:/viewer_thumb%d.jpg", trial);
        nnt::capsrv::DirectAlbumAccessor::SaveFile(thumb, path);

        // 編集済みとして保存してみる
        nn::capsrv::AlbumEntry entry = {};
        {
            std::vector<uint8_t> raw;
            {
                std::vector<uint8_t> rawWork;
                raw.resize(4 * dim.width * dim.height);
                nn::image::JpegDecoder decoder;
                decoder.SetImageData(data.data(), data.size());
                decoder.Analyze();
                rawWork.resize(decoder.GetAnalyzedWorkBufferSize());
                decoder.Decode(raw.data(), raw.size(), 1, rawWork.data(), rawWork.size());
            }
            std::vector<uint8_t> rawThumb;
            nn::image::Dimension dimThumb;
            {
                std::vector<uint8_t> rawThumbWork;
                nn::image::JpegDecoder decoder;
                decoder.SetImageData(thumb.data(), thumb.size());
                decoder.Analyze();
                rawThumbWork.resize(decoder.GetAnalyzedWorkBufferSize());
                dimThumb = decoder.GetAnalyzedDimension();
                rawThumb.resize(4 * dimThumb.width * dimThumb.height);
                decoder.Decode(rawThumb.data(), rawThumb.size(), 1, rawThumbWork.data(), rawThumbWork.size());
            }

            nn::capsrv::ScreenShotAttribute attribute;
            attribute.SetDefault();
            nn::capsrv::AppletData appletData = {};
            NN_ABORT_UNLESS_RESULT_SUCCESS(
                nn::capsrv::SaveEditedScreenShot(&entry, raw.data(), raw.size(), dim.width, dim.height, rawThumb.data(), rawThumb.size(), dimThumb.width, dimThumb.height, attribute, appletData, &fileId)
            );
        }
    }

    void LoadThumbnailFile(const nn::capsrv::AlbumFileId& fileId, int trial) NN_NOEXCEPT
    {
        NN_LOG("loading thumbnail file\n");
        std::vector<uint8_t> data;
        data.resize(nn::capsrv::ViewerThumbnailImageDataSize_EncodedLimit);
        size_t dataSize = 0;

        NN_ABORT_UNLESS_RESULT_SUCCESS(
            nn::capsrv::LoadAlbumFileThumbnail(&dataSize, data.data(), data.size(), &fileId)
        );

        char path[512] = {};
        nn::util::SNPrintf(path, sizeof(path), "tNA:/viewer_thumb_load%d.jpg", trial);
        nnt::capsrv::DirectAlbumAccessor::SaveFile(data, path);
    }

    void CheckImageAttribute(const nn::capsrv::AlbumFileId& fileId) NN_NOEXCEPT
    {
        NN_LOG("checking attribute\n");
        std::vector<uint8_t> data;
        data.resize(nn::capsrv::ViewerThumbnailImageDataSize_Raw);
        std::vector<uint8_t> work;
        work.resize(nn::capsrv::ViewerThumbnailImageDataSize_EncodedLimit);

        const nn::capsrv::AppletData zeroAppletData = {};

        int width = 0;
        int height = 0;
        nn::capsrv::ScreenShotAttribute attribute = {};
        nn::capsrv::AppletData appletData = {};
        NN_ABORT_UNLESS_RESULT_SUCCESS(
            nn::capsrv::LoadAlbumScreenShotThumbnailImage(&width, &height, &attribute, &appletData, data.data(), data.size(), &fileId, nn::capsrv::ScreenShotDecodeOption::GetDefaultValue(), work.data(), work.size())
        );
        EXPECT_EQ(nn::capsrv::ViewerThumbnailImageSize_Width, width);
        EXPECT_EQ(nn::capsrv::ViewerThumbnailImageSize_Height, height);
        EXPECT_EQ(nn::capsrv::ScreenShotSize_1280x720, attribute.size);
        EXPECT_EQ(nn::capsrv::ScreenShotOrientation_Default, attribute.orientation);
        EXPECT_EQ(nn::capsrv::AlbumFileDescription_ScreenShotCaptured, attribute.description);
        EXPECT_EQ(1, attribute.frameCount);
        EXPECT_EQ(0, attribute.frameRateNumerator);
        EXPECT_EQ(0, attribute.frameRateDenominator);
        EXPECT_EQ(0, attribute.dataDurationMilliseconds);
        EXPECT_EQ(0, attribute.keyFrameInterval);
        EXPECT_EQ(0, std::memcmp(&zeroAppletData, &appletData, sizeof(nn::capsrv::AppletData)));
    }


    void TakingScreenShotTest() NN_NOEXCEPT
    {
        uint64_t seqNo = std::numeric_limits<uint64_t>::max();
        nn::ncm::ProgramId targetIdBase = { 0x123456789ABCDEF0 };
        nn::applet::AppletResourceUserId targetAruid = { 0 };
        nn::capsrv::ScreenShotAttribute attribute;
        attribute.SetDefault();

        nn::capsrv::InitializeScreenShotControl();
        nn::capsrv::InitializeAlbumControl();
        nn::capsrv::InitializeAlbumAccess();
        NN_UTIL_SCOPE_EXIT{
            nn::capsrv::FinalizeAlbumAccess();
            nn::capsrv::FinalizeAlbumControl();
            nn::capsrv::FinalizeScreenShotControl();
        };

        nnt::capsrv::TestRawImage thumb;
        thumb.data.resize(nn::capsrv::ThumbnailImageDataSize_Default);
        thumb.width = nn::capsrv::ThumbnailImageSize_Width;
        thumb.height = nn::capsrv::ThumbnailImageSize_Height;

        for(int i = 0; i < 10; i++)
        {
            nn::ncm::ProgramId targetId = { targetIdBase.value + i };
            NN_LOG("TakeScreenShot %d\n", i);
            nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(800));
            nn::capsrv::RequestTakingScreenShot(seqNo, targetId, targetAruid, attribute, {}, nn::TimeSpan::FromMilliSeconds(500));

            // 即座にサムネイルを取りに行くと前のものが取れるので少し待つ
            nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(500));

            nn::capsrv::AlbumFileId fileId = {};
            size_t dataSize = 0;
            nn::capsrv::GetLastThumbnail(&fileId, &dataSize, thumb.data.data(), thumb.data.size());

            EXPECT_EQ(targetId.value, fileId.applicationId.value);
            EXPECT_EQ(thumb.data.size(), dataSize);

            // サムネイルを JPEG で保存
            SaveOverlayThumbnailToFile(thumb, i);

            // ついでに読んでみる
            LoadFileAndSaveAsEdited(fileId, i);

            // 更にサムネイルだけ読んでみる
            LoadThumbnailFile(fileId, i);

            // 属性を確認
            CheckImageAttribute(fileId);

            g_ScreenShotCount++;
        }
    } // NOLINT(impl/function_size)

    void TakingScreenShotTestRotation() NN_NOEXCEPT
    {
        nn::capsrv::InitializeScreenShotControl();
        //nn::capsrv::InitializeAlbumControl();
        //nn::capsrv::InitializeAlbumAccess();
        NN_UTIL_SCOPE_EXIT{
            //nn::capsrv::FinalizeAlbumAccess();
            //nn::capsrv::FinalizeAlbumControl();
            nn::capsrv::FinalizeScreenShotControl();
        };

        nn::capsrv::ScreenShotOrientationType orientationList[] =
        {
            nn::capsrv::ScreenShotOrientation_Default,
            nn::capsrv::ScreenShotOrientation_Rotate90,
            nn::capsrv::ScreenShotOrientation_Rotate180,
            nn::capsrv::ScreenShotOrientation_Rotate270,
            0xFFFF, // invalid
        };
        int orientationCount = sizeof(orientationList) / sizeof(orientationList[0]);

        uint64_t seqNo = std::numeric_limits<uint64_t>::max();
        nn::ncm::ProgramId targetId = { 0x123456 };
        nn::applet::AppletResourceUserId aruid = { 0x1234 };
        nn::TimeSpan timeout = nn::TimeSpan::FromMilliSeconds(100);

        for(int iOri = 0; iOri < orientationCount; iOri++)
        {
            auto ori = orientationList[iOri];

            nn::capsrv::ScreenShotAttribute attr;
            attr.SetDefault();
            attr.orientation = ori;

            NN_ABORT_UNLESS_RESULT_SUCCESS(nn::capsrv::RequestTakingScreenShot(seqNo, targetId, aruid, attr, {}, timeout));

            nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));
        }
    } // NOLINT(impl/function_size)

    void RawSceenShotReadStreamTest() NN_NOEXCEPT
    {
        g_ScreenShotCount = 0;

        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::capsrv::InitializeScreenShotControl());
        NN_UTIL_SCOPE_EXIT{ nn::capsrv::FinalizeScreenShotControl(); };

        size_t imageDataSize = 0;
        int width = 0;
        int height = 0;
        NN_LOG("OpenRawScreenShotReadStream\n");
        auto result = nn::capsrv::OpenRawScreenShotReadStreamForDevelop(
            &imageDataSize,
            &width,
            &height,
            nn::vi::LayerStack_Screenshot,
            nn::TimeSpan::FromMilliSeconds(100)
        );
        if(result.IsSuccess())
        {
            NN_LOG("  -> success\n");
            NN_LOG("  size   = %llu\n", imageDataSize);
            NN_LOG("  width  = %d\n", width);
            NN_LOG("  height = %d\n", height);
        }
        else
        {
            NN_LOG("  -> failure(%d-%d)\n", result.GetModule(), result.GetDescription());
            NN_ABORT_UNLESS_RESULT_SUCCESS(result);
        }

        auto streaming = [&](size_t stepSize)
        {
            std::vector<uint8_t> data;
            data.resize(std::max(imageDataSize, stepSize));

            ptrdiff_t n = imageDataSize;
            ptrdiff_t offset = 0;
            while(n > 0)
            {
                size_t readSize = 0;
                NN_ABORT_UNLESS_RESULT_SUCCESS(nn::capsrv::ReadRawScreenShotReadStreamForDevelop(
                    &readSize,
                    data.data() + offset,
                    stepSize,
                    offset
                ));
                offset += readSize;
                n -= readSize;
                NN_LOG("reading... %lld/%lld\n", offset, imageDataSize);
            }

            char path[512] = {};
            nn::util::SNPrintf(path, sizeof(path), "tNA:/stream_%d.raw", stepSize);
            nnt::capsrv::DirectAlbumAccessor::SaveFile(data, path);
        };

        // 色々なサイズで読んでみる
        // streaming(4); // 最小読込サイズ。とにかく効率が悪いので非推奨。
        streaming(4096);
        streaming(4 * 1280);
        streaming(4 * 1280 * 720);
        streaming(6 * 1280 * 720);


        NN_LOG("CloseRawScreenShotReadStream\n");
        nn::capsrv::CloseRawScreenShotReadStreamForDevelop();

    }

    void SceenShotCancelTest() NN_NOEXCEPT
    {
        static const int timeToWaitOverlay = 1000;

        nn::capsrv::InitializeScreenShotControl();
        nn::capsrv::InitializeAlbumControl();
        nn::capsrv::InitializeAlbumAccess();
        NN_UTIL_SCOPE_EXIT{
            nn::capsrv::FinalizeAlbumAccess();
            nn::capsrv::FinalizeAlbumControl();
            nn::capsrv::FinalizeScreenShotControl();
        };

        // 何もする前のサムネイル情報を取っておく
        nn::capsrv::AlbumFileId fileId = {};
        size_t thumbSize = 0;
        std::vector<uint8_t> thumb;
        thumb.resize(nn::capsrv::ThumbnailImageDataSize_Default);
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::capsrv::GetLastOverlayScreenShotThumbnail(&fileId, &thumbSize, thumb.data(), thumb.size()));

        // 最初にキャンセル状態を初期化する（テスト起動時に AM がキャンセルを発行しているのでリセット必要）
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::capsrv::SetTakingScreenShotCancelState(0, nn::capsrv::ResultInternalScreenShotCanceledAsNoTarget()));

        auto checkThumbnailUnchanged = [&]()->void
        {
            nn::capsrv::AlbumFileId cur = {};
            NN_ABORT_UNLESS_RESULT_SUCCESS(nn::capsrv::GetLastOverlayScreenShotThumbnail(&cur, &thumbSize, thumb.data(), thumb.size()));
            EXPECT_EQ(0, std::memcmp(&fileId, &cur, sizeof(nn::capsrv::AlbumFileId)));
        };
        auto checkThumbnailChangedAndUpdate = [&]()->void
        {
            nn::capsrv::AlbumFileId cur = {};
            NN_ABORT_UNLESS_RESULT_SUCCESS(nn::capsrv::GetLastOverlayScreenShotThumbnail(&cur, &thumbSize, thumb.data(), thumb.size()));
            EXPECT_NE(0, std::memcmp(&fileId, &cur, sizeof(nn::capsrv::AlbumFileId)));
            fileId = cur;
        };

        nn::ncm::ProgramId programId = {1}; // 0 以外
        nn::applet::AppletResourceUserId aruid = {0};
        nn::capsrv::ScreenShotAttribute attribute = {};
        attribute.SetDefault();

        uint64_t seqNo = 0x123456;

        NN_LOG("Checking CancelAsNoTarget\n");
        {
            NNT_EXPECT_RESULT_SUCCESS(nn::capsrv::CancelTakingScreenShot(seqNo, nn::capsrv::ResultInternalScreenShotCanceledAsNoTarget()));
            NNT_EXPECT_RESULT_SUCCESS(nn::capsrv::RequestTakingScreenShot(seqNo, programId, aruid, attribute, {}, nn::TimeSpan::FromMilliSeconds(300)));
            nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(500));
            checkThumbnailUnchanged();
            seqNo++;
        }
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(timeToWaitOverlay)); // オーバーレイ待ち


        NN_LOG("Checking CancelAsNoTarget\n");
        {
            NNT_EXPECT_RESULT_SUCCESS(nn::capsrv::CancelTakingScreenShot(seqNo, nn::capsrv::ResultInternalScreenShotCanceledAsProhibited()));
            NNT_EXPECT_RESULT_SUCCESS(nn::capsrv::RequestTakingScreenShot(seqNo, programId, aruid, attribute, {}, nn::TimeSpan::FromMilliSeconds(300)));
            nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(500));
            checkThumbnailUnchanged();
            seqNo++;
        }
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(timeToWaitOverlay)); // オーバーレイ待ち

        NN_LOG("Checking CancelExpiredSequenceNumber\n");
        {
            auto cancel  = seqNo;
            auto request = seqNo + 1;
            NNT_EXPECT_RESULT_SUCCESS(nn::capsrv::CancelTakingScreenShot(cancel, nn::capsrv::ResultInternalScreenShotCanceledAsProhibited()));
            NNT_EXPECT_RESULT_SUCCESS(nn::capsrv::RequestTakingScreenShot(request, programId, aruid, attribute, {}, nn::TimeSpan::FromMilliSeconds(300)));
            nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(500));
            checkThumbnailChangedAndUpdate();
            seqNo += 2;
        }
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(timeToWaitOverlay)); // オーバーレイ待ち

        NN_LOG("Checking CancelFutureSequenceNumber\n");
        {
            auto cancel = seqNo + 1;
            auto request = seqNo;
            NNT_EXPECT_RESULT_SUCCESS(nn::capsrv::CancelTakingScreenShot(cancel, nn::capsrv::ResultInternalScreenShotCanceledAsProhibited()));
            NNT_EXPECT_RESULT_SUCCESS(nn::capsrv::RequestTakingScreenShot(request, programId, aruid, attribute, {}, nn::TimeSpan::FromMilliSeconds(300)));
            nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(500));
            checkThumbnailUnchanged();
            seqNo += 2;
        }
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(timeToWaitOverlay)); // オーバーレイ待ち

        NN_LOG("Checking CancelOldSequenceNumber\n");
        {
            auto oldSeqNo = seqNo;
            auto newSeqNo = seqNo + 1;
            NNT_EXPECT_RESULT_SUCCESS(nn::capsrv::CancelTakingScreenShot(newSeqNo, nn::capsrv::ResultInternalScreenShotCanceledAsProhibited()));
            NNT_EXPECT_RESULT_SUCCESS(nn::capsrv::CancelTakingScreenShot(oldSeqNo, nn::capsrv::ResultInternalScreenShotCanceledAsNoTarget()));
            NNT_EXPECT_RESULT_SUCCESS(nn::capsrv::RequestTakingScreenShot(newSeqNo, programId, aruid, attribute, {}, nn::TimeSpan::FromMilliSeconds(300)));
            nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(500));
            checkThumbnailUnchanged();
            seqNo += 2;
        }
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(timeToWaitOverlay)); // オーバーレイ待ち

        // 最後にキャンセル状態を初期値に戻しておく
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::capsrv::SetTakingScreenShotCancelState(0, nn::capsrv::ResultInternalScreenShotCanceledAsNoTarget()));
    }

}

void CapsrvThreadFunction(void*)
{
    nn::os::SleepThread(nn::TimeSpan::FromSeconds(3));

    TakingScreenShotTest();
    TakingScreenShotTestRotation();
    RawSceenShotReadStreamTest();
    SceenShotCancelTest();

    nn::os::SignalEvent(&g_TestEndEvent);

}// NOLINT(impl/function_size)
