﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "testCapsrv_Utility.h"

#include <cstdlib>
#include <nn/nn_Log.h>
#include "../../../../../Programs/Iris/Sources/Libraries/capsrv/capture/capsrv_CaptureModule.h"

namespace nnt{ namespace capsrv{

    namespace{

        nn::gfx::MemoryPool g_DummyMemoryPool;

        void* MemoryAllocate(size_t size, void*) NN_NOEXCEPT
        {
            return malloc(size);
        }
        void MemoryFree(void* p, void*) NN_NOEXCEPT
        {
            free(p);
        }

    }

    CaptureMemoryPool::CaptureMemoryPool(void* pMemory, size_t size) NN_NOEXCEPT
    {
        auto pModule = nn::capsrv::capture::GetCaptureModule();
        m_MemoryPool.Initialize(pModule, pMemory, size);
        m_Allocator.Initialize(
            MemoryAllocate,
            nullptr,
            MemoryFree,
            nullptr,
            &g_DummyMemoryPool,
            0,
            size,
            nn::gfx::util::MemoryPoolAllocator::AlignmentMax,
            true
        );
    }

    CaptureMemoryPool::~CaptureMemoryPool() NN_NOEXCEPT
    {
        m_Allocator.Finalize();
        m_MemoryPool.Finalize();
    }

    ptrdiff_t CaptureMemoryPool::Allocate(size_t size, size_t alignment) NN_NOEXCEPT
    {
        return m_Allocator.Allocate(size, alignment);
    }

    void CaptureMemoryPool::Free(ptrdiff_t offset) NN_NOEXCEPT
    {
        m_Allocator.Free(offset);
    }

    nn::capsrv::capture::MemoryPool* CaptureMemoryPool::Get() NN_NOEXCEPT
    {
        return &m_MemoryPool;
    }

    //-----------------------------------------------------------------

    CaptureImageBuffer::CaptureImageBuffer(int width, int height, nn::capsrv::capture::ImageFormat format, CaptureMemoryPool* pMemoryPool) NN_NOEXCEPT
    {
        auto pModule = nn::capsrv::capture::GetCaptureModule();
        nn::capsrv::capture::ImageBufferInfo info;
        info.width = width;
        info.height = height;
        info.format = format;
        size_t size = nn::capsrv::capture::ImageBuffer::GetRequiredMemorySize(info);
        size_t alignment = nn::capsrv::capture::ImageBuffer::GetRequiredMemoryAlignment(pModule, info);
        NN_LOG("required: size %d / alignment %d\n", static_cast<int>(size), static_cast<int>(alignment));
        ptrdiff_t offset = pMemoryPool->Allocate(size, alignment);
        m_ImageBuffer.Initialize(pModule, info, pMemoryPool->Get(), offset, size);
        m_Offset = offset;
        m_pMemoryPool = pMemoryPool;
    }

    CaptureImageBuffer::~CaptureImageBuffer() NN_NOEXCEPT
    {
        m_ImageBuffer.Finalize();
        m_pMemoryPool->Free(m_Offset);
    }

    nn::capsrv::capture::ImageBuffer* CaptureImageBuffer::Get() NN_NOEXCEPT
    {
        return &m_ImageBuffer;
    }

    //-----------------------------------------------------------------

    CaptureDisplayBuffer::CaptureDisplayBuffer(int width, int height, nn::capsrv::capture::ImageFormat format, CaptureMemoryPool* pMemoryPool) NN_NOEXCEPT
    {
        auto pModule = nn::capsrv::capture::GetCaptureModule();
        if(format == nn::capsrv::capture::ImageFormat_Rgba8)
        {
            m_ImageBufferCount = 1;
            m_ImageBufferList[0].emplace(width, height, nn::capsrv::capture::ImageFormat_Rgba8, pMemoryPool);
            m_DisplayBuffer.InitializeRgba(pModule, nn::capsrv::capture::ImageFormat_Rgba8, m_ImageBufferList[0]->Get());
        }
        else if(format == nn::capsrv::capture::ImageFormat_Yuv422_Bt601_Planar)
        {
            m_ImageBufferCount = 3;
            m_ImageBufferList[0].emplace(width, height, nn::capsrv::capture::ImageFormat_Y_Bt601, pMemoryPool);
            m_ImageBufferList[1].emplace(width / 2, height, nn::capsrv::capture::ImageFormat_U_Bt601, pMemoryPool);
            m_ImageBufferList[2].emplace(width / 2, height, nn::capsrv::capture::ImageFormat_V_Bt601, pMemoryPool);
            m_DisplayBuffer.InitializeYuvPlanar(
                pModule,
                nn::capsrv::capture::ImageFormat_Yuv422_Bt601_Planar,
                m_ImageBufferList[0]->Get(),
                m_ImageBufferList[1]->Get(),
                m_ImageBufferList[2]->Get()
            );
        }
        else if(format == nn::capsrv::capture::ImageFormat_Yuv440_Bt601_Planar)
        {
            m_ImageBufferCount = 3;
            m_ImageBufferList[0].emplace(width, height, nn::capsrv::capture::ImageFormat_Y_Bt601, pMemoryPool);
            m_ImageBufferList[1].emplace(width, height / 2, nn::capsrv::capture::ImageFormat_U_Bt601, pMemoryPool);
            m_ImageBufferList[2].emplace(width, height / 2, nn::capsrv::capture::ImageFormat_V_Bt601, pMemoryPool);
            m_DisplayBuffer.InitializeYuvPlanar(
                pModule,
                nn::capsrv::capture::ImageFormat_Yuv440_Bt601_Planar,
                m_ImageBufferList[0]->Get(),
                m_ImageBufferList[1]->Get(),
                m_ImageBufferList[2]->Get()
            );
        }
        else
        {
            NN_ABORT("unsupported format");
        }
    }

    CaptureDisplayBuffer::~CaptureDisplayBuffer() NN_NOEXCEPT
    {
        m_DisplayBuffer.Finalize();
    }

    nn::capsrv::capture::DisplayBuffer* CaptureDisplayBuffer::Get() NN_NOEXCEPT
    {
        return &m_DisplayBuffer;
    }

    //-----------------------------------------------------------------

    CaptureDisplayCapture::CaptureDisplayCapture(int width, int height, nn::capsrv::capture::ImageFormat format, nn::vi::LayerStack layerStack) NN_NOEXCEPT
    {
        auto pModule = nn::capsrv::capture::GetCaptureModule();
        m_DisplayCapture.Initialize(pModule, DefaultCaptureName, layerStack, width, height, format);
    }

    CaptureDisplayCapture::~CaptureDisplayCapture() NN_NOEXCEPT
    {
        m_DisplayCapture.Finalize();
    }

    nn::capsrv::capture::DisplayCapture* CaptureDisplayCapture::Get() NN_NOEXCEPT
    {
        return &m_DisplayCapture;
    }

}}
