﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstdlib>
#include <cstdio>

#include <nn/nn_Assert.h>
#include <nn/nn_Log.h>
#include <nn/nn_Macro.h>
#include <nn/nn_TimeSpan.h>
#include <nn/os/os_SystemEvent.h>
#include <nn/os/os_Thread.h>
#include <nn/os/os_TickTypes.h>
#include <nn/settings/system/settings_BluetoothDevices.h>

#include <nn/btm/btm.h>
#include <nn/btm/system/btm_SystemApi.h>

namespace
{
    const char ProfileStr[2][16] = {
        "NONE",
        "HID"
    };

    nn::os::SystemEventType g_DeviceInfoEvent;
} // anonymous-namespace

namespace
{

char* BdaddrToString(const nn::btm::BdAddress &bdaddr)
{
    static char strBdaddr[18];
    snprintf(strBdaddr, sizeof(strBdaddr), "%02X:%02X:%02X:%02X:%02X:%02X",
        bdaddr.address[0], bdaddr.address[1], bdaddr.address[2], bdaddr.address[3], bdaddr.address[4], bdaddr.address[5]);
    return strBdaddr;
}

}

extern "C" void nnMain()
{
    nn::btm::InitializeBtmInterface();

    // DeviceInfo 変更完了を通知するイベントを登録
    nn::btm::RegisterSystemEventForRegisteredDeviceInfo(&g_DeviceInfoEvent);

    // ペアリング済みのデバイス情報を取得します
    nn::btm::DeviceInfoList list;
    nn::btm::GetDeviceInfo(&list);

    NN_LOG("-- Paired Device Info --\n");
    for (int i = 0; i < list.deviceCount; ++i)
    {
        NN_LOG("Dev:%d\n", i);
        NN_LOG("    BD_ADDR: %s\n", BdaddrToString(list.device[i].bdAddress));
        NN_LOG("    COD: %02X %02X %02X\n", list.device[i].classOfDevice.cod[0], list.device[i].classOfDevice.cod[1], list.device[i].classOfDevice.cod[2]);
        NN_LOG("    Name: %s\n", list.device[i].bdName.name);
        NN_LOG("    Profile:%s\n", ProfileStr[list.device[i].profile]);
    }
    NN_LOG("-- ------------------- --\n");

    if (list.deviceCount > 0)
    {
        nn::btm::DeviceInfo testInfo;
        memcpy(&testInfo, &list.device[0], sizeof(nn::btm::DeviceInfo));
        strlcat(reinterpret_cast<char*>(testInfo.bdName.name), "-Dummy", nn::btm::SIZE_OF_BDNAME);

        for (int i = 0; i < 10 - list.deviceCount; ++i)
        {
            testInfo.bdAddress.address[5] += 1;
            NN_LOG("+   BD_ADDR: %s\n", BdaddrToString(testInfo.bdAddress));
            nn::btm::AddDeviceInfo(&testInfo);
            NN_SDK_ASSERT(nn::os::TimedWaitSystemEvent(&g_DeviceInfoEvent, nn::TimeSpan::FromMilliSeconds(500)));
        }

        nn::os::DestroySystemEvent(&g_DeviceInfoEvent);
    }

    nn::btm::FinalizeBtmInterface();
}

