﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstdlib>
#include <cstdio>

#include <nn/nn_Assert.h>
#include <nn/nn_Log.h>
#include <nn/nn_Result.h>

#include <nn/os.h>
#include <nn/oe.h>
#include <nn/init.h>

#include <nn/util/util_Color.h>
#include <nn/gfx/util/gfx_DebugFontTextWriter.h>

#include <nn/btm/btm.h>
#include <nn/btm/system/btm_SystemApi.h>
#include <nn/btm/debug/btm_DebugApi.h>
#include <nn/btm/btm_Result.h>

#include "../Common/testBtm_Utility.h"
#include "../Common/testBtm_UtilGraphic.h"
#include "../Common/testBtm_UtilHid.h"


class Sprite
{
public:
    static const int TARGET_COUNT = 2;
    enum Target
    {
        Target_Button = 0x00,
        Target_Stick  = 0x01,
    };
    Sprite();
    int GetX(Target target) const;
    int GetY(Target target) const;
    void SetX(Target target, int val);
    void SetY(Target target,int val);
    void AddX(Target target,int val);
    void AddY(Target target,int val);
    void FrameCount();
    int GetFps() const;
protected:
    struct Character{
        int offsetX;
        int offsetY;
        int x;
        int y;
    };
    struct Frame
    {
        int count;
        nn::os::Tick oldTick;
    };
    Character m_character[TARGET_COUNT];
    Frame m_Frame;
    double m_Fps;
};

void Sprite::FrameCount()
{
    m_Frame.count++;

    const int CALC_INTERVAL = 60;
    if(m_Frame.count == CALC_INTERVAL)
    {
        nn::os::Tick currentTick = nn::os::GetSystemTick();
        int64_t intervalMs = (currentTick - m_Frame.oldTick).ToTimeSpan().GetMilliSeconds();
        double fpms = CALC_INTERVAL / double(intervalMs);
        m_Fps = fpms * 1000;

        m_Frame.oldTick = currentTick;
        m_Frame.count = 0;
    }
}

int Sprite::GetFps() const
{
    return m_Fps;
}

Sprite::Sprite()
{
    m_character[Target_Button].offsetX = 300;
    m_character[Target_Button].offsetY = 300;

    m_character[Target_Stick].offsetX = 400;
    m_character[Target_Stick].offsetY = 400;

    for(int i=0;i<TARGET_COUNT;i++)
    {
        m_character[i].x = 0;
        m_character[i].y = 0;
    }

    m_Frame.count = 0;
    m_Frame.oldTick = nn::os::GetSystemTick();
    m_Fps = 0;
}

int Sprite::GetX(Target target) const
{
    return m_character[target].x + m_character[target].offsetX;
}

int Sprite::GetY(Target target) const
{
    return m_character[target].y + m_character[target].offsetY;
}

void Sprite::AddX(Target target, int val)
{
    m_character[target].x += val;
}

void Sprite::AddY(Target target, int val)
{
    m_character[target].y += val;
}

void Sprite::SetX(Target target, int val)
{
    m_character[target].x = val;
}

void Sprite::SetY(Target target, int val)
{
    m_character[target].y = val;
}

class Command
{
public:
    static const int STR_SIZE = 16;
    struct Name{
        char name[STR_SIZE];//NULL終端
    };

    static const int OPTION_COUNT = 4;
    struct DefineSet{
        uint8_t commandId;
        uint8_t optionCount;
        Name commandName;
        Name optionName[OPTION_COUNT];
    };

    Command();
    void    Define(DefineSet defineSet);
    void    ChangeOption(int delta);
    void    ResetOption();

    Name    GetCommandName();
    Name    GetOptionName(uint8_t index);
    Name    GetCurrentOptionName();

    uint8_t GetCommandId();
    uint8_t GetOptionCount();
    uint8_t GetCurrentOption();

protected:
    Name m_CommandName;
    Name m_OptionName[OPTION_COUNT];
    uint8_t m_CommandId;
    uint8_t m_OptionCount;
    uint8_t m_CurrentOption;
};

Command::Command()
{
    m_CurrentOption = 0;
    DefineSet defineSet = {0, 4, {"==============="},   {{"1--------------"}, {"2--------------"}, {"3--------------"}, {"4--------------"}}};
    Define(defineSet);
}

uint8_t Command::GetCommandId()
{
    return m_CommandId;
}

void Command::Define(DefineSet defineSet)
{
    m_CommandId = defineSet.commandId;
    m_OptionCount = defineSet.optionCount;
    memcpy(&m_CommandName.name[0], &defineSet.commandName.name[0], STR_SIZE);
    for(int i=0;i<OPTION_COUNT;i++)
    {
        memcpy(&m_OptionName[i].name[0], &defineSet.optionName[i].name[0], STR_SIZE);
    }
}

void Command::ChangeOption(int delta)
{
    if(delta != 1 && delta != -1)
        return;

    int temp = m_CurrentOption;
    temp += delta;

    if(temp > m_OptionCount - 1)
        m_CurrentOption = 0;
    else if(temp < 0)
        m_CurrentOption = m_OptionCount - 1;
    else
        m_CurrentOption = temp;
}

void Command::ResetOption()
{
    m_CurrentOption = 0;
}

uint8_t Command::GetCurrentOption()
{
    return m_CurrentOption;
}

uint8_t Command::GetOptionCount()
{
    return m_OptionCount;
}

Command::Name Command::GetCommandName()
{
    return m_CommandName;
}

Command::Name Command::GetOptionName(uint8_t index)
{
    NN_SDK_REQUIRES(index < m_OptionCount);
    return m_OptionName[index];
}

Command::Name Command::GetCurrentOptionName()
{
    return m_OptionName[m_CurrentOption];
}

class CommandMenu
{
public:
    static const int COMMAND_COUNT = 7;
    CommandMenu();
    void ChangeIndex(int delta);
    uint8_t  GetIndex();
    Command& GetCurrentCommandRef();
    Command  GetCurrentCommand();
    Command  GetCommand(uint8_t index);
protected:
    Command m_Command[COMMAND_COUNT];
    uint8_t m_Index;
};

CommandMenu::CommandMenu()
{
    //[Todo]初期化関数等作る
    Command::DefineSet defineSet[COMMAND_COUNT];
    defineSet[0] = {0, 4, {"WlanMode"},        {{"Local8"}, {"Local4"}, {"None"}, {"User8"}}};
    defineSet[1] = {1, 2, {"BluetoothMode"},   {{"Active"}, {"Auto"}}};
    defineSet[2] = {2, 2, {"GamepadPairing"},  {{"Enable"}, {"Disable"}}};
    defineSet[3] = {3, 2, {"SlotSaving"},      {{"Enable"}, {"Disable"}}};
    defineSet[4] = {4, 2, {"Radio"},           {{"Enable"}, {"Disable"}}};
    defineSet[5] = {5, 2, {"UpperLayer"},      {{"NfpEnable"}, {"NfpDisable"}}};
    defineSet[6] = {6, 3, {"Test"},            {{"EmulateSleep"}, {"EmulateAwake"}, {"DevicesActive"}}};

    for(int i=0;i<COMMAND_COUNT;i++)
    {
        m_Command[i].Define(defineSet[i]);
    }

    m_Index = 0;
}

void CommandMenu::ChangeIndex(int delta)
{
    if(delta != 1 && delta != -1)
        return;

    int temp = m_Index;
    temp += delta;

    if(temp > COMMAND_COUNT - 1)
        m_Index = 0;
    else if(temp < 0)
        m_Index = COMMAND_COUNT - 1;
    else
        m_Index = temp;
}

uint8_t CommandMenu::GetIndex()
{
    return m_Index;
}

Command& CommandMenu::GetCurrentCommandRef()
{
    return  m_Command[m_Index];
}

Command CommandMenu::GetCurrentCommand()
{
    return m_Command[m_Index];
}

Command CommandMenu::GetCommand(uint8_t index)
{
    if(index > COMMAND_COUNT - 1)
        return m_Command[0];
    else
        return m_Command[index];
}


class CommandHistory{
public:
    CommandHistory();
    void Add(Command command, nn::Result result);
    uint8_t GetCount();
    Command GetCommand(uint8_t index);
    nn::Result GetResult(uint8_t index);

protected:
    static const int COMMAND_COUNT = 10;
    uint8_t m_EnabledCount;
    Command m_Command[COMMAND_COUNT];
    nn::Result m_Result[COMMAND_COUNT];
};

CommandHistory::CommandHistory()
{
    m_EnabledCount = 0;
}

void CommandHistory::Add(Command command, nn::Result result)
{
    if(m_EnabledCount < COMMAND_COUNT)
        m_EnabledCount++;
    //ひとつずつ後ろずらし
    for(int i=0;i<COMMAND_COUNT - 1;i++)
    {
        m_Command[COMMAND_COUNT - 1 - i] = m_Command[COMMAND_COUNT - 2 - i];
        m_Result[COMMAND_COUNT - 1 - i] = m_Result[COMMAND_COUNT - 2 - i];
    }

    //先頭に追加
    m_Command[0] = command;
    m_Result[0] = result;
}

uint8_t CommandHistory::GetCount()
{
    return m_EnabledCount;
}

Command CommandHistory::GetCommand(uint8_t index)
{
    if(index > COMMAND_COUNT - 1)
        return m_Command[0];
    else
        return m_Command[index];
}

nn::Result CommandHistory::GetResult(uint8_t index)
{
    if(index > COMMAND_COUNT - 1)
        return m_Result[0];
    else
        return m_Result[index];
}

enum Scene{
    Scene_Setting,
    Scene_Game,
};

namespace
{
    const size_t GraphicsSystemMemorySize = 8 * 1024 * 1024;
    nnt::btm::GraphicsSystem* g_pGraphicsSystem;
    CommandMenu g_CommandMenu;
    CommandHistory g_CommandHistory;
    nn::btm::DeviceConditionList g_CdcList;
    nn::hid::NpadIdType g_CurrentNpad;
    Scene g_Scene = Scene_Setting;
    Sprite g_Sprite;

    const size_t ThreadStackSize = 8192;
    NN_OS_ALIGNAS_THREAD_STACK char g_ThreadStack[ ThreadStackSize ];
    nn::os::ThreadType  g_Thread;
    char g_ThreadName[]="DeviceCondition";
}

void DeviceConditionThread(void* arg)
{
    NN_LOG("[TestBtm_ManualDisp]DeviceConditionThread started.\n");
    nn::os::SystemEventType systemEventForCdc;
    nn::btm::RegisterSystemEventForConnectedDeviceCondition(&systemEventForCdc);

    //表示目的で一度取得しておく
    nn::btm::GetConnectedDeviceCondition(&g_CdcList);

    for(;;)
    {
        nn::os::WaitSystemEvent(&systemEventForCdc);
        nn::btm::GetConnectedDeviceCondition(&g_CdcList);
        nnt::btm::PrintDeviceConditionList(&g_CdcList);
    }
}

void DrawCommandMenu(nn::gfx::CommandBuffer* pCommandBuffer, nns::gfx::PrimitiveRenderer::Renderer* pRenderer, nn::gfx::util::DebugFontTextWriter* pWriter)
{
    pWriter->SetScale(1.3f, 1.3f);
    const float Left = 5;
    const float Top = 5;
    const float Margin = 7;

    float textPosX = Left + Margin;
    float textPosY = Top + Margin;

    pWriter->SetTextColor(nn::util::Color4u8::White());
    pWriter->SetCursor(textPosX, textPosY);

    for(int i=0;i<CommandMenu::COMMAND_COUNT;i++)
    {
        //CommandNameの描画
        pWriter->SetTextColor(nn::util::Color4u8::White());
        pWriter->Print("%s", g_CommandMenu.GetCommand(i).GetCommandName().name);

        //CommandOptionNameの描画
        for(int j=0;j<g_CommandMenu.GetCommand(i).GetOptionCount();j++)
        {
            if(g_CommandMenu.GetIndex() == i && g_CommandMenu.GetCurrentCommand().GetCurrentOption() == j)
                pWriter->SetTextColor(nn::util::Color4u8::Cyan());
            else
                pWriter->SetTextColor(nn::util::Color4u8::Gray());
            pWriter->Print("  %s", g_CommandMenu.GetCommand(i).GetOptionName(j).name);
        }
        textPosY = pWriter->GetCursorY() + Margin + pWriter->GetFontHeight();
        pWriter->SetCursor(textPosX, textPosY);
    }
}

void DrawCommandHistory(nn::gfx::CommandBuffer* pCommandBuffer, nns::gfx::PrimitiveRenderer::Renderer* pRenderer, nn::gfx::util::DebugFontTextWriter* pWriter)
{
    pWriter->SetScale(1.3f, 1.3f);
    const float Left = 5;
    const float Top = 5;
    const float Margin = 7;
    const float OffsetY = 250;

    float textPosX = Left + Margin;
    float textPosY = Top + Margin + OffsetY;

    pWriter->SetTextColor(nn::util::Color4u8::Gray());
    pWriter->SetCursor(textPosX, textPosY);

    pWriter->Print("---CommandHistory---\n");
    for(int i=0;i<g_CommandHistory.GetCount();i++)
    {
        Command command = g_CommandHistory.GetCommand(i);
        nn::Result result = g_CommandHistory.GetResult(i);
        pWriter->Print("%d.%s %s %d\n", i, command.GetCommandName().name, command.GetCurrentOptionName().name, result.GetDescription());
    }
}

void DrawDeviceConditionList(nn::gfx::CommandBuffer* pCommandBuffer, nns::gfx::PrimitiveRenderer::Renderer* pRenderer, nn::gfx::util::DebugFontTextWriter* pWriter)
{
    pWriter->SetScale(1.3f, 1.3f);
    const float Left = 5;
    const float Top = 5;
    const float Margin = 7;
    const float OffsetX = 630;

    float textPosX = Left + Margin + OffsetX;
    float textPosY = Top + Margin;

    pWriter->SetTextColor(nn::util::Color4u8::Gray());
    pWriter->SetCursor(textPosX, textPosY);

    pWriter->Print("---DeviceConditionList---\n");
    char str[32];
    str[0] = '\0';
    nnt::btm::ConvertWlanMode(&str[0], g_CdcList.wlanMode);
    pWriter->Print("%s\n", &str[0]);
    nnt::btm::ConvertBluetoothMode(&str[0], g_CdcList.bluetoothMode);
    pWriter->Print("%s\n", &str[0]);
    pWriter->Print("isSlotSavingForPairing = %d\n", g_CdcList.isSlotSavingForPairing);
    pWriter->Print("isSlotSaving = %d\n", g_CdcList.isSlotSaving);
    pWriter->Print("deviceCountCapacity = %d\n", g_CdcList.deviceCountCapacity);
    pWriter->Print("deviceCount = %d\n\n", g_CdcList.deviceCount);

    pWriter->SetCursor(textPosX, textPosY + 190);
    for(int i=0;i<g_CdcList.deviceCount;i++)
    {
        pWriter->Print("%s\n", &g_CdcList.device[i].bdName);
        nnt::btm::ConvertBdAddress(&str[0], &g_CdcList.device[i].bdAddress);
        pWriter->Print("%s\n", &str[0]);
        nnt::btm::ConvertSniffMode(&str[0], g_CdcList.device[i].hidDeviceCondition.sniffMode);
        pWriter->Print("%s %d\n", &str[0],  g_CdcList.device[i].hidDeviceCondition.isLargeSlotRequired);
        nnt::btm::ConvertSlotMode(&str[0], g_CdcList.device[i].hidDeviceCondition.slotMode);
        pWriter->Print("%s\n\n", &str[0]);

        if(i == 3)
            pWriter->SetCursor(textPosX + 240, textPosY + 190);
    }
}

void DrawFps(nn::gfx::CommandBuffer* pCommandBuffer, nns::gfx::PrimitiveRenderer::Renderer* pRenderer, nn::gfx::util::DebugFontTextWriter* pWriter)
{
    pWriter->SetScale(1.3f, 1.3f);
    pWriter->SetCursor(1100, 670);
    pWriter->SetTextColor(nn::util::Color4u8::Gray());
    pWriter->Print("%4f FPS", g_Sprite.GetFps());
}

void DrawSprite(nn::gfx::CommandBuffer* pCommandBuffer, nns::gfx::PrimitiveRenderer::Renderer* pRenderer, nn::gfx::util::DebugFontTextWriter* pWriter)
{
    pWriter->SetScale(1.3f, 1.3f);

    pWriter->SetTextColor(nn::util::Color4u8::Gray());
    pWriter->SetCursor(10, g_Sprite.GetY(Sprite::Target_Button) - 10);
    pWriter->Print("|_|_|_|_|_|_|_|_|_|_|");
    pWriter->SetCursor(g_Sprite.GetX(Sprite::Target_Button), g_Sprite.GetY(Sprite::Target_Button));
    pWriter->SetTextColor(nn::util::Color4u8::Cyan());
    pWriter->Print("B");

    pWriter->SetTextColor(nn::util::Color4u8::Cyan());
    pWriter->SetCursor(g_Sprite.GetX(Sprite::Target_Stick), g_Sprite.GetY(Sprite::Target_Stick));
    pWriter->Print("+");
}

void MakeCommandFunc(nns::gfx::GraphicsFramework* pGraphicsFramework, int bufferIndex, void* pUserData)
{
    auto& gfxsys = *reinterpret_cast<nnt::btm::GraphicsSystem*>(pUserData);

    auto* renderer = gfxsys.GetPrimitiveRenderer();
    auto* writer = gfxsys.GetDebugFontTextWriter();

    renderer->Update(bufferIndex);

    pGraphicsFramework->BeginFrame(bufferIndex);
    {
        nn::gfx::CommandBuffer* rootCommandBuffer = pGraphicsFramework->GetRootCommandBuffer(bufferIndex);

        nn::gfx::ColorTargetView* target = pGraphicsFramework->GetColorTargetView();
        rootCommandBuffer->ClearColor(target,
            nn::util::Color4u8::Black().GetR(),
            nn::util::Color4u8::Black().GetG(),
            nn::util::Color4u8::Black().GetB(),
            nn::util::Color4u8::Black().GetA(), 0);
        rootCommandBuffer->SetRenderTargets(1, &target, nullptr);
        rootCommandBuffer->SetViewportScissorState(pGraphicsFramework->GetViewportScissorState());

        renderer->SetDefaultParameters();

        nn::util::Matrix4x3fType viewMatrix;
        nn::util::Matrix4x4fType projectionMatrix;
        nn::util::Matrix4x3f modelMatrix;

        nn::util::MatrixIdentity(&viewMatrix);
        nn::util::MatrixIdentity(&projectionMatrix);
        nn::util::MatrixIdentity(&modelMatrix);
        renderer->SetViewMatrix(&viewMatrix);
        renderer->SetProjectionMatrix(&projectionMatrix);
        renderer->SetModelMatrix(&modelMatrix);

        if(g_Scene == Scene_Setting)
        {
            DrawCommandMenu(rootCommandBuffer, renderer, writer);
            DrawCommandHistory(rootCommandBuffer, renderer, writer);
            DrawDeviceConditionList(rootCommandBuffer, renderer, writer);
            DrawFps(rootCommandBuffer, renderer, writer);
        }
        else if(g_Scene == Scene_Game)
        {
            DrawSprite(rootCommandBuffer, renderer, writer);
            DrawFps(rootCommandBuffer, renderer, writer);
        }

        writer->Draw(rootCommandBuffer);
    }
    pGraphicsFramework->EndFrame(bufferIndex, true);
} // NOLINT(impl/function_size)

bool ExitApplicationRequest()
{
    nn::oe::Message message;
    if (nn::oe::TryPopNotificationMessage(&message))
    {
        switch(message)
        {
        // フォーカス状態変更通知
        case nn::oe::MessageFocusStateChanged:
            {
                auto state = nn::oe::GetCurrentFocusState();
                //  HOME メニューから復帰した
                if (state == nn::oe::FocusState_InFocus)
                {
                }
                break;
            }
        case nn::oe::MessageExitRequest:
            {
                NN_LOG("Received MessageExitRequest\n");
                return true;
            }
        // 動作モード（携帯／据置）が変更
        case nn::oe::MessageOperationModeChanged:
            break;
        // 性能モード（ノーマル／ブースト）が変更
        case nn::oe::MessagePerformanceModeChanged:
            break;
        // 未定義メッセージ
        default:
            {
                NN_LOG("Received unknown message= 0x%08x", message);
            }
            break;
        }
    }
    return false;
}

nn::Result ExecuteCommand()
{
    Command command = g_CommandMenu.GetCurrentCommand();
    nn::Result result = nn::ResultSuccess();

    //[Todo]enum化
    switch(command.GetCommandId())
    {
    case 0://WlanMode
    {
        if(command.GetCurrentOption() == 0)
            result = nn::btm::SetWlanMode(nn::btm::WlanMode_Local8);
        else if(command.GetCurrentOption() == 1)
            result = nn::btm::SetWlanMode(nn::btm::WlanMode_Local4);
        else if(command.GetCurrentOption() == 2)
            result = nn::btm::SetWlanMode(nn::btm::WlanMode_None);
        else if(command.GetCurrentOption() == 3)
            result = nn::btm::SetWlanMode(nn::btm::WlanMode_User8);
        else
            NN_ABORT("[testBtm_ManualDisp]\n");
        break;
    }
    case 1://BluetoothMode
    {
        if(command.GetCurrentOption() == 0)
            result = nn::btm::SetBluetoothMode(nn::btm::BluetoothMode_Active);
        else if(command.GetCurrentOption() == 1)
            result = nn::btm::SetBluetoothMode(nn::btm::BluetoothMode_Auto);
        else
            NN_ABORT("[testBtm_ManualDisp]\n");
        break;
    }
    case 2://GamepadPairing
    {
        if(command.GetCurrentOption() == 0)
            result = nn::btm::system::StartGamepadPairing();
        else if(command.GetCurrentOption() == 1)
            nn::btm::system::CancelGamepadPairing();
        else
            NN_ABORT("[testBtm_ManualDisp]\n");
        break;
    }
    case 3://SlotSaving
    {
        if(command.GetCurrentOption() == 0)
        {
            nn::oe::SetWirelessPriorityMode(nn::oe::WirelessPriorityMode_OptimizedForWlan);
            //nn::btm::EnableSlotSaving(true);//oe, amを経由させない場合はこちらに切り替え
        }
        else if(command.GetCurrentOption() == 1)
        {
            nn::oe::SetWirelessPriorityMode(nn::oe::WirelessPriorityMode_Default);
            //nn::btm::EnableSlotSaving(false);//oe, amを経由させない場合はこちらに切り替え
        }
        else
        {
            NN_ABORT("[testBtm_ManualDisp]\n");
        }
        break;
    }
    case 4://Radio
    {
        if(command.GetCurrentOption() == 0)
            nn::btm::system::EnableRadio(true);
        else if(command.GetCurrentOption() == 1)
            nn::btm::system::EnableRadio(false);
        else
            NN_ABORT("[testBtm_ManualDisp]\n");
        break;
    }
    case 5://UpperLayer
    {
        if(command.GetCurrentOption() == 0)
            result = nnt::btm::StartNfpScan(g_CurrentNpad);
        else if(command.GetCurrentOption() == 1)
            result = nnt::btm::StopNfpScan(g_CurrentNpad);
        else
            NN_ABORT("[testBtm_ManualDisp]\n");
        break;
    }
    case 6://Test
    {
        if(command.GetCurrentOption() == 0)
            nn::btm::debug::GeneralTest(7);
        else if(command.GetCurrentOption() == 1)
            nn::btm::debug::GeneralTest(8);
        else if(command.GetCurrentOption() == 2)
        {
            nn::btm::DeviceConditionList dcl;
            nn::btm::GetConnectedDeviceCondition(&dcl);
            nn::btm::DeviceSlotModeList dsl;

            dsl.deviceCount = dcl.deviceCount;
            for(int i=0;i<dsl.deviceCount;i++)
            {
                dsl.device[i].bdAddress = dcl.device[i].bdAddress;
                dsl.device[i].slotMode = nn::btm::SlotMode_Active;
            }
            for(;;)
            {
                result = nn::btm::SetSlotMode(&dsl);
                if(nn::btm::ResultBusy::Includes(result))
                    ::nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(10));
                else
                    break;
            }
            break;
        }
        else
            NN_ABORT("[testBtm_ManualDisp]\n");
        break;
    }
    default:
    {
        break;
    }
    }

    return result;
}//NOLINT(impl/function_size)

void Update()
{
    static nnt::btm::HidInput hidInput;

    hidInput = nnt::btm::HidUpdate();
    g_CurrentNpad = hidInput.npadIdType;
    g_Sprite.FrameCount();

    if(g_Scene == Scene_Setting)
    {
        if(hidInput.hidButton == nnt::btm::HidButton_Up)
        {
            g_CommandMenu.ChangeIndex(-1);
            g_CommandMenu.GetCurrentCommandRef().ResetOption();
        }
        else if(hidInput.hidButton == nnt::btm::HidButton_Down)
        {
            g_CommandMenu.ChangeIndex(1);
            g_CommandMenu.GetCurrentCommandRef().ResetOption();
        }
        else if(hidInput.hidButton == nnt::btm::HidButton_Left)
        {
            g_CommandMenu.GetCurrentCommandRef().ChangeOption(-1);
        }
        else if(hidInput.hidButton == nnt::btm::HidButton_Right)
        {
            g_CommandMenu.GetCurrentCommandRef().ChangeOption(1);
        }
        else if(hidInput.hidButton == nnt::btm::HidButton_A)
        {
            auto result = ExecuteCommand();
            g_CommandHistory.Add(g_CommandMenu.GetCurrentCommand(), result);
        }
        else if(hidInput.hidButton == nnt::btm::HidButton_R)
        {
            g_Scene = Scene_Game;
        }
    }
    else if(g_Scene == Scene_Game)
    {
        if(hidInput.isUpdated)
        {
            if(hidInput.npadIdType == nn::hid::NpadId::Handheld)
            {
                if(hidInput.handheldState.buttons.Test<nn::hid::NpadJoyButton::Right>() == true)
                {
                    g_Sprite.AddX(Sprite::Target_Button, 10);
                }
                if(hidInput.handheldState.buttons.Test<nn::hid::NpadJoyButton::Left>() == true)
                {
                    g_Sprite.AddX(Sprite::Target_Button, -10);
                }
                g_Sprite.SetX(Sprite::Target_Stick, hidInput.handheldState.analogStickL.x / 300);//生値は大きすぎてディスプレイからはみ出すので、適宜縮小
                g_Sprite.SetY(Sprite::Target_Stick, -hidInput.handheldState.analogStickL.y / 300);//生値は大きすぎてディスプレイからはみ出すので、適宜縮小
            }
            else
            {
                if(hidInput.joyDualState.buttons.Test<nn::hid::NpadButton::Right>() == true)
                {
                    g_Sprite.AddX(Sprite::Target_Button, 10);
                }
                if(hidInput.joyDualState.buttons.Test<nn::hid::NpadJoyButton::Left>() == true)
                {
                    g_Sprite.AddX(Sprite::Target_Button, -10);
                }
                g_Sprite.SetX(Sprite::Target_Stick, hidInput.joyDualState.analogStickL.x / 300);//生値は大きすぎてディスプレイからはみ出すので、適宜縮小
                g_Sprite.SetY(Sprite::Target_Stick, -hidInput.joyDualState.analogStickL.y / 300);//生値は大きすぎてディスプレイからはみ出すので、適宜縮小
            }
        }
        if(hidInput.hidButton == nnt::btm::HidButton_L)
        {
            g_Scene = Scene_Setting;
        }
    }
}

extern "C" void nnMain()
{
    nn::oe::Initialize();
    nn::oe::SetFocusHandlingMode(nn::oe::FocusHandlingMode_Notify);
    nn::oe::EnterExitRequestHandlingSection();

    // フレームワーク初期化
    g_pGraphicsSystem = new nnt::btm::GraphicsSystem();
    g_pGraphicsSystem->Initialize(GraphicsSystemMemorySize, MakeCommandFunc);

    // デバッグフォント初期化
    g_pGraphicsSystem->InitializeDebugFont();

    // プリミティブレンダラ初期化
    g_pGraphicsSystem->InitializePrimitiveRenderer();

    nnt::btm::HidInit();
    nn::btm::InitializeBtmInterface();
    nn::btm::debug::InitializeBtmDebugInterface();

    //DeviceCondition監視スレッド
    auto result = nn::os::CreateThread( &g_Thread, DeviceConditionThread, g_ThreadName, g_ThreadStack, ThreadStackSize, nn::os::DefaultThreadPriority );
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);
    nn::os::StartThread( &g_Thread );

    for (int frame = 0; !ExitApplicationRequest(); ++frame)
    {
        Update();
        g_pGraphicsSystem->Update();
    }

    //[Todo]Thread等々終了処理

    nn::btm::FinalizeBtmInterface();

    // デバッグフォント終了
    g_pGraphicsSystem->FinalizeDebugFont();

    // プリミティブレンダラ終了
    g_pGraphicsSystem->FinalizePrimitiveRenderer();

    // フレームワーク終了
    g_pGraphicsSystem->Finalize();
    delete g_pGraphicsSystem;

    nn::oe::LeaveExitRequestHandlingSection();
}
