﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <cstdio>

#include <nn/os.h>
#include <nn/nn_Log.h>

#include <nn/btm/btm.h>
#include <nn/btm/debug/btm_DebugApi.h>

namespace nnt { namespace btm {
//------------------------------
//　テスト用のユーティリティ
//------------------------------
//pBuf >= 13byte
void ConvertBdAddress(char* pBuf, const nn::btm::BdAddress* pBdAddress)
{
    sprintf(pBuf, "%02X%02X%02X%02X%02X%02X",pBdAddress->address[0],pBdAddress->address[1],pBdAddress->address[2],pBdAddress->address[3],pBdAddress->address[4],pBdAddress->address[5]);
}

//pBuf >= 32byte
void ConvertBluetoothMode(char* pBuf, nn::btm::BluetoothMode bluetoothMode)
{
    char enumString[2][32] =
    {
            "BluetoothMode_Active", "BluetoothMode_Auto",
    };

    sprintf(pBuf, "%s", &enumString[bluetoothMode][0]);
}

//pBuf >= 32byte
void ConvertWlanMode(char* pBuf, nn::btm::WlanMode wlanMode)
{
    char enumString[4][32] =
    {
            "WlanMode_Local4", "WlanMode_Local8", "WlanMode_None", "WlanMode_User8",
    };

    sprintf(pBuf, "%s", &enumString[wlanMode][0]);
}

//pBuf >= 32byte
void ConvertSniffMode(char* pBuf, nn::btm::SniffMode sniffMode)
{
    char enumString[4][32] =
    {
            "SniffMode_5ms", "SniffMode_10ms", "SniffMode_15ms", "SniffMode_Active",
    };

    sprintf(pBuf, "%s", &enumString[sniffMode][0]);
}

//pBuf >= 32byte
void ConvertSlotMode(char* pBuf, nn::btm::SlotMode slotMode)
{
    char enumString[4][32] =
    {
            "SlotMode_2", "SlotMode_4", "SlotMode_6", "SlotMode_Active",
    };

    sprintf(pBuf, "%s", &enumString[slotMode][0]);
}

void PrintByteDataAsHex(const void* pBuffer, uint16_t length)
{
    const uint8_t* buffer = reinterpret_cast<const uint8_t*>(pBuffer);
    uint8_t byteCount=0;
    NN_UNUSED( buffer );

    for(uint16_t i=0;i<length;i++)
    {
        NN_LOG("%02X ",buffer[i]);
        byteCount++;
        if(byteCount==16)
        {
            NN_LOG("\n");
            byteCount=0;
        }
    }
    if(byteCount!=0)
    {
        NN_LOG("\n");
    }
}

bool IsSameBdAddress(const nn::btm::BdAddress* pBdAddressAlpha, const nn::btm::BdAddress* pBdAddressBeta)
{
    for(int i=0;i<nn::btm::SIZE_OF_BDADDRESS;i++)
    {
        if(pBdAddressAlpha->address[i] != pBdAddressBeta->address[i])
        {
            return false;
        }
    }
    return true;
}

void PrintBdAddress(nn::btm::BdAddress bdAddress)
{
    char buf[32];
    ConvertBdAddress(&buf[0], &bdAddress);
    NN_LOG("%s\n", &buf[0]);
}

void PrintDevicePropertyList(const nn::btm::DevicePropertyList* pDevicePropertyList)
{
    NN_LOG("DevicePropertyList----------\n");
    NN_LOG("Device count = %d\n", pDevicePropertyList->deviceCount);
    for(int i=0;i<pDevicePropertyList->deviceCount;i++)
    {
        PrintBdAddress(pDevicePropertyList->device[i].bdAddress);
        NN_LOG("%s\n",pDevicePropertyList->device[i].bdName.name);
        NN_LOG("Class of device ");
        PrintByteDataAsHex(&pDevicePropertyList->device[i].classOfDevice.cod[0], nn::btm::SIZE_OF_COD);
    }
}

void PrintBluetoothMode(nn::btm::BluetoothMode bluetoothMode)
{
    char buf[32];
    ConvertBluetoothMode(&buf[0], bluetoothMode);
    NN_LOG("%s\n", &buf[0]);
}

void PrintWlanMode(nn::btm::WlanMode wlanMode)
{
    char buf[32];
    ConvertWlanMode(&buf[0], wlanMode);
    NN_LOG("%s\n", &buf[0]);
}

void PrintSniffMode(nn::btm::SniffMode sniffMode)
{
    char buf[32];
    ConvertSniffMode(&buf[0], sniffMode);
    NN_LOG("%s\n", &buf[0]);
}

void PrintSlotMode(nn::btm::SlotMode slotMode)
{
    char buf[32];
    ConvertSlotMode(&buf[0], slotMode);
    NN_LOG("%s\n", &buf[0]);
}

void PrintDeviceConditionList(const nn::btm::DeviceConditionList* pDeviceConditionList)
{
    NN_LOG("DeviceConditionList==========\n");
    NN_LOG("Bluetooth mode   = ");PrintBluetoothMode(pDeviceConditionList->bluetoothMode);
    NN_LOG("WLAN mode        = ");PrintWlanMode(pDeviceConditionList->wlanMode);
    NN_LOG("isSlotSavingForP = %d\n",pDeviceConditionList->isSlotSavingForPairing);
    NN_LOG("Device count     = %d\n",pDeviceConditionList->deviceCount);
    NN_LOG("Device count cap = %d\n",pDeviceConditionList->deviceCountCapacity);
    for(int i=0;i<pDeviceConditionList->deviceCount;i++)
    {
        PrintBdAddress(pDeviceConditionList->device[i].bdAddress);
        NN_LOG("%s\n",pDeviceConditionList->device[i].bdName.name);
        //NN_LOG("Profile      = enum(%d)\n",pDeviceConditionList->device[i].profile);
        //NN_LOG("IsBurst      = %d\n",pDeviceConditionList->device[i].hidDeviceCondition.isBurstMode);
        //NN_LOG("ZeroRtCount  = %d\n",pDeviceConditionList->device[i].hidDeviceCondition.zeroRetransmissionList.enabledReportIdCount);
        NN_LOG("SniffMode    = ");PrintSniffMode(pDeviceConditionList->device[i].hidDeviceCondition.sniffMode);
        NN_LOG("SlotMode     = ");PrintSlotMode(pDeviceConditionList->device[i].hidDeviceCondition.slotMode);
        //NN_LOG("Vid          = %d\n",pDeviceConditionList->device[i].hidDeviceCondition.vid);
        //NN_LOG("Pid          = %d\n",pDeviceConditionList->device[i].hidDeviceCondition.pid);
    }
}

void PrintDeviceInfoList(const nn::btm::DeviceInfoList* pDeviceInfoList)
{
    NN_LOG("DeviceInfoList**********\n");
    for(int i=0;i<pDeviceInfoList->deviceCount;i++)
    {
        NN_LOG("DeviceInfoList[%d]\n",i);
        PrintBdAddress(pDeviceInfoList->device[i].bdAddress);
        NN_LOG("%s\n",pDeviceInfoList->device[i].bdName);
        NN_LOG("Class of device = ");
        PrintByteDataAsHex(&pDeviceInfoList->device[i].classOfDevice.cod[0], nn::btm::SIZE_OF_COD);
        NN_LOG("Link key = ");
        PrintByteDataAsHex(&pDeviceInfoList->device[i].linkKey, nn::btm::SIZE_OF_LINKKEY);
        NN_LOG("Profile = enum(%d)\n",pDeviceInfoList->device[i].profile);
        NN_LOG("Vid     = %d\n",pDeviceInfoList->device[i].hidDeviceInfo.vid);
        NN_LOG("Pid     = %d\n",pDeviceInfoList->device[i].hidDeviceInfo.pid);
    }
}
}}
