﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <vector>
#include <nn/nn_Common.h>
#include <nn/nn_Log.h>
#include <nn/nn_Abort.h>
#include <nnt.h>
#include <nnt/nntest.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/os.h>
#include <nn/bpc/bpc_RtcTypes.h>
#include <nn/bpc/bpc_WakeupConfig.h>
#include <nn/bpc/bpc_WakeupConfigServer.h>
#include <nn/bpc/bpc_WakeupConfigSystem.h>
#include <nn/i2c/i2c.h>

struct AlarmParam
{
    int64_t                   secondsFromNow; // no matter if absolute or relative
    nn::bpc::WakeupTimerType  type;
};

struct AlarmRegistrationTestParam
{
    const AlarmParam*           pAlarmParamList;
    int                         alarmCount;
    bool                        expectedIsEnabled;
    int                         expectedEnabledAlarmParamIndex;
};

TEST(BpcCleanAllWakeupTimersTest, CleanAllWakeupTimers)
{
    nn::bpc::InitializeWakeupConfig();
    NN_UTIL_SCOPE_EXIT{
        nn::bpc::CleanAllWakeupTimers(nn::bpc::WakeupTimerType_BackgroundTask);
        nn::bpc::CleanAllWakeupTimers(nn::bpc::WakeupTimerType_FullWakeup);
        nn::bpc::FinalizeWakeupConfig();
    };

    // タイマーを確保し、テストパラメータに沿ってセットアップ
    const AlarmParam AlarmParamList[] =
    {
        { 4LL, nn::bpc::WakeupTimerType_FullWakeup },
        { 6LL, nn::bpc::WakeupTimerType_BackgroundTask },
        { 2LL, nn::bpc::WakeupTimerType_FullWakeup },
        { 7LL, nn::bpc::WakeupTimerType_BackgroundTask },
        { 3LL, nn::bpc::WakeupTimerType_FullWakeup },
        { 8LL, nn::bpc::WakeupTimerType_BackgroundTask },
        { 3LL, nn::bpc::WakeupTimerType_BackgroundTask },
        { 9LL, nn::bpc::WakeupTimerType_BackgroundTask },
    };
    const int ExpectedEnabledAlarmParamIndex = 2;
    const int ExpectedEnabledAlarmParamAfterClearingFullWakeupIndex = 6;

    // タイマーを確保し、テストパラメータに沿ってセットアップ
    std::vector<int> timerHandleList(NN_ARRAY_SIZE(AlarmParamList));
    for ( int i = 0; i < NN_ARRAY_SIZE(AlarmParamList); ++i )
    {
        NNT_ASSERT_RESULT_SUCCESS(nn::bpc::CreateWakeupTimer(
            &timerHandleList[i],
            nn::TimeSpan::FromSeconds(AlarmParamList[i].secondsFromNow),
            AlarmParamList[i].type
        ));
    }

    // RTC アラームを更新する
    bool actualIsEnabled;
    int enabledTimerHandle;
    nn::bpc::WakeupTimerType enabledTimerType;
    NNT_EXPECT_RESULT_SUCCESS(nn::bpc::EnableWakeupTimerOnDevice(&actualIsEnabled, &enabledTimerHandle, &enabledTimerType));
    EXPECT_TRUE(actualIsEnabled);
    EXPECT_EQ(timerHandleList[ExpectedEnabledAlarmParamIndex], enabledTimerHandle);
    EXPECT_EQ(AlarmParamList[ExpectedEnabledAlarmParamIndex].type, enabledTimerType);
    EXPECT_EQ(AlarmParamList[ExpectedEnabledAlarmParamIndex].type, nn::bpc::GetLastEnabledWakeupTimerType());

    // 指定したタイプのみクリア
    NNT_EXPECT_RESULT_SUCCESS(nn::bpc::CleanAllWakeupTimers(nn::bpc::WakeupTimerType_FullWakeup));

    // RTC アラームを更新すると BG タスク側のアラームが残って有効になる
    NNT_EXPECT_RESULT_SUCCESS(nn::bpc::EnableWakeupTimerOnDevice(&actualIsEnabled, &enabledTimerHandle, &enabledTimerType));
    EXPECT_TRUE(actualIsEnabled);
    EXPECT_EQ(timerHandleList[ExpectedEnabledAlarmParamAfterClearingFullWakeupIndex], enabledTimerHandle);
    EXPECT_EQ(AlarmParamList[ExpectedEnabledAlarmParamAfterClearingFullWakeupIndex].type, enabledTimerType);
    EXPECT_EQ(AlarmParamList[ExpectedEnabledAlarmParamAfterClearingFullWakeupIndex].type, nn::bpc::GetLastEnabledWakeupTimerType());

    // 残ったタイプもクリア
    NNT_EXPECT_RESULT_SUCCESS(nn::bpc::CleanAllWakeupTimers(nn::bpc::WakeupTimerType_BackgroundTask));

    // RTC アラームを再度更新すると、無効になる（リストが空になっている）
    NNT_EXPECT_RESULT_SUCCESS(nn::bpc::EnableWakeupTimerOnDevice(&actualIsEnabled, &enabledTimerHandle, &enabledTimerType));
    EXPECT_FALSE(actualIsEnabled);
}

class BpcWakeupTimerRegistrationTest : public ::testing::TestWithParam<AlarmRegistrationTestParam>
{};

TEST_P(BpcWakeupTimerRegistrationTest, AlarmRegistration)
{
    nn::bpc::InitializeWakeupConfig();
    NN_UTIL_SCOPE_EXIT{
        nn::bpc::CleanAllWakeupTimers(nn::bpc::WakeupTimerType_BackgroundTask);
        nn::bpc::CleanAllWakeupTimers(nn::bpc::WakeupTimerType_FullWakeup);
        nn::bpc::FinalizeWakeupConfig();
    };

    // タイマーを確保し、テストパラメータに沿ってセットアップ
    std::vector<int> timerHandleList(GetParam().alarmCount);
    for (int i = 0; i < GetParam().alarmCount; ++i)
    {
        NNT_ASSERT_RESULT_SUCCESS(nn::bpc::CreateWakeupTimer(
            &timerHandleList[i],
            nn::TimeSpan::FromSeconds(GetParam().pAlarmParamList[i].secondsFromNow),
            GetParam().pAlarmParamList[i].type
            ));
    }

    // RTC アラームを更新する
    bool actualIsEnabled;
    int enabledTimerHandle;
    nn::bpc::WakeupTimerType enabledTimerType;
    NNT_EXPECT_RESULT_SUCCESS(nn::bpc::EnableWakeupTimerOnDevice(&actualIsEnabled, &enabledTimerHandle, &enabledTimerType));
    EXPECT_EQ(GetParam().expectedIsEnabled, actualIsEnabled);
    if ( GetParam().expectedIsEnabled )
    {
        EXPECT_EQ(timerHandleList[GetParam().expectedEnabledAlarmParamIndex], enabledTimerHandle);
        EXPECT_EQ(GetParam().pAlarmParamList[GetParam().expectedEnabledAlarmParamIndex].type, enabledTimerType);
        EXPECT_EQ(GetParam().pAlarmParamList[GetParam().expectedEnabledAlarmParamIndex].type, nn::bpc::GetLastEnabledWakeupTimerType());
    }

    // 全タイマーをキャンセル
    for (auto& timerHandle : timerHandleList)
    {
        NNT_EXPECT_RESULT_SUCCESS(nn::bpc::CancelWakeupTimer(timerHandle));
    }

    // RTC アラームを再度更新すると、無効になる
    NNT_EXPECT_RESULT_SUCCESS(nn::bpc::EnableWakeupTimerOnDevice(&actualIsEnabled, &enabledTimerHandle, &enabledTimerType));
    EXPECT_FALSE(actualIsEnabled);
}

const AlarmParam AlarmParamNxSetNormal1[] =
{
    { 8LL, nn::bpc::WakeupTimerType_BackgroundTask },
    { 5LL, nn::bpc::WakeupTimerType_BackgroundTask },
    { 3LL, nn::bpc::WakeupTimerType_BackgroundTask },
};
const AlarmParam AlarmParamNxSetNormal2[] =
{
    { 10000000LL, nn::bpc::WakeupTimerType_BackgroundTask },
    { 100000LL, nn::bpc::WakeupTimerType_BackgroundTask },
    { 10000LL, nn::bpc::WakeupTimerType_BackgroundTask },
};
const AlarmParam AlarmParamNxSetNormal3[] =
{
    { 10000000LL, nn::bpc::WakeupTimerType_BackgroundTask },
    { 11000000LL, nn::bpc::WakeupTimerType_BackgroundTask },
    { 20000000LL, nn::bpc::WakeupTimerType_BackgroundTask },
    { 20000000LL, nn::bpc::WakeupTimerType_BackgroundTask },
    { 40000000LL, nn::bpc::WakeupTimerType_BackgroundTask },
    { nn::bpc::MaxRtcTimeInSeconds - 1LL, nn::bpc::WakeupTimerType_BackgroundTask },
};
const AlarmParam AlarmParamNxSetTypeMixed1[] =
{
    { 3LL, nn::bpc::WakeupTimerType_FullWakeup },
    { 2LL, nn::bpc::WakeupTimerType_BackgroundTask },
    { 3LL, nn::bpc::WakeupTimerType_BackgroundTask },
    { 4LL, nn::bpc::WakeupTimerType_FullWakeup },
    { 5LL, nn::bpc::WakeupTimerType_BackgroundTask },
};
const AlarmParam AlarmParamNxSetTypeMixed2[] =
{
    { 4LL, nn::bpc::WakeupTimerType_FullWakeup },
    { 2LL, nn::bpc::WakeupTimerType_BackgroundTask },
    { 2LL, nn::bpc::WakeupTimerType_FullWakeup },
    { 3LL, nn::bpc::WakeupTimerType_BackgroundTask },
    { 3LL, nn::bpc::WakeupTimerType_FullWakeup },
    { 2LL, nn::bpc::WakeupTimerType_BackgroundTask },
    { 3LL, nn::bpc::WakeupTimerType_BackgroundTask },
    { 2LL, nn::bpc::WakeupTimerType_BackgroundTask },
};
const AlarmParam AlarmParamNxSetShortest[] =
{
    { 50LL, nn::bpc::WakeupTimerType_BackgroundTask },
    { 50000LL, nn::bpc::WakeupTimerType_BackgroundTask },
    { 2LL, nn::bpc::WakeupTimerType_BackgroundTask },
};
const AlarmParam AlarmParamNxSetLongest[] =
{
    { nn::bpc::MaxRtcTimeInSeconds - 1LL, nn::bpc::WakeupTimerType_FullWakeup },
    { nn::bpc::MaxRtcTimeInSeconds - 1LL, nn::bpc::WakeupTimerType_BackgroundTask },
};

const AlarmRegistrationTestParam AlarmRegistrationTestParamNxSet[] =
{
    { AlarmParamNxSetNormal1, NN_ARRAY_SIZE(AlarmParamNxSetNormal1), true, 2 },
    { AlarmParamNxSetNormal2, NN_ARRAY_SIZE(AlarmParamNxSetNormal2), true, 2 },
    { AlarmParamNxSetNormal3, NN_ARRAY_SIZE(AlarmParamNxSetNormal3), true, 0 },
    { AlarmParamNxSetTypeMixed1, NN_ARRAY_SIZE(AlarmParamNxSetTypeMixed1), true, 1 },
    { AlarmParamNxSetTypeMixed2, NN_ARRAY_SIZE(AlarmParamNxSetTypeMixed2), true, 2 },
    { AlarmParamNxSetShortest, NN_ARRAY_SIZE(AlarmParamNxSetShortest), true, 2 },
    { AlarmParamNxSetLongest, NN_ARRAY_SIZE(AlarmParamNxSetLongest), true, 0 },
};

INSTANTIATE_TEST_CASE_P(
    BpcWakeupTimerRegistrationTestNx,
    BpcWakeupTimerRegistrationTest,
    ::testing::ValuesIn(AlarmRegistrationTestParamNxSet)
);
