﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Abort.h>
#include <nn/nn_Log.h>
#include <nn/nn_TimeSpan.h>

#include <nn/os.h>

#include <nn/i2c/i2c.h>
#include <nn/bpc/bpc_ResultPrivate.h>
#include <nn/result/result_HandlingUtility.h>

#include <nnt/gtest/gtest.h>

namespace nnt { namespace bpc {

class PtcModeTest : public ::testing::Test
{
    virtual void SetUp() NN_OVERRIDE
    {
        nn::i2c::Initialize();
    }

    virtual void TearDown() NN_OVERRIDE
    {
        nn::i2c::Finalize();
    }
};

nn::Result ReadRtcDoubleBufferredRegister(uint8_t* pOutValue, nn::i2c::I2cSession& session, const uint8_t* pAddress) NN_NOEXCEPT
{
    const uint8_t RtcUpdate0Address = 0x04;
    const uint8_t RbudrMask = 0x10;
    const int MaxRetryCount = 4;

    NN_RESULT_DO(nn::i2c::WriteSingleRegister(session, &RtcUpdate0Address, &RbudrMask));

    int count = 0;
    uint8_t value = 0x00;
    do
    {
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(16));
        NN_RESULT_DO(nn::i2c::ReadSingleRegister(&value, session, &RtcUpdate0Address));
        count++;
    }
    while ( ((value & RbudrMask) == RbudrMask) && count < MaxRetryCount );

    if ( (value & RbudrMask) == RbudrMask )
    {
        return nn::bpc::ResultRtcAccessFailed();
    }

    NN_RESULT_DO(nn::i2c::ReadSingleRegister(pOutValue, session, pAddress));

    NN_RESULT_SUCCESS;
}

nn::Result WriteRtcDoubleBufferredRegister(nn::i2c::I2cSession& session, const uint8_t* pAddress, const uint8_t* pValue) NN_NOEXCEPT
{
    const uint8_t RtcUpdate0Address = 0x04;
    const uint8_t UdrMask = 0x01;
    const int MaxRetryCount = 4;

    NN_RESULT_DO(nn::i2c::WriteSingleRegister(session, pAddress, pValue));

    NN_RESULT_DO(nn::i2c::WriteSingleRegister(session, &RtcUpdate0Address, &UdrMask));

    int count = 0;
    uint8_t value = 0x00;
    do
    {
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(16));
        NN_RESULT_DO(nn::i2c::ReadSingleRegister(&value, session, &RtcUpdate0Address));
        count++;
    }
    while ( ((value & UdrMask) == UdrMask) && count < MaxRetryCount );

    if ( (value & UdrMask) == UdrMask )
    {
        return nn::bpc::ResultRtcAccessFailed();
    }

    NN_RESULT_SUCCESS;
}

TEST(RtcModeTest, ChangeRtcMode)
{
    nn::i2c::I2cSession session;
    nn::i2c::OpenSession(&session, nn::i2c::I2cDevice_Max77620Rtc);

    // RTCCNTLM を汚す
    {
        const uint8_t address = 2;
        const uint8_t value = 3;
        uint8_t outValue = 0;
        NN_ABORT_UNLESS_RESULT_SUCCESS(WriteRtcDoubleBufferredRegister(session, &address, &value));
        NN_ABORT_UNLESS_RESULT_SUCCESS(ReadRtcDoubleBufferredRegister(&outValue, session, &address));
        NN_LOG("Wrote 0x%02x to 0x%02x.\n", outValue, address);
    }

    // RTCCNTLM を汚す
    {
        const uint8_t address = 3;
        const uint8_t value = 1;
        uint8_t outValue = 0;
        NN_ABORT_UNLESS_RESULT_SUCCESS(WriteRtcDoubleBufferredRegister(session, &address, &value));
        NN_ABORT_UNLESS_RESULT_SUCCESS(ReadRtcDoubleBufferredRegister(&outValue, session, &address));
        NN_LOG("Wrote 0x%02x to 0x%02x.\n", outValue, address);
    }

    nn::i2c::CloseSession(session);

    // システムを再起動させて汚れが落ちることを確認する（timesrv 的には RTC 電源断と同じ挙動になる）
}

}} // namespace nnt::bpc
