﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Log.h>
#include <nn/nn_Assert.h>
#include <nn/os.h>

#include <nn/hid/hid_Npad.h>
#include <nn/hid/hid_NpadPalma.h>
#include <nn/hid/hid_PalmaConnectionApi.h>

#include <nn/bluetooth/bluetooth_BleScanParameterIdPalma.h>

#include "../Common/BluetoothTools_Color.h"
#include "../Common/BluetoothTools_Utility.h"
#include "BleTestTool_ScanScene.h"
#include "BleTestTool_Uuid.h"
#include "BleTestTool_ConnectionInfoDumper.h"

namespace
{
    const char* GetScanTargetString(ScanScene::ScanTarget target)
    {
        switch (target)
        {
        case ScanScene::ScanTarget_None:
            return "None";
        case ScanScene::ScanTarget_UnpairedPalma:
            return "Unpaired Palma";
        case ScanScene::ScanTarget_PairedPalma:
            return "Paired Palma";
        case ScanScene::ScanTarget_DummyDevices:
            return "Dummy BLE Devices";
        case ScanScene::ScanTarget_DummyDevice_1111:
            return "Dummy BLE Device (0x11 0x11)";
        case ScanScene::ScanTarget_DummyDevice_2222:
            return "Dummy BLE Device (0x22 0x22)";
        case ScanScene::ScanTarget_DummyDevice_3333:
            return "Dummy BLE Device (0x33 0x33)";
        case ScanScene::ScanTarget_DummyDevice_4444:
            return "Dummy BLE Device (0x44 0x44)";
        case ScanScene::ScanTarget_DummyDevice_5555:
            return "Dummy BLE Device (0x55 0x55)";
        case ScanScene::ScanTarget_DefaultSmartDevice:
            return "Default Smart Device (0x99)";
        case ScanScene::ScanTarget_DummySmartDevices:
            return "Dummy Smart Devices";
        case ScanScene::ScanTarget_DummySmartDevice_88:
            return "Dummy Smart Device (0x88)";
        case ScanScene::ScanTarget_DummySmartDevice_77:
            return "Dummy Smart Device (0x77)";
        case ScanScene::ScanTarget_DummySmartDevice_66:
            return "Dummy Smart Device (0x66)";
        case ScanScene::ScanTarget_DummySmartDevice_55:
            return "Dummy Smart Device (0x55)";
        case ScanScene::ScanTarget_DummySmartDevice_44:
            return "Dummy Smart Device (0x44)";
        default:
            NN_UNEXPECTED_DEFAULT;
        }
    }

    bool GetScanFilter(nn::bluetooth::GattAttributeUuid* pUuid, ScanScene::ScanTarget target)
    {
        NN_ABORT_UNLESS_NOT_NULL(pUuid);
        bool res = true;

        *pUuid = ADV_SERVICE_UUID;

        switch (target)
        {
        case ScanScene::ScanTarget_DummySmartDevice_88:
            pUuid->uu.uuid128[nn::bluetooth::GattAttributeUuidLength_128 - 1] = 0x88;
            break;
        case ScanScene::ScanTarget_DummySmartDevice_77:
            pUuid->uu.uuid128[nn::bluetooth::GattAttributeUuidLength_128 - 1] = 0x77;
            break;
        case ScanScene::ScanTarget_DummySmartDevice_66:
            pUuid->uu.uuid128[nn::bluetooth::GattAttributeUuidLength_128 - 1] = 0x66;
            break;
        case ScanScene::ScanTarget_DummySmartDevice_55:
            pUuid->uu.uuid128[nn::bluetooth::GattAttributeUuidLength_128 - 1] = 0x55;
            break;
        case ScanScene::ScanTarget_DummySmartDevice_44:
            pUuid->uu.uuid128[nn::bluetooth::GattAttributeUuidLength_128 - 1] = 0x44;
            break;
        case ScanScene::ScanTarget_DefaultSmartDevice:
            break;
        default:
            res = false;
            break;
        }

        return res;
    }

    bool GetScanFilter(nn::bluetooth::BleAdvertisePacketParameter* pParam, ScanScene::ScanTarget target)
    {
        NN_ABORT_UNLESS_NOT_NULL(pParam);
        bool res = true;

        nn::bluetooth::GetBleScanParameter(pParam, nn::bluetooth::BleScanParameterId_Palma);

        switch (target)
        {
        case ScanScene::ScanTarget_DummyDevice_1111:
            pParam->serverId[0] = 0x11;
            pParam->serverId[1] = 0x11;
            break;
        case ScanScene::ScanTarget_DummyDevice_2222:
            pParam->serverId[0] = 0x22;
            pParam->serverId[1] = 0x22;
            break;
        case ScanScene::ScanTarget_DummyDevice_3333:
            pParam->serverId[0] = 0x33;
            pParam->serverId[1] = 0x33;
            break;
        case ScanScene::ScanTarget_DummyDevice_4444:
            pParam->serverId[0] = 0x44;
            pParam->serverId[1] = 0x44;
            break;
        case ScanScene::ScanTarget_DummyDevice_5555:
            pParam->serverId[0] = 0x55;
            pParam->serverId[1] = 0x55;
            break;
        default:
            res = false;
            break;
        }

        return res;
    }
}   // namespace


ScanScene::ScanScene(std::string title)
    : SceneBase(title)
{
    ClearScanningTarget();

    m_SelectingScanTarget = ScanTarget_DefaultSmartDevice;
}

void ScanScene::Draw(nn::gfx::util::DebugFontTextWriter* pTextWriter)
{
    NN_ASSERT_NOT_NULL(pTextWriter);

    pTextWriter->SetCursor(20, 20);
    pTextWriter->SetTextColor(Color::White);
    pTextWriter->Print("%s", m_Title.c_str());

    int scanFilterCount = isScanning();
    pTextWriter->SetCursor(20, 40);
    pTextWriter->Print("SCAN STATE: %s (%d)", (scanFilterCount > 0) ? "SCANNING" : "NOT SCANNING", scanFilterCount);

    pTextWriter->SetCursor(20, 65);
    pTextWriter->Print("Scan Results:");

    float offsetY = 65;
    for (int i = 0; i < NN_ARRAY_SIZE(m_ScanResults); ++i)
    {
        if (m_Cursor.posY == i)
        {
            pTextWriter->SetTextColor(Color::Orange);
        }

        offsetY += 25;

        pTextWriter->SetCursor(40, offsetY);
        pTextWriter->Print("#%d    Address: %02X:%02X:%02X:%02X:%02X:%02X    RSSI: %d",
            i,
            m_ScanResults[i].address.address[0], m_ScanResults[i].address.address[1], m_ScanResults[i].address.address[2],
            m_ScanResults[i].address.address[3], m_ScanResults[i].address.address[4], m_ScanResults[i].address.address[5],
            m_ScanResults[i].rssi);

        pTextWriter->SetTextColor(Color::White);
    }

    pTextWriter->SetCursor(650, 65);
    pTextWriter->Print("Advertise Packet Content:");

    offsetY = 65;

    nn::bluetooth::BleScanResult selectedResult = m_ScanResults[m_Cursor.posY];

    for (int i = 0; i < selectedResult.adStructureNum; ++i)
    {
        offsetY += 30;
        pTextWriter->SetCursor(670, offsetY);
        pTextWriter->Print("AD Type: %s    Length: %d",
            ConvertAdTypeToString(selectedResult.adStructures[i].adType), selectedResult.adStructures[i].length);

        offsetY += 25;
        pTextWriter->SetCursor(670, offsetY);

        int offsetX = 0;
        for (int j = 0; j < selectedResult.adStructures[i].length - 1; ++j)
        {
            pTextWriter->SetCursorX(670 + offsetX * pTextWriter->GetFontWidth() * 2);
            pTextWriter->Print("%02X ", selectedResult.adStructures[i].data[j]);

            offsetX++;

            if (j % 16 == 15)
            {
                offsetX = 0;
                offsetY += 20;
                pTextWriter->SetCursorY(offsetY);
            }
        }

        if (selectedResult.adStructures[i].adType == nn::bluetooth::BleAdType_CompleteLocalName ||
            selectedResult.adStructures[i].adType == nn::bluetooth::BleAdType_ShortenedLocalName)
        {
            offsetY += 20;
            pTextWriter->SetCursorY(offsetY);
            pTextWriter->Print("Device Name: %s", selectedResult.adStructures[i].data);

            ConnectionInfoDumper::StoreDeviceName(reinterpret_cast<const char*>(selectedResult.adStructures[i].data),
                                                  selectedResult.adStructures[i].length - 1,
                                                  selectedResult.address);
        }
    }

    pTextWriter->SetCursor(650, 340);
    pTextWriter->Print("Selecting scan target: ");

    pTextWriter->SetCursor(840, 340);
    if (isScanning(m_SelectingScanTarget))
    {
        pTextWriter->SetTextColor(Color::Gray);
    }
    else
    {
        pTextWriter->SetTextColor(Color::Orange);
    }
    pTextWriter->Print("%s", GetScanTargetString(m_SelectingScanTarget));
    pTextWriter->SetTextColor(Color::White);

    pTextWriter->SetCursor(650, 370);
    pTextWriter->Print("On going scan list:");

    int offset = 20;
    for (const auto& target : m_ScanningDevices)
    {
        if (target != ScanTarget_None)
        {
            pTextWriter->SetCursor(670, 370 + offset);
            pTextWriter->Print("%s", GetScanTargetString(target));
            offset += 20;
        }
    }

    pTextWriter->SetCursor(20, 690);
    pTextWriter->Print("ZR/ZL: Select Scan Target    X: Start Scan    Y: Stop Scan    Up/Down: Select Scanned Device    B: Refresh Scan Result   A: Connect    +: Exit");
} // NOLINT(impl/function_size)

void ScanScene::ProcessControllerInput(nns::hid::ButtonSet buttons)
{
    if ((buttons & nns::hid::Button::X::Mask).IsAnyOn())
    {
        NN_LOG("Start BLE Scan\n");

        switch (m_SelectingScanTarget)
        {
        case ScanTarget_UnpairedPalma:
        case ScanTarget_PairedPalma:
            StartScanPalma(m_SelectingScanTarget == ScanTarget_UnpairedPalma);
            break;
        case ScanTarget_DummyDevices:
        case ScanTarget_DummyDevice_1111:
        case ScanTarget_DummyDevice_2222:
        case ScanTarget_DummyDevice_3333:
        case ScanTarget_DummyDevice_4444:
        case ScanTarget_DummyDevice_5555:
            StartBleScanGeneral();
            break;
        case ScanTarget_DefaultSmartDevice:
        case ScanTarget_DummySmartDevices:
        case ScanTarget_DummySmartDevice_44:
        case ScanTarget_DummySmartDevice_55:
        case ScanTarget_DummySmartDevice_66:
        case ScanTarget_DummySmartDevice_77:
        case ScanTarget_DummySmartDevice_88:
            StartBleScanSmartDevice();
            break;
        default:
            return;
        }
    }
    else if ((buttons & nns::hid::Button::Y::Mask).IsAnyOn())
    {
        NN_LOG("Stop BLE Scan\n");

        StopScanPalma(true);
        StopBleScanGeneral();
        StopBleScanSmartDevice();
    }
    else if ((buttons & nns::hid::Button::B::Mask).IsAnyOn())
    {
        NN_LOG("Refresh BLE Scan\n");

        UpdateScanResult();
    }
    else if ((buttons & nns::hid::Button::Up::Mask).IsAnyOn())
    {
        if (m_Cursor.posY > 0)
        {
            m_Cursor.posY--;
        }
    }
    else if ((buttons & nns::hid::Button::Down::Mask).IsAnyOn())
    {
        if (m_Cursor.posY < NN_ARRAY_SIZE(m_ScanResults) - 1)
        {
            m_Cursor.posY++;
        }
    }
    else if ((buttons & nns::hid::Button::A::Mask).IsAnyOn())
    {
        NN_LOG("Connect to GATT Server\n");

        nn::bluetooth::BleScanResult selectedResult = m_ScanResults[m_Cursor.posY];

        auto result = nn::bluetooth::ConnectToGattServer(selectedResult.address);
        //!< 既に接続が存在します。
        //!< 本サンプルでは、GATT Client Scene で接続を確認できます。
        if (nn::bluetooth::ResultBleAlreadyConnected().Includes(result))
        {
            NN_LOG("Connection already exist.\n");
        }
        //!< 接続数が最大に達しています。
        //!< 本サンプルでは、GATT Client Scene でX ボタンを押して切断できます。
        else if (nn::bluetooth::ResultBleConnectionFull().Includes(result))
        {
            NN_LOG("Connection full. Disconnect existing connection and try again.\n");
        }
        //!< Bluetooth がOFF になっています。
        else if (nn::bluetooth::ResultBluetoothOff().Includes(result))
        {
            NN_LOG("Bluetooth is OFF.\n");
        }
        //!< BLE 機能を使える状態にありません。必要なコントローラーのみを接続してください。
        else if (nn::bluetooth::ResultInvalidWirelessState().Includes(result))
        {
            NN_LOG("You must disconnect unnecessary controllers.\n");
        }
    }
    else if ((buttons & nns::hid::Button::ZR::Mask).IsAnyOn())
    {
        if (m_SelectingScanTarget == ScanTarget_DummySmartDevice_44)
        {
            // 何もしない
        }
        else
        {
            m_SelectingScanTarget = static_cast<ScanTarget>(static_cast<uint8_t>(m_SelectingScanTarget) + 1);
        }
    }
    else if ((buttons & nns::hid::Button::ZL::Mask).IsAnyOn())
    {
        if (m_SelectingScanTarget == ScanTarget_UnpairedPalma)
        {
            // 何もしない
        }
        else
        {
            m_SelectingScanTarget = static_cast<ScanTarget>(static_cast<uint8_t>(m_SelectingScanTarget) - 1);
        }
    }
} // NOLINT(impl/function_size)

bool ScanScene::StartBleScanGeneral()
{
    int startTarget = 0;
    int endTarget = 0;

    if (m_SelectingScanTarget == ScanTarget_DummyDevices)
    {
        startTarget = static_cast<int>(ScanTarget_DummyDevice_1111);
        endTarget   = static_cast<int>(ScanTarget_DummyDevice_4444);
    }
    else
    {
        startTarget = static_cast<int>(m_SelectingScanTarget);
        endTarget   = static_cast<int>(m_SelectingScanTarget);
    }

    for (int target = startTarget; target <= endTarget; ++target)
    {
        nn::bluetooth::BleAdvertisePacketParameter param;

        if (!GetScanFilter(&param, static_cast<ScanTarget>(target)))
        {
            NN_LOG("Failed to start BLE Scan. %s is not a BLE device.\n",
                    GetScanTargetString(static_cast<ScanTarget>(target)));
            return false;
        }

        auto result = nn::bluetooth::StartBleScanGeneral(param);
        if (result.IsFailure())
        {
            NN_LOG("Failed to start BLE Scan General.  Manu ID = %02X:%02X, Client ID = %02X:%02X:%02X, Server ID = %02X:%02X:%02X\n",
                param.manufacturerId[0], param.manufacturerId[1],
                param.clientId[0], param.clientId[1], param.clientId[2],
                param.serverId[0], param.serverId[1], param.serverId[2]);

            //!< システムが他のタスクを実行中なので、少し待ってから再試行してください。
            //!< 本サンプルでは、X ボタンを押すことで再試行します。
            if (nn::bluetooth::ResultSystemBusy().Includes(result))
            {
                NN_LOG("System Busy. Wait a little and try again .\n");
            }
            //!< システムで何らかのエラーが発生しました。再試行することで成功することがあります。
            //!< 本サンプルでは、X ボタンを押すことで再試行します。
            else if (nn::bluetooth::ResultGeneralError().Includes(result))
            {
                NN_LOG("General Error. Wait a little and try again.\n");
            }
            //!< BLE スキャンの対象が上限に達しています。不要な既存のBLE スキャンを停止してから再試行してください。
            //!< 本サンプルでは、Y ボタンを押すことでBLE スキャンを停止します。
            else if (nn::bluetooth::ResultBleScanFull().Includes(result))
            {
                NN_LOG("Scan filter is full. Stop on going BLE scan and try again.\n");
            }
            //!< Bluetooth がOFF になっています。
            else if (nn::bluetooth::ResultBluetoothOff().Includes(result))
            {
                NN_LOG("Bluetooth is OFF.\n");
            }
            return false;
        }
        else
        {
            AddScanningTarget(static_cast<ScanTarget>(target));
        }
    }

    ClearScanResult();

    return true;
}

bool ScanScene::StopBleScanGeneral()
{
    auto result = nn::bluetooth::StopBleScanGeneral();
    if (result.IsFailure())
    {
        NN_LOG("Failed to stop BLE Scan General.\n");

        //!< システムが他のタスクを実行中なので、少し待ってからリトライしてください
        if (nn::bluetooth::ResultSystemBusy().Includes(result))
        {
            NN_LOG("Wait a little and try again .\n");
        }
        //!< システムで何らかのエラーが発生しました。再試行することで成功することがあります。
        else if (nn::bluetooth::ResultGeneralError().Includes(result))
        {
            NN_LOG("Wait a little and trye again.\n");
        }
        //!< Bluetooth がOFF になっています。
        else if (nn::bluetooth::ResultBluetoothOff().Includes(result))
        {
            NN_LOG("Bluetooth is OFF.\n");
        }

        return false;
    }

    for (int target = static_cast<int>(ScanTarget_DummyDevice_1111); target <= static_cast<int>(ScanTarget_DummyDevice_5555); ++target)
    {
        if (target == static_cast<int>(ScanTarget_DummyDevices))
        {
            continue;
        }

        RemoveScanningTaraget(static_cast<ScanTarget>(target));
    }

    return true;
}

bool ScanScene::StartScanPalma(bool includingUnpaired)
{
    nn::hid::NpadStyleSet supportedStyles = nn::hid::GetSupportedNpadStyleSet();

    if (!supportedStyles.Test(nn::hid::NpadStylePalma::Index))
    {
        nn::hid::SetSupportedNpadStyleSet(supportedStyles | nn::hid::NpadStylePalma::Mask);
        AddScanningTarget(ScanTarget_PairedPalma);
    }

    if (includingUnpaired && !isScanning(ScanTarget_UnpairedPalma))
    {
        nn::hid::EnableAnyPalmaConnection();
        AddScanningTarget(ScanTarget_UnpairedPalma);
    }


    return true;
}

bool ScanScene::StopScanPalma(bool includingPaired)
{
    if (isScanning(ScanTarget_UnpairedPalma))
    {
        nn::hid::DisableAnyPalmaConnection();
        RemoveScanningTaraget(ScanTarget_UnpairedPalma);
    }

    if (includingPaired)
    {
        nn::hid::NpadStyleSet supportedStyles = nn::hid::GetSupportedNpadStyleSet();

        if (supportedStyles.Test(nn::hid::NpadStylePalma::Index))
        {
            nn::hid::SetSupportedNpadStyleSet(supportedStyles ^ nn::hid::NpadStylePalma::Mask);
            RemoveScanningTaraget(ScanTarget_PairedPalma);
        }
    }

    return true;
}

bool ScanScene::StartBleScanSmartDevice()
{
    int startTarget = 0;
    int endTarget = 0;

    if (m_SelectingScanTarget == ScanTarget_DummySmartDevices)
    {
        startTarget = static_cast<int>(ScanTarget_DummySmartDevice_88);
        endTarget   = static_cast<int>(ScanTarget_DummySmartDevice_55);
    }
    else
    {
        startTarget = static_cast<int>(m_SelectingScanTarget);
        endTarget   = static_cast<int>(m_SelectingScanTarget);
    }

    for (int target = startTarget; target <= endTarget; ++target)
    {
        nn::bluetooth::GattAttributeUuid advertiseUuid;

        if (!GetScanFilter(&advertiseUuid, static_cast<ScanTarget>(target)))
        {
            NN_LOG("Failed to start BLE Scan. Selected target is not a smart device.\n");
            return false;
        }

        auto result = nn::bluetooth::StartBleScanSmartDevice(advertiseUuid);
        if (result.IsFailure())
        {
            NN_LOG("Failed to start BLE Scan Smart Device.\n");

            //!< システムが他のタスクを実行中なので、少し待ってから再試行してください。
            //!< 本サンプルでは、X ボタンを押すことで再試行します。
            if (nn::bluetooth::ResultSystemBusy().Includes(result))
            {
                NN_LOG("Wait a little and try again .\n");
            }
            //!< システムで何らかのエラーが発生しました。再試行することで成功することがあります。
            //!< 本サンプルでは、X ボタンを押すことで再試行します。
            else if (nn::bluetooth::ResultGeneralError().Includes(result))
            {
                NN_LOG("Wait a little and trye again.\n");
            }
            //!< BLE スキャンの対象が上限に達しています。不要な既存のBLE スキャンを停止してから再試行してください。
            //!< 本サンプルでは、Y ボタンを押すことでBLE スキャンを停止します。
            else if (nn::bluetooth::ResultBleScanFull().Includes(result))
            {
                NN_LOG("Stop on going BLE scan and try again.\n");
            }
            //!< Bluetooth がOFF になっています。
            else if (nn::bluetooth::ResultBluetoothOff().Includes(result))
            {
                NN_LOG("Bluetooth is OFF.\n");
            }
            return false;
        }
        else
        {
            AddScanningTarget(static_cast<ScanTarget>(target));
        }
    }

    ClearScanResult();

    return true;
}

bool ScanScene::StopBleScanSmartDevice()
{
    auto result = nn::bluetooth::StopBleScanSmartDevice();
    if (result.IsFailure())
    {
        NN_LOG("Failed to stop BLE Scan Smart Device.\n");

        //!< システムが他のタスクを実行中なので、少し待ってからリトライしてください
        if (nn::bluetooth::ResultSystemBusy().Includes(result))
        {
            NN_LOG("Wait a little and try again .\n");
        }
        //!< システムで何らかのエラーが発生しました。再試行することで成功することがあります。
        else if (nn::bluetooth::ResultGeneralError().Includes(result))
        {
            NN_LOG("Wait a little and trye again.\n");
        }
        //!< Bluetooth がOFF になっています。
        else if (nn::bluetooth::ResultBluetoothOff().Includes(result))
        {
            NN_LOG("Bluetooth is OFF.\n");
        }
        return false;
    }

    for (int target = static_cast<int>(ScanTarget_DefaultSmartDevice); target <= static_cast<int>(ScanTarget_DummySmartDevice_44); ++target)
    {
        if (target == static_cast<int>(ScanTarget_DummySmartDevices))
        {
            continue;
        }

        RemoveScanningTaraget(static_cast<ScanTarget>(target));
    }

    return true;
}

void ScanScene::UpdateScanResult()
{
    //!< スキャン結果を取得
    nn::bluetooth::GetBleScanResult(m_ScanResults, NN_ARRAY_SIZE(m_ScanResults));
}

void ScanScene::ClearScanResult()
{
    for (int i = 0; i < NN_ARRAY_SIZE(m_ScanResults); ++i)
    {
        memset(&m_ScanResults[i], 0x00, sizeof(nn::bluetooth::BleScanResult));
    }
}

const char* ScanScene::ConvertAdTypeToString(nn::bluetooth::BleAdType type) const
{
    switch (type)
    {
    case nn::bluetooth::BleAdType_Unkwnon:
        return "Unkown";
    case nn::bluetooth::BleAdType_Flags:
        return "Flags";
    case nn::bluetooth::BleAdType_IncompleteListOf16bitUuid:
        return "Incomp 16 bit UUID";
    case nn::bluetooth::BleAdType_CompleteListOf16bitUuid:
        return "Comp 16 bit UUID";
    case nn::bluetooth::BleAdType_IncompleteListOf32bitUuid:
        return "Incomp 32 bit UUID";
    case nn::bluetooth::BleAdType_CompleteListOf32bitUuid:
        return "Compl 32 bit UUID";
    case nn::bluetooth::BleAdType_IncompleteListOf128bitUuid:
        return "Incomp 128 bit UUID";
    case nn::bluetooth::BleAdType_CompleteListOf128bitUuid:
        return "Comp 128 bit UUID";
    case nn::bluetooth::BleAdType_ShortenedLocalName:
        return "Shortened Local Name";
    case nn::bluetooth::BleAdType_CompleteLocalName:
        return "Complete Local Name";
    case nn::bluetooth::BleAdType_TxPowerLevel:
        return "Tx Power Level";
    case nn::bluetooth::BleAdType_DeviceId:
        return "Device ID";
    case nn::bluetooth::BleAdType_16bitServiceData:
        return "16 bit Service Data";
    case nn::bluetooth::BleAdType_32bitServiceData:
        return "32 bit Service Data";
    case nn::bluetooth::BleAdType_128bitServiceData:
        return "128 bit Service Data";
    case nn::bluetooth::BleAdType_ManufactureSpecificData:
        return "Manufacturer Specific Data";
    default:
        NN_UNEXPECTED_DEFAULT;
    }
}

bool ScanScene::isScanning(ScanTarget target) const
{
    for (const auto& targetEntry : m_ScanningDevices)
    {
        if (targetEntry == target)
        {
            return true;
        }
    }

    return false;
}

int ScanScene::isScanning() const
{
    int count = 0;

    for (const auto& targetEntry : m_ScanningDevices)
    {
        if (targetEntry != ScanTarget_None)
        {
            count++;
        }
    }

    return count;
}

void ScanScene::AddScanningTarget(ScanTarget target)
{
    if (isScanning(target))
    {
        return;
    }

    for (auto& targetEntry : m_ScanningDevices)
    {
        if (targetEntry == ScanTarget_None)
        {
            targetEntry = target;
            return;
        }
    }

    NN_ABORT("Scanning target list is full.\n");
}

void ScanScene::ClearScanningTarget()
{
    for (auto& targetEntry : m_ScanningDevices)
    {
        targetEntry = ScanTarget_None;
    }
}

void ScanScene::RemoveScanningTaraget(ScanTarget target)
{
    for (auto& targetEntry : m_ScanningDevices)
    {
        if (targetEntry == target)
        {
            targetEntry = ScanTarget_None;
        }
    }
}
