﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#include <cstdio>

#include <nnt/nntest.h>
#include <nn/nn_Log.h>

#include <nn/os.h>
#include <nn/os/os_SystemEventTypes.h>
#include <nn/settings/system/settings_BluetoothDevices.h>
#include <nn/bluetooth/bluetooth_Api.h>

using namespace nn::bluetooth;

namespace
{
    nn::os::SystemEventType g_SystemEvent;
    nn::os::SystemEventType g_SystemEventForHid;
    nn::os::MultiWaitType g_MultiWait;
    nn::os::MultiWaitHolderType g_HolderForBluetoothCore;
    nn::os::MultiWaitHolderType g_HolderForHid;

    enum EventType
    {
        EVENTTYPE_FROM_CORE = 1,
        EVENTTYPE_FROM_HID
    };

    class DiscoveryTest : public testing::Test
    {
    protected:
        static void SetUpTestCase()
        {
            nn::settings::system::BluetoothDevicesSettings settings[nn::settings::system::BluetoothDevicesSettingsCountMax];
            auto count = nn::settings::system::GetBluetoothDevicesSettings(settings, nn::settings::system::BluetoothDevicesSettingsCountMax);
            if (count > 0)
            {
                nn::settings::system::SetBluetoothDevicesSettings(nullptr, 0);
                NN_LOG("%d pairing settings are deleted.\n", count);
            }

            nn::bluetooth::InitializeBluetoothDriver();
            nn::bluetooth::InitializeBluetooth(&g_SystemEvent);
            nn::bluetooth::EnableBluetooth();
            nn::bluetooth::InitializeHid(&g_SystemEventForHid);

            nn::os::InitializeMultiWaitHolder(&g_HolderForBluetoothCore, &g_SystemEvent);
            nn::os::SetMultiWaitHolderUserData(&g_HolderForBluetoothCore, EVENTTYPE_FROM_CORE);

            nn::os::InitializeMultiWaitHolder(&g_HolderForHid, &g_SystemEventForHid);
            nn::os::SetMultiWaitHolderUserData(&g_HolderForHid, EVENTTYPE_FROM_HID);

            nn::os::InitializeMultiWait(&g_MultiWait);
            nn::os::LinkMultiWaitHolder(&g_MultiWait, &g_HolderForBluetoothCore);
            nn::os::LinkMultiWaitHolder(&g_MultiWait, &g_HolderForHid);
        }

        static void TearDownTestCase()
        {
            nn::os::UnlinkAllMultiWaitHolder(&g_MultiWait);
            nn::os::FinalizeMultiWaitHolder(&g_HolderForBluetoothCore);
            nn::os::FinalizeMultiWaitHolder(&g_HolderForHid);
            nn::os::FinalizeMultiWait(&g_MultiWait);
            nn::os::DestroySystemEvent(&g_SystemEvent);
            nn::os::DestroySystemEvent(&g_SystemEventForHid);

            nn::bluetooth::DisableBluetooth();
            nn::bluetooth::CleanupBluetooth();
            nn::bluetooth::CleanupHid();
            nn::bluetooth::FinalizeBluetoothDriver();
        }
    };
}

static char* BdaddrToString(const uint8_t bdaddr[])
{
    static char strBdaddr[18];
    sprintf(strBdaddr, "%02X:%02X:%02X:%02X:%02X:%02X",
        bdaddr[0], bdaddr[1], bdaddr[2], bdaddr[3], bdaddr[4], bdaddr[5]);
    return strBdaddr;
}

static void PrintEventInfo(nn::bluetooth::EventType eventType, const uint8_t buffer[])
{
    NN_LOG("<- Event Core: ");
    switch (eventType)
    {
    case nn::bluetooth::EventFromDiscoveryStateChangedCallback:
        {
            NN_LOG("DiscoveryStateChanged\n");
            BluetoothDiscoveryState state = reinterpret_cast<const nn::bluetooth::InfoFromDiscoveryStateChangedCallback*>(&buffer[0])->state;
            if (state == BT_DISCOVERY_STOPPED)
            {
                NN_LOG("STOPPED\n");
            }
            else
            {
                NN_LOG("STARTED\n");
            }
        }
        break;
    case nn::bluetooth::EventFromDeviceFoundCallback:
        {
            NN_LOG("DeviceFound\n");
            const nn::bluetooth::InfoFromDeviceFoundCallback* pInfo;
            pInfo = reinterpret_cast<const nn::bluetooth::InfoFromDeviceFoundCallback*>(&buffer[0]);
            NN_LOG("%s\n", reinterpret_cast<const char*>(&pInfo->BdName.name[0]));
            NN_LOG("%s\n", BdaddrToString(pInfo->BdAddress.address));
        }
        break;
    case nn::bluetooth::EventFromPinRequestCallback:
        {
            NN_LOG("PinRequest\n");
        }
        break;
    case nn::bluetooth::EventFromSspRequestCallback:
        {
            NN_LOG("SspRequest\n");
            const nn::bluetooth::InfoFromSspRequestCallback* pInfo;
            pInfo = reinterpret_cast<const nn::bluetooth::InfoFromSspRequestCallback*>(&buffer[0]);
            NN_LOG("%s [%s]\n", pInfo->bluetoothName, BdaddrToString(pInfo->bluetoothAddress.address));
        }
        break;
    case nn::bluetooth::EventFromBondStateChangedCallback:
        {
            NN_LOG("BondStateChanged\n");
            const nn::bluetooth::InfoFromBondStateChangedCallback* pInfo;
            pInfo = reinterpret_cast<const nn::bluetooth::InfoFromBondStateChangedCallback*>(&buffer[0]);
            NN_LOG("%s\n", BdaddrToString(pInfo->bluetoothAddress.address));
            if (pInfo->status == BT_STATUS_SUCCESS)
            {
                if (pInfo->state == BT_BOND_STATE_NONE)
                {
                    NN_LOG("BT_BOND_STATE_NONE\n");
                }
                else if (pInfo->state == BT_BOND_STATE_BONDING)
                {
                    NN_LOG("BT_BOND_STATE_BONDING\n");
                }
                else if (pInfo->state == BT_BOND_STATE_BONDED)
                {
                    NN_LOG("BT_BOND_STATE_BONDED\n");
                }
            }
        }
        break;
    case nn::bluetooth::EventFromAdapterPropertiesCallback:
        {
            NN_LOG("AdapterProperties\n");
        }
        break;
    default:
        {
            NN_LOG("Other. Event type =%d\n", eventType);
        }
    }
}

#if 0
static void PrintEventInfo(nn::bluetooth::HidEventType eventType, const uint8_t buffer[])
{
    if (eventType != nn::bluetooth::EventFromGetReportCallback)
    {
        NN_LOG("<- Event Hid: ");
    }
    switch (eventType)
    {
    case nn::bluetooth::EventFromConnectionStateCallback:
        {
            NN_LOG("ConnectionState\n");
            const nn::bluetooth::InfoFromConnectionStateCallback* pInfo;
            pInfo = reinterpret_cast<const nn::bluetooth::InfoFromConnectionStateCallback*>(&buffer[0]);
            if (pInfo->state == BTHH_CONN_STATE_CONNECTED)
            {
                NN_LOG("Connected %s\n", BdaddrToString(pInfo->bluetoothAddress));
            }
            else
            {
                NN_LOG("Disconnected %s\n", BdaddrToString(pInfo->bluetoothAddress));
            }
        }
        break;
    case nn::bluetooth::EventFromGetReportCallback:
        {
            const nn::bluetooth::InfoFromGetReportCallback* pInfo = reinterpret_cast<const nn::bluetooth::InfoFromGetReportCallback*>(&buffer[0]);
            const nn::bluetooth::HidReport* pReport = &pInfo->rptData;

            //"Basic Input Report" came.
            if (pReport->reportData.data[0] == 0x30)
            {
                //NN_LOG(".");
            }

            //"Reply for the Get Device Info" came.
            else if (pReport->reportData.data[0] == 0x21)
            {
                NN_LOG("[!!!!]");
                //Time stamp for measuring the latency of data transaction.
                //NN_LOG("[MeasuringLatency]ending time   = %d (msec)\n",nn::os::ConvertToTimeSpan(nn::os::GetSystemTick()).GetMilliSeconds());
                //PrintByteDataAsHex(32,&pUipcReport->report.report_data.data[0]);
            }
            else
            {
                NN_LOG("[????]");
            }
        }
        break;
    case nn::bluetooth::EventFromExtensionCallbacks:
        {
            NN_LOG("Extension %d\n", reinterpret_cast<const nn::bluetooth::InfoFromExtensionCallbacks*>(&buffer[0])->eventType);
        }
        break;
    default:
        {
            NN_LOG("\n\n other event. EventType= %d", eventType);
        }
    }
}
#endif

TEST_F(DiscoveryTest, BT_TC_2_1)
{
    nn::TimeSpan current = nn::os::GetSystemTick().ToTimeSpan();
    NN_LOG("\n\t Discovery test BT_TC_2_1, start discovery\n");
    NN_LOG("-----------------------------------------------------------------------\n");
    nn::bluetooth::StartDiscovery();

    for (;;)
    {
        nn::os::MultiWaitHolderType* holder = nn::os::WaitAny(&g_MultiWait);
        int holderType = nn::os::GetMultiWaitHolderUserData(holder);
        switch (holderType)
        {
        case EVENTTYPE_FROM_CORE:
            {
                nn::os::TryWaitSystemEvent(&g_SystemEvent);
                nn::bluetooth::EventType type;
                uint8_t buffer[nn::bluetooth::BUFFER_SIZE_OF_CORE_OUT] = {0};
                nn::bluetooth::GetEventInfo(&type, buffer, nn::bluetooth::BUFFER_SIZE_OF_CORE_OUT);
                PrintEventInfo(type, buffer);
                if (type == nn::bluetooth::EventFromDiscoveryStateChangedCallback)
                {
                    BluetoothDiscoveryState state = reinterpret_cast<const nn::bluetooth::InfoFromDiscoveryStateChangedCallback*>(&buffer[0])->state;
                    if (state == BT_DISCOVERY_STOPPED)
                    {
                        NN_LOG("%lld ms\n", (nn::os::GetSystemTick().ToTimeSpan() - current).GetMilliSeconds());
                        SUCCEED();
                        return;
                    }
                }
                else if (type == nn::bluetooth::EventFromDeviceFoundCallback)
                {}
                else
                {
                    // Unexpected cb
                    FAIL();
                    return;
                }
            }
            break;
        case EVENTTYPE_FROM_HID:
            {
                // Unexpected cb
                FAIL();
                return;
            }
            break;
        default: NN_UNEXPECTED_DEFAULT;
        }
    }
}

TEST_F(DiscoveryTest, BT_TC_2_2)
{
    int count = 0;
    //nn::TimeSpan start = nn::os::GetSystemTick().ToTimeSpan();
    NN_LOG("\n\t Discovery test BT_TC_2_2, start discovery 10 times\n");
    NN_LOG("-----------------------------------------------------------------------\n");
    nn::bluetooth::StartDiscovery();
    while (count < 10)
    {
        nn::os::MultiWaitHolderType* holder = nn::os::WaitAny(&g_MultiWait);
        int holderType = nn::os::GetMultiWaitHolderUserData(holder);
        switch (holderType)
        {
        case EVENTTYPE_FROM_CORE:
            {
                nn::os::TryWaitSystemEvent(&g_SystemEvent);
                nn::bluetooth::EventType type;
                uint8_t buffer[nn::bluetooth::BUFFER_SIZE_OF_CORE_OUT] = {0};
                nn::bluetooth::GetEventInfo(&type, buffer, nn::bluetooth::BUFFER_SIZE_OF_CORE_OUT);
                PrintEventInfo(type, buffer);
                if (type == nn::bluetooth::EventFromDiscoveryStateChangedCallback)
                {
                    BluetoothDiscoveryState state = reinterpret_cast<const nn::bluetooth::InfoFromDiscoveryStateChangedCallback*>(&buffer[0])->state;
                    if (state == BT_DISCOVERY_STOPPED)
                    {
                        //NN_LOG("%lld ms\n", (nn::os::GetSystemTick().ToTimeSpan() - start).GetMilliSeconds());
                        //start = nn::os::GetSystemTick().ToTimeSpan();
                        nn::bluetooth::StartDiscovery();
                        count++;
                    }
                }
                else if (type == nn::bluetooth::EventFromDeviceFoundCallback)
                {}
                else
                {
                    // Unexpected cb
                    FAIL();
                    return;
                }
            }
            break;
        case EVENTTYPE_FROM_HID:
            {
                // Unexpected cb
                FAIL();
                return;
            }
            break;
        default: NN_UNEXPECTED_DEFAULT;
        }
    }
    SUCCEED();
}

TEST_F(DiscoveryTest, BT_TC_2_3)
{
    nn::TimeSpan start = nn::os::GetSystemTick().ToTimeSpan();
    NN_LOG("\n\t Discovery test BT_TC_2_3, start and cancel discovery\n");
    NN_LOG("-----------------------------------------------------------------------\n");
    nn::bluetooth::StartDiscovery();

    nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(1000));

    nn::bluetooth::CancelDiscovery();

    for (;;)
    {
        nn::os::MultiWaitHolderType* holder = nn::os::WaitAny(&g_MultiWait);
        int holderType = nn::os::GetMultiWaitHolderUserData(holder);
        if (holderType == 1)
        {
            nn::os::TryWaitSystemEvent(&g_SystemEvent);
            nn::bluetooth::EventType type;
            uint8_t buffer[nn::bluetooth::BUFFER_SIZE_OF_CORE_OUT] = {0};
            nn::bluetooth::GetEventInfo(&type, buffer, nn::bluetooth::BUFFER_SIZE_OF_CORE_OUT);
            PrintEventInfo(type, buffer);
            if (type == nn::bluetooth::EventFromDiscoveryStateChangedCallback)
            {
                BluetoothDiscoveryState state = reinterpret_cast<const nn::bluetooth::InfoFromDiscoveryStateChangedCallback*>(&buffer[0])->state;
                if (state == BT_DISCOVERY_STOPPED)
                {
                    NN_LOG("%lld ms\n", (nn::os::GetSystemTick().ToTimeSpan() - start).GetMilliSeconds());
                    SUCCEED();
                    return;
                }
            }
        }
    }
}

TEST_F(DiscoveryTest, BT_TC_2_4)
{
    NN_LOG("\n\t Discovery test BT_TC_2_4, cancel discovery 10 times\n");
    NN_LOG("-----------------------------------------------------------------------\n");
    for (int i = 0; i < 10; i++)
    {
        nn::bluetooth::CancelDiscovery();
    }
    SUCCEED();
}
