﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#include <cstdio>

#include <nnt/nntest.h>
#include <nn/nn_Log.h>

#include <nn/os.h>
#include <nn/os/os_SystemEventTypes.h>
#include <nn/settings/system/settings_BluetoothDevices.h>
#include <nn/bluetooth/bluetooth_Api.h>
#include <nn/bluetooth/bluetooth_TypesHal.h>

#include "../../Common/testBluetooth_TargetDevices.h"
#include "../testBluetooth_Bonding.h"

using namespace nn::bluetooth;

extern nn::os::SystemEventType g_SystemEvent;
extern nn::os::SystemEventType g_SystemEventForHid;
extern nn::os::MultiWaitType g_MultiWait;
extern nn::os::MultiWaitHolderType g_HolderForBluetoothCore;
extern nn::os::MultiWaitHolderType g_HolderForHid;

void BondingTest::SetUpTestCase()
{
    NN_LOG("\n\t Setting up test case for BondingTest BT_TC_3_3\n");
    NN_LOG("-----------------------------------------------------------------------\n");
    NN_LOG("InitializeBluetoothDriver.!\n");
    nn::bluetooth::InitializeBluetoothDriver();

    nnt::bluetooth::CheckFilterOption();

    NN_LOG("InitializeBluetooth...\n");
    ASSERT_TRUE( nn::bluetooth::InitializeBluetooth(&g_SystemEvent).IsSuccess() );
    NN_LOG("EnableBluetooth...\n");
    ASSERT_TRUE( nn::bluetooth::EnableBluetooth().IsSuccess() );
    NN_LOG("InitializeHid...\n");
    ASSERT_TRUE( nn::bluetooth::InitializeHid(&g_SystemEventForHid).IsSuccess() );

    NN_LOG("InitializeMultiWaitHolder EVENTTYPE_FROM_CORE...\n");
    nn::os::InitializeMultiWaitHolder(&g_HolderForBluetoothCore, &g_SystemEvent);
    nn::os::SetMultiWaitHolderUserData(&g_HolderForBluetoothCore, BondingTestSpace::EVENTTYPE_FROM_CORE);

    NN_LOG("InitializeMultiWaitHolder EVENTTYPE_FROM_HID...\n");
    nn::os::InitializeMultiWaitHolder(&g_HolderForHid, &g_SystemEventForHid);
    nn::os::SetMultiWaitHolderUserData(&g_HolderForHid, BondingTestSpace::EVENTTYPE_FROM_HID);

    NN_LOG("InitializeMultiWait...\n");
    nn::os::InitializeMultiWait(&g_MultiWait);
    nn::os::LinkMultiWaitHolder(&g_MultiWait, &g_HolderForBluetoothCore);
    nn::os::LinkMultiWaitHolder(&g_MultiWait, &g_HolderForHid);

    NN_LOG("Leaving SetUpTestCase.\n");
}

void BondingTest::TearDownTestCase()
{
    nn::os::UnlinkAllMultiWaitHolder(&g_MultiWait);
    nn::os::FinalizeMultiWaitHolder(&g_HolderForBluetoothCore);
    nn::os::FinalizeMultiWaitHolder(&g_HolderForHid);
    nn::os::FinalizeMultiWait(&g_MultiWait);
    nn::os::DestroySystemEvent(&g_SystemEvent);
    nn::os::DestroySystemEvent(&g_SystemEventForHid);

    nn::bluetooth::DisableBluetooth();
    nn::bluetooth::CleanupBluetooth();
    nn::bluetooth::CleanupHid();
    nn::bluetooth::FinalizeBluetoothDriver();
}

TEST_F(BondingTest, BT_TC_3_3)
{
    nn::bluetooth::BluetoothAddress foundAddr;
    nn::bluetooth::BluetoothName foundName;

    NN_LOG("\n\t Please long press reset button on a controller to begin bonding\n");
    NN_LOG("-----------------------------------------------------------------------\n");
    nn::os::SleepThread( nn::TimeSpan::FromMilliSeconds(3000));
    NN_LOG("Starting discovery ...\n");
    nn::bluetooth::StartDiscovery();

    NN_LOG("WaitDeviceFoundCallback ...\n");
    BondingTestSpace::WaitDeviceFoundCallback(&foundAddr, &foundName);

    for (;;)
    {
        NN_LOG("nn::os::TimedWaitAny ...\n");
        nn::os::MultiWaitHolderType* holder = nn::os::WaitAny(&g_MultiWait);

        NN_LOG("GetMultiWaitHolderUserData ...\n");
        int holderType = nn::os::GetMultiWaitHolderUserData(holder);
        if (holderType == BondingTestSpace::EVENTTYPE_FROM_CORE)
        {
            NN_LOG("EVENTTYPE_FROM_CORE ...\n");
            nn::os::TryWaitSystemEvent(&g_SystemEvent);
            nn::bluetooth::EventType type;
            uint8_t buffer[nn::bluetooth::BUFFER_SIZE_OF_CORE_OUT] = {0};
            nn::bluetooth::GetEventInfo(&type, buffer, nn::bluetooth::BUFFER_SIZE_OF_CORE_OUT);
            BondingTestSpace::PrintEventInfo(type, buffer);
            if (type == nn::bluetooth::EventFromSspRequestCallback)
            {
                nn::bluetooth::InfoFromSspRequestCallback* info = reinterpret_cast<nn::bluetooth::InfoFromSspRequestCallback*>(buffer);
                nn::bluetooth::CancelBond(&info->bluetoothAddress);
                break;
            }
            else if (type == nn::bluetooth::EventFromBondStateChangedCallback)
            {
                BluetoothBondState bondState = reinterpret_cast<nn::bluetooth::InfoFromBondStateChangedCallback*>(buffer)->state;
                EXPECT_EQ(bondState, BT_BOND_STATE_BONDING);
                break;
            }
            else if (type == nn::bluetooth::EventFromDiscoveryStateChangedCallback)
            {
                BluetoothDiscoveryState state = reinterpret_cast<const nn::bluetooth::InfoFromDiscoveryStateChangedCallback*>(&buffer[0])->state;
                if (state == BT_DISCOVERY_STOPPED)
                {
                    nn::bluetooth::CreateBond(&foundAddr, nn::bluetooth::NoPreference);
                }
            }
        }
        else
        {
            // Unexpected cb
            NN_LOG("Unexpected nn::bluetooth event holder type [%d].\n", holderType);
            break;
        }
    }

    for (;;)
    {
        NN_LOG("nn::os::WaitAny ...\n");
        nn::os::MultiWaitHolderType* holder = nn::os::WaitAny(&g_MultiWait);
        NN_LOG("GetMultiWaitHolderUserData ...\n");
        int holderType = nn::os::GetMultiWaitHolderUserData(holder);
        switch (holderType)
        {
        case BondingTestSpace::EVENTTYPE_FROM_CORE:
            {
                nn::os::TryWaitSystemEvent(&g_SystemEvent);
                nn::bluetooth::EventType type;
                uint8_t buffer[nn::bluetooth::BUFFER_SIZE_OF_CORE_OUT] = {0};
                nn::bluetooth::GetEventInfo(&type, buffer, nn::bluetooth::BUFFER_SIZE_OF_CORE_OUT);
                BondingTestSpace::PrintEventInfo(type, buffer);
                if (type == nn::bluetooth::EventFromBondStateChangedCallback)
                {
                    BluetoothBondState bondState = reinterpret_cast<nn::bluetooth::InfoFromBondStateChangedCallback*>(buffer)->state;
                    EXPECT_EQ(bondState, BT_BOND_STATE_NONE) << "Unexpected Bonding state, expected BT_BOND_STATE_NONE.";
                    //goto OUT_OF_LOOP_1;
                }
                else if (type == nn::bluetooth::EventFromSspRequestCallback)
                {
                    NN_LOG("\n\t Canceled bond successfully!\n");
                    NN_LOG("-----------------------------------------------------------------------\n");
                    SUCCEED();
                    goto OUT_OF_LOOP_1;
                }
                else
                {
                    // Unexpected cb
                    NN_LOG("Unexpected bonding state changed callback [%d].\n", type);
                    FAIL();
                    goto OUT_OF_LOOP_1;
                }
            }
            break;
        case BondingTestSpace::EVENTTYPE_FROM_HID:
            {
                NN_LOG("Did not expect a HID event at this time.\n");
            }
            break;
        default: NN_UNEXPECTED_DEFAULT;
        }
    }


OUT_OF_LOOP_1:
    NN_LOG("Test exiting now...\n");
}
