﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nn/os.h>
#include <nnt/nntest.h>

#include <nn/nn_Log.h>

#include "../Common/testBle_Common.h"
#include "../Common/testBle_ScanCommon.h"

#include <nn/bluetooth/bluetooth_LeApi.h>

//!< Radio On 中、Scan を開始せずに結果を取得した場合、結果の数は 0
TEST(TestBleScan, GetResultWithoutScan)
{
    nnt::bluetooth::common::SetUp();
    nnt::bluetooth::scan::SetUp();

    nn::bluetooth::BleScanResult scanResult[nn::bluetooth::BleScanResultCountMax];

    int resultCount = nn::bluetooth::GetBleScanResult(scanResult, NN_ARRAY_SIZE(scanResult));

    EXPECT_EQ(resultCount, 0);

    nnt::bluetooth::common::TearDown();
    nnt::bluetooth::scan::TearDown();
}

//!< Radio Off 中、Scan を開始せずに結果を取得した場合、結果の数は 0
TEST(TestBleScan, GetResultWithoutScanRadioOff)
{
    nnt::bluetooth::common::SetUp();
    nnt::bluetooth::scan::SetUp();

    nnt::bluetooth::common::EnableRadio(false);

    nn::bluetooth::BleScanResult scanResult[nn::bluetooth::BleScanResultCountMax];

    int resultCount = nn::bluetooth::GetBleScanResult(scanResult, NN_ARRAY_SIZE(scanResult));

    EXPECT_EQ(resultCount, 0);

    nnt::bluetooth::common::TearDown();
    nnt::bluetooth::scan::TearDown();
}

//!< Radio On 中、Scan を開始していない状態で、Scan を停止しても成功する
TEST(TestBleScan, StopScanWithoutScan)
{
    nnt::bluetooth::common::SetUp();
    nnt::bluetooth::scan::SetUp();

    nn::Result result = nn::ResultSuccess();

    while (NN_STATIC_CONDITION(true))
    {
        result = nn::bluetooth::StopBleScanGeneral();

        if (nn::bluetooth::ResultSystemBusy().Includes(result) ||
            nn::bluetooth::ResultGeneralError().Includes(result))           // General Error は返さないようにしたい
        {
            nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(5));
        }
        else
        {
            break;
        }
    }

    EXPECT_TRUE(result.IsSuccess());

    while (NN_STATIC_CONDITION(true))
    {
        result = nn::bluetooth::StopBleScanSmartDevice();

        if (nn::bluetooth::ResultSystemBusy().Includes(result) ||
            nn::bluetooth::ResultGeneralError().Includes(result))           // General Error は返さないようにしたい
        {
            nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(5));
        }
        else
        {
            break;
        }
    }

    EXPECT_TRUE(result.IsSuccess());

    nnt::bluetooth::common::TearDown();
    nnt::bluetooth::scan::TearDown();
}

//!< Radio Off 中、Scan を開始していない状態で、Scan を停止しようとすると、nn::bluetooth::ResultBluetoothOff()
TEST(TestBleScan, StopScanWithoutScanRadioOff)
{
    nnt::bluetooth::common::SetUp();
    nnt::bluetooth::scan::SetUp();

    nnt::bluetooth::common::EnableRadio(false);

    nn::Result result = nn::ResultSuccess();

    while (NN_STATIC_CONDITION(true))
    {
        result = nn::bluetooth::StopBleScanGeneral();

        if (nn::bluetooth::ResultSystemBusy().Includes(result) ||
            nn::bluetooth::ResultGeneralError().Includes(result))           // General Error は返さないようにしたい
        {
            nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(5));
        }
        else
        {
            break;
        }
    }

    EXPECT_TRUE(nn::bluetooth::ResultBluetoothOff().Includes(result));

    while (NN_STATIC_CONDITION(true))
    {
        result = nn::bluetooth::StopBleScanSmartDevice();

        if (nn::bluetooth::ResultSystemBusy().Includes(result) ||
            nn::bluetooth::ResultGeneralError().Includes(result))           // General Error は返さないようにしたい
        {
            nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(5));
        }
        else
        {
            break;
        }
    }

    EXPECT_TRUE(nn::bluetooth::ResultBluetoothOff().Includes(result));

    nnt::bluetooth::common::TearDown();
    nnt::bluetooth::scan::TearDown();
}
