﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nn/os.h>
#include <nnt/nntest.h>

#include <nn/bluetooth/bluetooth_LeApi.h>

#include "testBle_Common.h"
#include "testBle_ScanCommon.h"

namespace
{
    bool g_IsInitialized = false;               // Initialize() が呼ばれたかどうか

    nn::os::SystemEventType g_BleScanEvent;     //!< BLE Scan イベント

    nn::os::EventType* gp_BleScanEventInternal; //!< 各テストから渡される内部向け BLE Scan イベント

    const size_t ThreadStackSize = 4096;
    NN_OS_ALIGNAS_THREAD_STACK char g_EventHandleThreadStack[ThreadStackSize];
    nn::os::ThreadType g_EventHandleThread;
    void EventHandleThreadFunction(void* arg);
    volatile bool g_IsToFinishEventHandleThread;

    void Initialize()
    {
        if (!g_IsInitialized)
        {
            nn::bluetooth::AcquireBleScanEvent(&g_BleScanEvent);

            g_IsInitialized = true;
        }
    }

    void StopScan()
    {
        nn::Result result = nn::ResultSuccess();

        int retryCount = 0;

        while (NN_STATIC_CONDITION(true))
        {
            EXPECT_LT(retryCount, 5);       // 5 回までリトライ許容

            result = nn::bluetooth::StopBleScanGeneral();

            if (result.IsSuccess())
            {
                break;
            }
            else if (nn::bluetooth::ResultBluetoothOff().Includes(result))
            {
                nnt::bluetooth::common::EnableRadio(true);
            }

            nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(5));
            retryCount++;
        }

        retryCount = 0;

        while (NN_STATIC_CONDITION(true))
        {
            EXPECT_LT(retryCount, 5);       // 5 回までリトライ許容

            result = nn::bluetooth::StopBleScanSmartDevice();

            if (result.IsSuccess())
            {
                break;
            }
            else if (nn::bluetooth::ResultBluetoothOff().Includes(result))
            {
                nnt::bluetooth::common::EnableRadio(true);
            }

            nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(5));
            retryCount++;
        }
    }

    void EventHandleThreadFunction(void* arg)
    {
        NN_UNUSED(arg);

        while (!g_IsToFinishEventHandleThread)
        {
            if (nn::os::TryWaitSystemEvent(&g_BleScanEvent))
            {
                if (gp_BleScanEventInternal != nullptr)
                {
                    nn::os::SignalEvent(gp_BleScanEventInternal);
                }
            }

            nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(50));
        }
    }
}

namespace nnt { namespace bluetooth { namespace scan {
    void Finalize()
    {
        if (g_IsInitialized)
        {
            nn::os::DestroySystemEvent(&g_BleScanEvent);

            g_IsInitialized = false;
        }
    }

    void SetUp(nn::os::EventType *pScanEvent)
    {
        Initialize();

        StopScan();

        gp_BleScanEventInternal = pScanEvent;

        if (gp_BleScanEventInternal != nullptr)
        {
            g_IsToFinishEventHandleThread = false;

            auto result = nn::os::CreateThread(&g_EventHandleThread, EventHandleThreadFunction, nullptr, g_EventHandleThreadStack, ThreadStackSize, nn::os::DefaultThreadPriority);
            EXPECT_TRUE(result.IsSuccess());

            nn::os::StartThread(&g_EventHandleThread);
        }
    }

    void TearDown()
    {
        StopScan();

        if (gp_BleScanEventInternal != nullptr)
        {
            g_IsToFinishEventHandleThread = true;
            nn::os::WaitThread(&g_EventHandleThread);
            nn::os::DestroyThread(&g_EventHandleThread);
        }
    }
}}}; // namespace nnt::bluetooth::scan
