﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "../../Common/testBcat_Common.h"

namespace
{
    // td1, BcatBenchmarkApp01
    const nn::ApplicationId AppId = {0x01005c00015f2000};
}

void DumpProgress(const nn::bcat::DeliveryCacheProgress& progress)  NN_NOEXCEPT
{
    switch (progress.GetStatus())
    {
    case nn::bcat::DeliveryCacheProgressStatus_Queued:
        NN_LOG("Progress: status = queued\n");
        break;
    case nn::bcat::DeliveryCacheProgressStatus_Connect:
        NN_LOG("Progress: status = connect\n");
        break;
    case nn::bcat::DeliveryCacheProgressStatus_ProcessList:
        NN_LOG("Progress: status = process list\n");
        break;
    case nn::bcat::DeliveryCacheProgressStatus_Download:
        NN_LOG("Progress: status = download, current = (%s/%s, %lld/%lld), whole = (%lld/%lld)\n",
            progress.GetCurrentDirectoryName().value, progress.GetCurrentFileName().value,
            progress.GetCurrentDownloaded(), progress.GetCurrentTotal(),
            progress.GetWholeDownloaded(), progress.GetWholeTotal());
        break;
    case nn::bcat::DeliveryCacheProgressStatus_Commit:
        NN_LOG("Progress: status = commit, current = (%s), whole = (%lld/%lld)\n",
            progress.GetCurrentDirectoryName().value,
            progress.GetWholeDownloaded(), progress.GetWholeTotal());
        break;
    case nn::bcat::DeliveryCacheProgressStatus_Done:
        NN_LOG("Progress: status = done, whole = (%lld/%lld), result = %03d-%04d\n",
            progress.GetWholeDownloaded(), progress.GetWholeTotal(),
            progress.GetResult().GetModule(), progress.GetResult().GetDescription());
        break;
    default:
        break;
    }
}

TEST(DirectorySync, Initialize)
{
    nn::bcat::Initialize();
    nn::nifm::Initialize();
}

TEST(DirectorySync, Do)
{
    nn::nifm::SubmitNetworkRequestAndWait();

    ASSERT_TRUE(nn::nifm::IsNetworkAvailable());

    nn::bcat::ClearDeliveryCacheStorage(AppId);

    struct
    {
        const char* name;
        size_t size;
    }
    dirInfos[] =
    {
        {"dir1", 8 * 1024 * 1024},
        {"dir2", 16 * 1024 * 1024},
        {"dir3", 30 * 1024 * 1024},
        {"dir_empty", 0},
        {"dir_not_exist"}
    };

    for (int i = 0; i < NN_ARRAY_SIZE(dirInfos); i++)
    {
        nn::bcat::DeliveryCacheProgress progress;

        ASSERT_RESULT_SUCCESS(nn::bcat::RequestSyncDeliveryCache(&progress, dirInfos[i].name));

        while (progress.GetStatus() != nn::bcat::DeliveryCacheProgressStatus_Done)
        {
            progress.Update();

            DumpProgress(progress);

            nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(500));
        }

        if (nn::util::Strncmp(dirInfos[i].name, "dir_not_exist", sizeof (nn::bcat::DirectoryName::value)) == 0)
        {
            ASSERT_RESULT(progress.GetResult(), nn::bcat::ResultNotFound);
        }
        else
        {
            EXPECT_EQ(progress.GetWholeDownloaded(), progress.GetWholeTotal());
            ASSERT_RESULT_SUCCESS(progress.GetResult());
        }
    }

    {
        nn::bcat::DeliveryCacheProgress progress;

        ASSERT_RESULT_SUCCESS(nn::bcat::RequestSyncDeliveryCache(&progress));

        while (progress.GetStatus() != nn::bcat::DeliveryCacheProgressStatus_Done)
        {
            progress.Update();

            DumpProgress(progress);

            nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(500));
        }

        // 最後に全体同期で dir4/4mb.bin が同期されるはず。
        EXPECT_EQ(progress.GetWholeDownloaded(), 4 * 1024 * 1024ll);

        EXPECT_EQ(progress.GetWholeDownloaded(), progress.GetWholeTotal());
        ASSERT_RESULT_SUCCESS(progress.GetResult());
    }

    nn::nifm::CancelNetworkRequest();
}
