﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "../../Common/testBcat_Common.h"

#include <nn/bcat/detail/service/core/bcat_DeliveryCacheStorageManager.h>

using namespace nn::bcat::detail::service::core;

namespace
{
    const nn::ApplicationId AppIds[DeliveryCacheStorageManager::MountCountMax] =
    {
        {0x0100db60000a8000},
        {0x01006080000b4000},
        {0x01006610000b6000},
        {0x01003170000b8000}
    };
}

TEST(DeliveryCacheStorageManager, Initialize)
{
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::MountHost("bcat-dc-00", "C:\\siglo\\bcat-dc-00"));
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::MountHost("bcat-dc-01", "C:\\siglo\\bcat-dc-01"));
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::MountHost("bcat-dc-02", "C:\\siglo\\bcat-dc-02"));
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::MountHost("bcat-dc-03", "C:\\siglo\\bcat-dc-03"));
    DeliveryCacheStorageManager::GetInstance().EnableTestMode();
}

TEST(DeliveryCacheStorageManager, MountUnmount)
{
    for (int i = 0; i < DeliveryCacheStorageManager::MountCountMax; i++)
    {
        ASSERT_RESULT_SUCCESS(DeliveryCacheStorageManager::GetInstance().Mount(AppIds[i]));
        ASSERT_RESULT(DeliveryCacheStorageManager::GetInstance().Mount(AppIds[i]), nn::bcat::ResultLocked);
    }

    for (int i = 0; i < DeliveryCacheStorageManager::MountCountMax; i++)
    {
        DeliveryCacheStorageManager::GetInstance().Unmount(AppIds[i]);
    }
}

TEST(DeliveryCacheStorageManager, Do)
{
    // 登録順なので AppIds[1] → bcat-save-00, AppIds[0] → bcat-save-01 となる。
    ASSERT_RESULT_SUCCESS(DeliveryCacheStorageManager::GetInstance().Mount(AppIds[1]));
    ASSERT_RESULT_SUCCESS(DeliveryCacheStorageManager::GetInstance().Mount(AppIds[0]));

    NN_UTIL_SCOPE_EXIT
    {
        DeliveryCacheStorageManager::GetInstance().Unmount(AppIds[1]);
        DeliveryCacheStorageManager::GetInstance().Unmount(AppIds[0]);
    };

    // ストレージにアクセスできるかどうかの確認。
    {
        nn::Result result = nn::fs::CreateDirectory("bcat-dc-00:/directories");
        EXPECT_TRUE(result.IsSuccess() || nn::fs::ResultPathAlreadyExists::Includes(result));

        ASSERT_RESULT_SUCCESS(DeliveryCacheStorageManager::GetInstance().Commit(AppIds[1]));
    }
    {
        nn::Result result = nn::fs::CreateDirectory("bcat-dc-01:/directories");
        EXPECT_TRUE(result.IsSuccess() || nn::fs::ResultPathAlreadyExists::Includes(result));

        ASSERT_RESULT_SUCCESS(DeliveryCacheStorageManager::GetInstance().Commit(AppIds[0]));
    }

    char path[256] = {};

    DeliveryCacheStorageManager::GetInstance().MakeFilePath(path, sizeof (path), AppIds[1], "dir1", "file1.txt");
    EXPECT_TRUE(nn::util::Strncmp(path, "bcat-dc-00:/directories/dir1/files/file1.txt", sizeof (path)) == 0);
    DeliveryCacheStorageManager::GetInstance().MakeFilePath(path, sizeof (path), AppIds[0], "dir1", "file1.txt");
    EXPECT_TRUE(nn::util::Strncmp(path, "bcat-dc-01:/directories/dir1/files/file1.txt", sizeof (path)) == 0);

    DeliveryCacheStorageManager::GetInstance().MakeFileMetaPath(path, sizeof (path), AppIds[1], "dir1");
    EXPECT_TRUE(nn::util::Strncmp(path, "bcat-dc-00:/directories/dir1/files.meta", sizeof (path)) == 0);
    DeliveryCacheStorageManager::GetInstance().MakeFileMetaPath(path, sizeof (path), AppIds[0], "dir1");
    EXPECT_TRUE(nn::util::Strncmp(path, "bcat-dc-01:/directories/dir1/files.meta", sizeof (path)) == 0);

    DeliveryCacheStorageManager::GetInstance().MakeDirectoryPath(path, sizeof (path), AppIds[1], "dir1");
    EXPECT_TRUE(nn::util::Strncmp(path, "bcat-dc-00:/directories/dir1", sizeof (path)) == 0);
    DeliveryCacheStorageManager::GetInstance().MakeDirectoryPath(path, sizeof (path), AppIds[0], "dir1");
    EXPECT_TRUE(nn::util::Strncmp(path, "bcat-dc-01:/directories/dir1", sizeof (path)) == 0);

    DeliveryCacheStorageManager::GetInstance().MakeDirectoryMetaPath(path, sizeof (path), AppIds[1]);
    EXPECT_TRUE(nn::util::Strncmp(path, "bcat-dc-00:/directories.meta", sizeof (path)) == 0);
    DeliveryCacheStorageManager::GetInstance().MakeDirectoryMetaPath(path, sizeof (path), AppIds[0]);
    EXPECT_TRUE(nn::util::Strncmp(path, "bcat-dc-01:/directories.meta", sizeof (path)) == 0);
}
