﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "../../Common/testBcat_Common.h"

namespace
{
    const nn::ApplicationId AppId = {0x0100db60000a8000};
    const uint32_t AppVersion = 0;

    const char* Passphrase = "e3ac0d7a05b67c3e228369dff4aba298c645e8bd356ae45ac4910fac93d20d13";
}

void DumpTaskInfos(const nn::bcat::TaskInfo* infos, int count) NN_NOEXCEPT
{
    NN_LOG("--------------------------------------------------\n");
    NN_LOG("TaskInfo\n");
    NN_LOG("--------------------------------------------------\n");
    NN_LOG("Count = %d\n", count);

    for (int i = 0; i < count; i++)
    {
        nn::Result result;
        std::memcpy(&result, &infos[i].lastError, sizeof (result));

        NN_LOG("--------------------------------------------------\n");
        NN_LOG("TaskInfo[%d]\n", i);
        NN_LOG("--------------------------------------------------\n");
        NN_LOG("appId:\n");
        NN_LOG("    %016llx\n", infos[i].appId.value);
        NN_LOG("appVersion:\n");
        NN_LOG("    %u\n", infos[i].appVersion);
        NN_LOG("status:\n");
        NN_LOG("    %c\n", infos[i].status != nn::bcat::TaskStatus_Empty ? infos[i].status : '-');
        NN_LOG("subscription:\n");
        NN_LOG("    %c\n", infos[i].subscription != nn::bcat::SubscriptionStatus_Empty ? infos[i].subscription : '-');
        NN_LOG("lastError:\n");
        NN_LOG("    %03d-%04d\n", result.GetModule(), result.GetDescription());
        NN_LOG("registrationTime:\n");
        NN_LOG("    %lld\n", infos[i].registrationTime.value);
        NN_LOG("nextRunnableTime:\n");
        NN_LOG("    %lld\n", infos[i].nextRunnableTime.value);
    }
}

void DumpPushNotificationLogs(const nn::bcat::PushNotificationLog* logs, int count) NN_NOEXCEPT
{
    NN_LOG("--------------------------------------------------\n");
    NN_LOG("PushNotificationLog\n");
    NN_LOG("--------------------------------------------------\n");
    NN_LOG("Count = %d\n", count);

    for (int i = 0; i < count; i++)
    {
        NN_LOG("--------------------------------------------------\n");
        NN_LOG("PushNotificationLog[%d]\n", i);
        NN_LOG("--------------------------------------------------\n");
        NN_LOG("appId:\n");
        NN_LOG("    %016llx\n", logs[i].appId.value);
        NN_LOG("receivedTime:\n");
        NN_LOG("    %lld\n", logs[i].receivedTime.value);
    }
}

void DumpProgress(const nn::bcat::DeliveryCacheProgress& progress)  NN_NOEXCEPT
{
    switch (progress.GetStatus())
    {
    case nn::bcat::DeliveryCacheProgressStatus_Queued:
        NN_LOG("Progress: status = queued\n");
        break;
    case nn::bcat::DeliveryCacheProgressStatus_Connect:
        NN_LOG("Progress: status = connect\n");
        break;
    case nn::bcat::DeliveryCacheProgressStatus_ProcessList:
        NN_LOG("Progress: status = process list\n");
        break;
    case nn::bcat::DeliveryCacheProgressStatus_Download:
        NN_LOG("Progress: status = download, current = (%s/%s, %lld/%lld), whole = (%lld/%lld)\n",
            progress.GetCurrentDirectoryName().value, progress.GetCurrentFileName().value,
            progress.GetCurrentDownloaded(), progress.GetCurrentTotal(),
            progress.GetWholeDownloaded(), progress.GetWholeTotal());
        break;
    case nn::bcat::DeliveryCacheProgressStatus_Commit:
        NN_LOG("Progress: status = commit, current = (%s), whole = (%lld/%lld)\n",
            progress.GetCurrentDirectoryName().value,
            progress.GetWholeDownloaded(), progress.GetWholeTotal());
        break;
    case nn::bcat::DeliveryCacheProgressStatus_Done:
        NN_LOG("Progress: status = done, whole = (%lld/%lld), result = %03d-%04d\n",
            progress.GetWholeDownloaded(), progress.GetWholeTotal(),
            progress.GetResult().GetModule(), progress.GetResult().GetDescription());
        break;
    default:
        break;
    }
}

TEST(Basic, Initialize)
{
    nn::bcat::Initialize();
    nn::nifm::Initialize();
}

TEST(Basic, Mount)
{
    for (int i = 0; i < 100; i++)
    {
        nn::ApplicationId appId = {static_cast<uint64_t>(i + 1)};

        ASSERT_RESULT(nn::bcat::MountDeliveryCacheStorage(appId), nn::bcat::ResultStorageNotFound);
    }
}

TEST(Basic, Register)
{
    ASSERT_RESULT_SUCCESS(nn::bcat::SetPassphrase(AppId, Passphrase));
    ASSERT_RESULT_SUCCESS(nn::bcat::RegisterBackgroundDeliveryTask(AppId, AppVersion));

    static nn::bcat::TaskInfo s_Infos[nn::bcat::TaskCountMax] = {};
    int count = 0;

    ASSERT_RESULT_SUCCESS(nn::bcat::EnumerateBackgroundDeliveryTask(&count, s_Infos, NN_ARRAY_SIZE(s_Infos)));
    EXPECT_TRUE(count > 0);

    DumpTaskInfos(s_Infos, count);

    bool isFound = false;

    for (int i = 0; i < count; i++)
    {
        if (s_Infos[i].appId == AppId)
        {
            isFound = true;
            break;
        }
    }

    EXPECT_TRUE(isFound);

    ASSERT_RESULT_SUCCESS(nn::bcat::UnregisterBackgroundDeliveryTask(AppId));

    ASSERT_RESULT_SUCCESS(nn::bcat::EnumerateBackgroundDeliveryTask(&count, s_Infos, NN_ARRAY_SIZE(s_Infos)));

    if (count > 0)
    {
        isFound = false;

        for (int i = 0; i < count; i++)
        {
            if (s_Infos[i].appId == AppId)
            {
                isFound = true;
                break;
            }
        }

        EXPECT_FALSE(isFound);
    }
}

TEST(Basic, Cancel1)
{
    nn::nifm::SubmitNetworkRequestAndWait();

    ASSERT_TRUE(nn::nifm::IsNetworkAvailable());

    ASSERT_RESULT_SUCCESS(nn::bcat::RegisterBackgroundDeliveryTask(AppId, AppVersion));

    nn::bcat::CancelSyncDeliveryCacheRequest();

    for (int i = 0; i < 20; i++)
    {
        NN_LOG("Cancel(%d)...\n", i);

        nn::bcat::DeliveryCacheProgress progress;

        ASSERT_RESULT_SUCCESS(nn::bcat::RequestSyncDeliveryCache(&progress, AppId, AppVersion));

        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(i * 100));

        nn::bcat::CancelSyncDeliveryCacheRequest();

        while (progress.GetStatus() != nn::bcat::DeliveryCacheProgressStatus_Done)
        {
            progress.Update();

            DumpProgress(progress);

            nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(500));
        }

        if (progress.GetResult().IsFailure())
        {
            ASSERT_RESULT(progress.GetResult(), nn::bcat::ResultCanceledByUser);
        }
    }

    nn::nifm::CancelNetworkRequest();
}

TEST(Basic, Cancel2)
{
    nn::nifm::SubmitNetworkRequestAndWait();

    ASSERT_TRUE(nn::nifm::IsNetworkAvailable());

    ASSERT_RESULT_SUCCESS(nn::bcat::RegisterBackgroundDeliveryTask(AppId, AppVersion));

    nn::bcat::DeliveryCacheProgress progresses[4];

    for (int i = 0; i < NN_ARRAY_SIZE(progresses); i++)
    {
        ASSERT_RESULT_SUCCESS(nn::bcat::RequestSyncDeliveryCache(&progresses[i], AppId, AppVersion));
    }

    nn::bcat::CancelSyncDeliveryCacheRequest();

    for (int i = 0; i < NN_ARRAY_SIZE(progresses); i++)
    {
        while (progresses[i].GetStatus() != nn::bcat::DeliveryCacheProgressStatus_Done)
        {
            progresses[i].Update();
            nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(10));
        }
    }

    for (int i = 0; i < NN_ARRAY_SIZE(progresses); i++)
    {
        ASSERT_RESULT(progresses[i].GetResult(), nn::bcat::ResultCanceledByUser);
    }

    nn::nifm::CancelNetworkRequest();
}

TEST(Basic, Sync)
{
    nn::nifm::SubmitNetworkRequestAndWait();

    ASSERT_TRUE(nn::nifm::IsNetworkAvailable());

    ASSERT_RESULT_SUCCESS(nn::bcat::RegisterBackgroundDeliveryTask(AppId, AppVersion));

    nn::bcat::DeliveryCacheProgress progress;

    ASSERT_RESULT_SUCCESS(nn::bcat::RequestSyncDeliveryCache(&progress, AppId, AppVersion));

    if (progress.GetStatus() == nn::bcat::DeliveryCacheProgressStatus_Queued)
    {
        nn::os::Tick start = nn::os::GetSystemTick();

        while (progress.GetStatus() == nn::bcat::DeliveryCacheProgressStatus_Queued)
        {
            progress.Update();

            DumpProgress(progress);

            nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(500));

            // タイムアウトを 30 秒とする。
            if ((nn::os::GetSystemTick() - start).ToTimeSpan().GetSeconds() >= 30)
            {
                NN_LOG("Timed out.\n");
                break;
            }
        }
    }

    if (progress.GetStatus() != nn::bcat::DeliveryCacheProgressStatus_Queued)
    {
        while (progress.GetStatus() != nn::bcat::DeliveryCacheProgressStatus_Done)
        {
            progress.Update();

            DumpProgress(progress);

            nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(500));
        }

        ASSERT_RESULT_SUCCESS(progress.GetResult());

        nn::bcat::CancelSyncDeliveryCacheRequest();

        // 処理完了後、キャンセルによって結果が変わらないことを確認。
        progress.Update();
        ASSERT_RESULT_SUCCESS(progress.GetResult());
    }

    nn::nifm::CancelNetworkRequest();
}

TEST(Basic, CancelBackgroundTask)
{
    // ネットワーク接続要求を事前に出しておくことで、BG のタスク実行を阻害する。
    nn::nifm::SubmitNetworkRequestAndWait();

    // タスク新規登録状態にする。
    ASSERT_RESULT_SUCCESS(nn::bcat::UnregisterBackgroundDeliveryTask(AppId));
    ASSERT_RESULT_SUCCESS(nn::bcat::RegisterBackgroundDeliveryTask(AppId, AppVersion));

    NN_LOG("Sleep 5 seconds...\n");

    nn::os::SleepThread(nn::TimeSpan::FromSeconds(5));

    nn::bcat::DeliveryCacheProgress progress;

    ASSERT_RESULT_SUCCESS(nn::bcat::RequestSyncDeliveryCache(&progress, AppId, AppVersion));

    if (progress.GetStatus() == nn::bcat::DeliveryCacheProgressStatus_Queued)
    {
        nn::os::Tick start = nn::os::GetSystemTick();

        while (progress.GetStatus() == nn::bcat::DeliveryCacheProgressStatus_Queued)
        {
            progress.Update();

            DumpProgress(progress);

            nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(500));

            // 実行に移らない場合、失敗とする。
            ASSERT_TRUE((nn::os::GetSystemTick() - start).ToTimeSpan().GetSeconds() < 10);
        }
    }

    if (progress.GetStatus() != nn::bcat::DeliveryCacheProgressStatus_Queued)
    {
        while (progress.GetStatus() != nn::bcat::DeliveryCacheProgressStatus_Done)
        {
            progress.Update();

            DumpProgress(progress);

            nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(500));
        }
    }

    nn::nifm::CancelNetworkRequest();
}

TEST(Basic, PushNotificationLog)
{
    static nn::bcat::PushNotificationLog s_Logs[nn::bcat::PushNotificationLogCountMax] = {};
    int count = 0;

    ASSERT_RESULT_SUCCESS(nn::bcat::GetPushNotificationLog(&count, s_Logs, NN_ARRAY_SIZE(s_Logs)));

    DumpPushNotificationLogs(s_Logs, count);
}
