﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nn/nn_Abort.h>
#include <nn/nn_Log.h>
#include <nn/init.h>
#include <nn/os.h>
#include <nn/oe.h>
#include <nn/oe/oe_DebugApis.h>
#include <nn/oe/oe_HdcpApis.private.h>
#include <nn/applet/applet.h>
#include <nn/am/am_Result.h>
#include <nn/sf/cmif/sf_CmifResult.h>
#include <nn/audio.h>

#include <cstdlib>
#include "../Common/applet_Configs.h"

#pragma clang optimize off

//-----------------------------------------------------------------------------

void MainSequenceTestOfLibraryAppletInvoking() NN_NOEXCEPT;

#define WaitForFocusStateChanged(expected) WaitForFocusStateChangedImpl(expected, __LINE__)
#define CheckAndProcessMessage(expected) CheckAndProcessMessageImpl(expected, __LINE__)


//-----------------------------------------------------------------------------
//  テスト用のメインシーケンス
//
void CheckAndProcessMessageImpl(nn::oe::Message expectMessage, int line) NN_NOEXCEPT
{
retry:
    // テスト用に TryPopNotificationMessage() も呼ぶ
    nn::oe::Message message;
    bool ret = nn::oe::TryPopNotificationMessage(&message);
    if (!ret)
    {
        message = nn::oe::PopNotificationMessage();
    }

    switch (message)
    {
        case nn::oe::MessageFocusStateChanged:
                if (!(message == expectMessage))
                {
                    goto retry;
                }
                break;

        case nn::oe::MessageResume:
                NN_LOG("APL: Received MessageResume\n");
                break;

        case nn::oe::MessageOperationModeChanged:
                NN_LOG("APL: Received MessageOperationModeChanged\n");
                NN_LOG("APL: Currently Operation Mode is %s Mode\n", nn::oe::GetOperationMode() == nn::oe::OperationMode_Handheld ? "Handheld" : "Console");
                break;

        case nn::oe::MessagePerformanceModeChanged:
                NN_LOG("APL: Received MessagePerformanceModeChanged\n");
                NN_LOG("APL: Currently Performance Mode is %s Mode\n", nn::oe::GetPerformanceMode() == nn::oe::PerformanceMode_Normal ? "Normal" : "Boost");
                break;

        case nn::oe::MessageExitRequest:
                NN_LOG("APL: Received MessageExitRequest\n");
                break;

        default:
                NN_ABORT("APL: Received unknown message= 0x%08x\n", message);
                break;
    }
    NN_ABORT_UNLESS(message == expectMessage, "APL: Received unexpected expect=0x%08x message=0x%08x at line:%n", expectMessage, message, line);
}

//-----------------------------------------------------------------------------
//  フォーカス状態変更通知の待機
//
void WaitForFocusStateChangedImpl(nn::oe::FocusState expect, int line) NN_NOEXCEPT
{
    for (;;)
    {
        auto current = nn::oe::GetCurrentFocusState();
        switch (current)
        {
            case nn::oe::FocusState_InFocus:
            {
                NN_LOG("APL: Received FocusChanged: current= InFocus\n");
                break;
            }
            case nn::oe::FocusState_OutOfFocus:
            {
                NN_LOG("APL: Received FocusChanged: current= OutOfFocus\n");
                break;
            }
            case nn::oe::FocusState_Background:
            {
                NN_LOG("APL: Received FocusChanged: current= Background\n");
                break;
            }
            default: NN_UNEXPECTED_DEFAULT;
        }

        if (current == expect)
        {
            return;
        }

        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(100));
    }
}


namespace {

void DumpMemory(const char* buffer)
{
    NN_LOG("OE: buffer=");
    for (int i=0; i<16; ++i)
    {
        NN_LOG(" %02x", buffer[i]);
    }
    NN_LOG("\n");
}

void FillMemory(char* buffer, size_t size, char seed)
{
    char c = seed;
    for (int i=0; i<size; ++i)
    {
        buffer[i] = c;

        c = (c < 0x7e) ? c + 1 : ' ';
    }
}

void CheckMemory(char* buffer, size_t size, char seed)
{
    DumpMemory(buffer);
    NN_LOG("OE: seed= 0x%02x\n", seed);

    char c = seed;
    for (int i=0; i<size; ++i)
    {
        NN_ABORT_UNLESS(buffer[i] == c);

        c = (c < 0x7e) ? c + 1 : ' ';
    }
}

}

bool MainApplicationJumpTest() NN_NOEXCEPT
{
    static char g_Buffer[1024];
    char input;
    {
        nn::applet::StorageHandle storageHandle;
        if (!TryPopFromApplicationParameterChannel(&storageHandle, nn::applet::LaunchParameterKind_Account))
        {
            // 何もないときは、テストを抜ける
            return true;
        }
        NN_ABORT_UNLESS(GetStorageSize(storageHandle) == sizeof(g_Buffer));
        NN_ABORT_UNLESS_RESULT_SUCCESS(ReadFromStorage(storageHandle, 0, g_Buffer, sizeof(g_Buffer)));
        ReleaseStorage(storageHandle);
        input = g_Buffer[0];
        NN_LOG("APL: launched -> get %c\n", input);
        CheckMemory(g_Buffer, sizeof(g_Buffer), input);
    }
    switch (input)
    {
        case 'X':
        {
            NN_LOG("APL: launch 0\n");
            // relaunch をする
            NN_LOG("APL: RequestToRelaunchApplication()\n");
            {
                FillMemory(g_Buffer, sizeof(g_Buffer), 'x');
                nn::applet::StorageHandle storageHandle;
                NN_ABORT_UNLESS_RESULT_SUCCESS(CreateStorage(&storageHandle, sizeof(g_Buffer)));
                NN_ABORT_UNLESS_RESULT_SUCCESS(WriteToStorage(storageHandle, 0, g_Buffer, sizeof(g_Buffer)));
                nn::applet::RequestToRelaunchApplication(storageHandle);
            }
            // 無限スリープしていても殺してもらえるはず
            for(;;)
            {
                nn::os::SleepThread(nn::TimeSpan::FromDays(1));
            }
        }
        case 'Y':
        {
            NN_LOG("APL: launch 1\n");
            {
                // 'x' を受け取れることを確認
#if 1   // nn::oe::TryPopLaunchParameter() を利用する実装

                size_t realParameterSize;
                NN_ABORT_UNLESS(nn::oe::TryPopLaunchParameter(&realParameterSize, g_Buffer, sizeof(g_Buffer)));
                NN_ABORT_UNLESS(realParameterSize == sizeof(g_Buffer));

#else   // nn::applet の内部 API を利用する実装
                nn::applet::StorageHandle storageHandle;
                NN_ABORT_UNLESS(TryPopFromApplicationParameterChannel(&storageHandle, nn::applet::LaunchParameterKind_User));
                NN_ABORT_UNLESS(GetStorageSize(storageHandle) == sizeof(g_Buffer));
                NN_ABORT_UNLESS_RESULT_SUCCESS(ReadFromStorage(storageHandle, 0, g_Buffer, sizeof(g_Buffer)));
                ReleaseStorage(storageHandle);
#endif
                CheckMemory(g_Buffer, sizeof(g_Buffer), 'x');
            }
            nn::oe::EnterExitRequestHandlingSection();
            nn::oe::EnterExitRequestHandlingSection();  // ネストのチェック
            // launch をする
            NN_LOG("APL: RequestToLaunchApplication(0x%016llx)\n", appletSequenceOeApplication.value);
            {
                FillMemory(g_Buffer, sizeof(g_Buffer), 'y');
                nn::applet::StorageHandle storageHandle;
                NN_ABORT_UNLESS_RESULT_SUCCESS(CreateStorage(&storageHandle, sizeof(g_Buffer)));
                NN_ABORT_UNLESS_RESULT_SUCCESS(WriteToStorage(storageHandle, 0, g_Buffer, sizeof(g_Buffer)));
                nn::applet::RequestToLaunchApplication(appletSequenceOeApplication, storageHandle);
            }
            CheckAndProcessMessage(nn::oe::MessageExitRequest);
            NN_LOG("APL: exit\n");
            nn::oe::LeaveExitRequestHandlingSection();  // ネストのチェック
            nn::oe::LeaveExitRequestHandlingSection();
            return false;
        }
        case 'Z':
        {
            NN_LOG("APL: launch 2\n");
            {
                // 'y' を受け取れることを確認
#if 1   // nn::oe::TryPopLaunchParameter() を利用する実装

                size_t realParameterSize;
                NN_ABORT_UNLESS(nn::oe::TryPopLaunchParameter(&realParameterSize, g_Buffer, sizeof(g_Buffer)));
                NN_ABORT_UNLESS(realParameterSize == sizeof(g_Buffer));

#else   // nn::applet の内部 API を利用する実装
                nn::applet::StorageHandle storageHandle;
                NN_ABORT_UNLESS(TryPopFromApplicationParameterChannel(&storageHandle, nn::applet::LaunchParameterKind_User));
                NN_ABORT_UNLESS(GetStorageSize(storageHandle) == sizeof(g_Buffer));
                NN_ABORT_UNLESS_RESULT_SUCCESS(ReadFromStorage(storageHandle, 0, g_Buffer, sizeof(g_Buffer)));
                ReleaseStorage(storageHandle);
#endif
                CheckMemory(g_Buffer, sizeof(g_Buffer), 'y');
            }
            nn::oe::EnterExitRequestHandlingSection();
            NN_LOG("APL: RequestToLaunchApplication(0x%016llx) (dummy)\n", appletSequenceOeApplication.value);
            {
                FillMemory(g_Buffer, sizeof(g_Buffer), 'z');
                nn::applet::StorageHandle storageHandle;
                NN_ABORT_UNLESS_RESULT_SUCCESS(CreateStorage(&storageHandle, sizeof(g_Buffer)));
                NN_ABORT_UNLESS_RESULT_SUCCESS(WriteToStorage(storageHandle, 0, g_Buffer, sizeof(g_Buffer)));
                nn::applet::RequestToLaunchApplication(appletSequenceOeApplication, storageHandle);
            }
            CheckAndProcessMessage(nn::oe::MessageExitRequest);
            NN_LOG("APL: abort (expected for test)\n"); // ハンドリングを有効にした上で異常終了する
            NN_ABORT("for test");
        }
        default: NN_UNEXPECTED_DEFAULT;
    }
}

void PassSleepSequenceInFg() NN_NOEXCEPT
{
    CheckAndProcessMessage(nn::oe::MessageResume);

    // スリープ復帰によって FG を回復する
    WaitForFocusStateChanged(nn::oe::FocusState_InFocus);
    NN_LOG("APL: Received nn::oe::FocusState_InFocus\n");
}

void MainSequenceTest() NN_NOEXCEPT
{
    // アプリの起動を SA 側で検知

    // 同期のためにダミーのアプリ起動リクエストを投げる
    NN_LOG("APL: RequestToLaunchApplication(0x%016llx) (dummy2)\n", appletSequenceOeApplication.value);
    {
        char c;
        nn::applet::StorageHandle storageHandle;
        NN_ABORT_UNLESS_RESULT_SUCCESS(CreateStorage(&storageHandle, sizeof(c)));
        NN_ABORT_UNLESS_RESULT_SUCCESS(WriteToStorage(storageHandle, 0, &c, sizeof(c)));
        nn::applet::RequestToLaunchApplication(appletSequenceOeApplication, storageHandle);
    }


    // SA 側でハンドリングなしで StartSleepSequence() を行なった場合、
    // AM での SuspendWindow(), ResumeWindow() による一時動作停止もあって、
    // アプリ側での MessageInFocus と MessageOutOfFocus が相殺されるため、
    // その場合だけメッセージハンドリングを行なわないようにする。

    PassSleepSequenceInFg();
    PassSleepSequenceInFg();
    PassSleepSequenceInFg();
    PassSleepSequenceInFg();

#if 0
// NEW_AM: TODO: スリープ復帰からのメッセージに関しては要調整

    // スリープ要求を許諾する
    CheckAndProcessMessage(nn::oe::Message_ChangeIntoSleep);
    nn::oe::AllowToEnterSleep();
    NN_LOG("APL: Done nn::oe::AllowToEnterSleep()\n");

    // アプリを BG に遷移させることに成功する
    WaitForFocusStateChanged(nn::oe::FocusState_Background);
    NN_LOG("APL: Received nn::oe::FocusState_Background\n");

    // スリープから復帰するまでメッセージを待つ
    CheckAndProcessMessage(nn::oe::Message_AwakeFromSleep);

    // アプリを再度 FG に戻す
    WaitForFocusStateChanged(nn::oe::FocusState_InFocus);
    NN_LOG("APL: Received nn::oe::FocusState_InFocus\n");
#endif
}

void MainSequenceTestOfEnd() NN_NOEXCEPT
{
    // アプリを BG に遷移させることに成功する
    WaitForFocusStateChanged(nn::oe::FocusState_Background);
    NN_LOG("APL: Received nn::oe::FocusState_Background\n");

    // BG 状態のまま、アプリに終了要求
    NN_LOG("APL: Wait for MessageExitRequest.\n");
    CheckAndProcessMessage(nn::oe::MessageExitRequest);
    NN_LOG("APL: Shutdown from now.\n");
    return;
}


#if 0
//-----------------------------------------------------------------------------
//  am 内のメモリをストレージで枯渇させるテスト
//  枯渇させた後には全て一旦開放し、確保できた回数を返す。
//
const int StorageTestHandleCountMax = 500;
const size_t StorageTestSize = 2008;
nn::applet::StorageHandle g_StorageHandle[ StorageTestHandleCountMax ];
int StorageExhaustionTest() NN_NOEXCEPT
{
    // アボートせずに正しく Result をハンドリングできれば問題なし
    int i = 0;
    for (; i < StorageTestHandleCountMax; ++i)
    {
        auto result = nn::applet::CreateStorage(&g_StorageHandle[i], StorageTestSize);
        if (result <= nn::am::ResultOutOfStorageMemory() ||
            result <= nn::sf::cmif::ResultCmifProxyAllocationFailed() )
        {
            break;
        }
        NN_ABORT_UNLESS_RESULT_SUCCESS(result);

    }
    int allocatedCount = i;
    NN_LOG("APL: Allocated Storage count = %d\n", allocatedCount);

    // 全開放する
    for (i = 0; i < allocatedCount; ++i)
    {
        nn::applet::ReleaseStorage( g_StorageHandle[i] );
    }

    return allocatedCount;
}
#endif

//-----------------------------------------------------------------------------
//  メモリ関連の初期化です。
//
void InitializeAudio() NN_NOEXCEPT
{
#if 0
    const int RenderRate    = 32000;
    const int RenderCount   = (RenderRate / 200);

    // レンダラのパラメータを指定
    nn::audio::AudioRendererParameter parameter;
    parameter.sampleRate            = RenderRate;
    parameter.sampleCount           = RenderCount;
    parameter.mixBufferCount        = 6 + 2; // FinalMix(6) + SubMix(2)
    parameter.voiceCount            = 24;
    parameter.subMixCount           = 2;
    parameter.sinkCount             = 1;
    parameter.effectCount           = 3;
    parameter.performanceFrameCount = 0;

    // パラメータがシステムでサポートされているかどうかを確認します。
    NN_ABORT_UNLESS(
        nn::audio::IsValidAudioRendererParameter(parameter),
        "Invalid AudioRendererParameter specified."
    );

    // レンダラのワークバッファを準備
    size_t workBufferSize   = nn::audio::GetAudioRendererWorkBufferSize(parameter);
    void* workBuffer        = std::aligned_alloc(nn::os::MemoryPageSize, workBufferSize);
    NN_ABORT_UNLESS_NOT_NULL(workBuffer);

    // レンダラを初期化
    nn::os::SystemEvent systemEvent;
    nn::audio::AudioRendererHandle handle;
    NN_ABORT_UNLESS(
        nn::audio::OpenAudioRenderer(&handle, &systemEvent, parameter, workBuffer, workBufferSize).IsSuccess(),
        "Failed to open AudioRenderer"
    );
#endif
}

//-----------------------------------------------------------------------------
//  メモリ関連の初期化です。
//
NN_ALIGNAS(4096) char g_MallocBuffer[0x200000];
extern "C" void nninitStartup()
{
    nn::os::SetMemoryHeapSize( 0x800000 );
    nn::init::InitializeAllocator(g_MallocBuffer, sizeof(g_MallocBuffer));
}

//-----------------------------------------------------------------------------
//  デバイス種別の文字列を取得
//
const char* GetStorageTypeString(nn::oe::StorageTypeForDebug type)
{
    switch (type)
    {
    case nn::oe::StorageTypeForDebug_None:
            return "None";

    case nn::oe::StorageTypeForDebug_Host:
            return "PC-Host";

    case nn::oe::StorageTypeForDebug_GameCard:
            return "GameCard";

    case nn::oe::StorageTypeForDebug_BuiltInStorage:
            return "NAND";

    case nn::oe::StorageTypeForDebug_SdCard:
            return "SD-Card";

    default:
            return "Unknown";
    }
}

//-----------------------------------------------------------------------------
//  メイン関数
//
extern "C" void nnMain()
{
    NN_LOG("APL: Launched an application.\n");

    InitializeAudio();

    nn::oe::Initialize();
    NN_LOG("APL: Invoked nn::oe::Initialize().\n");

    auto id = nn::applet::GetAppletResourceUserId();
    NN_LOG("APL: ARUID = 0x%016llx\n", id.lower);
    NN_ABORT_UNLESS(id.lower != 0);

    // 動作モードと性能モードの変更通知を受け取る
    nn::oe::SetOperationModeChangedNotificationEnabled(true);
    nn::oe::SetPerformanceModeChangedNotificationEnabled(true);

    // 動作モードと性能モードの取得
    NN_LOG("APP: Currently Operation Mode is %s Mode\n", nn::oe::GetOperationMode() == nn::oe::OperationMode_Handheld ? "Handheld" : "Console");
    NN_LOG("APP: Currently Performance Mode is %s Mode\n", nn::oe::GetPerformanceMode() == nn::oe::PerformanceMode_Normal ? "Normal" : "Boost");

    // メッセージハンドリングモードの設定
    nn::oe::SetFocusHandlingMode(nn::oe::FocusHandlingMode_Notify);
    nn::oe::SetResumeNotificationEnabled(true);

    // 起動デバイスの取得
    nn::oe::LaunchStorageInfoForDebug storageInfo;
    nn::oe::GetLaunchStorageInfoForDebug(&storageInfo);
    NN_LOG("APP: LaunchStorage = %s\n", GetStorageTypeString(storageInfo.launchStorage));
    NN_LOG("APP: patchStorage  = %s\n", GetStorageTypeString(storageInfo.patchStorage));

    // API の呼出し確認
    NN_LOG("APP: Invoke nn::oe::SetUserInactivityDetectionTimeExtended(true)\n");
    nn::oe::SetUserInactivityDetectionTimeExtended(true);

    NN_ABORT_UNLESS( nn::oe::IsUserInactivityDetectionTimeExtended() == true);
    NN_LOG("APP: Invoked nn::oe::IsUserInactivityDetectionTimeExtended() = true\n");

    NN_LOG("APP: Invoke nn::oe::SetUserInactivityDetectionTimeExtended(false)\n");
    nn::oe::SetUserInactivityDetectionTimeExtended(false);

    NN_ABORT_UNLESS( nn::oe::IsUserInactivityDetectionTimeExtended() == false);
    NN_LOG("APP: Invoked nn::oe::IsUserInactivityDetectionTimeExtended() = false\n");

    // 0 byte のストレージを作成できるか
    {
        nn::applet::StorageHandle storage;
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::applet::CreateStorage(&storage, 0));
        nn::applet::ReleaseStorage(storage);
    }

    // SIGLO-69853: ハンドルリークしないかの確認
    for (int i=0; i<1000; ++i)
    {
        nn::os::SystemEvent event;
        nn::oe::GetDefaultDisplayResolutionChangeEvent(&event);
    }
    for (int i=0; i<1000; ++i)
    {
        nn::os::SystemEvent event;
        nn::oe::GetHdcpAuthenticationStateChangeEvent(&event);
    }

    // シーケンステスト
    if (!MainApplicationJumpTest())
    {
        return;
    }

#if 0
    // タイミング依存で nn::am::ResultOutOfStorageMemory になって、
    // 他のアプレットが落ちることがあるので、一旦無効化しておきます。
    {
        auto allocatedCount1 = StorageExhaustionTest();
        auto allocatedCount2 = StorageExhaustionTest();
        NN_ABORT_UNLESS(allocatedCount1 == allocatedCount2);
    }
#endif

    MainSequenceTest();
    nn::os::SleepThread( nn::TimeSpan::FromMilliSeconds(300) );
    MainSequenceTestOfLibraryAppletInvoking();
    MainSequenceTestOfEnd();
}

