﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nn/nn_Abort.h>
#include <nn/nn_Log.h>
#include <nn/init.h>
#include <nn/os.h>
#include <nn/ae.h>
#include <nn/applet/applet.h>

#include "../Common/applet_Common.h"

//-----------------------------------------------------------------------------

namespace {

const size_t RequestCommandSize = 128 * 1024;
const size_t ReplyMessageSize   = 128 * 1024;
NN_ALIGNAS(4096) char g_RequestCommandBuffer[ RequestCommandSize ];
NN_ALIGNAS(4096) char g_ReplayMessageBuffer[ ReplyMessageSize ];

}

//-----------------------------------------------------------------------------
//  Audio 制御
//
void ControlMainAppletAudio()
{
    // アプリのボリューム操作
    auto maExpectedVolume = nn::ae::GetMainAppletExpectedMasterVolume();
    auto laExpectedVolume = nn::ae::GetLibraryAppletExpectedMasterVolume();

    NN_LOG("LA2: Current Application's expected volume   = %1.2f\n", maExpectedVolume);
    NN_LOG("LA2: Current LibraryApplet's expected volume = %1.2f\n", laExpectedVolume);
    NN_ABORT_UNLESS(maExpectedVolume == 0.5f);
    NN_ABORT_UNLESS(laExpectedVolume == 0.2f);

    nn::ae::ChangeMainAppletMasterVolume( 0.48f, nn::TimeSpan::FromMilliSeconds(12) );
    NN_LOG("LA2: Invoked nn::ae::ChangeMainAppletMasterVolume()\n");
}


//-----------------------------------------------------------------------------
//  メモリ関連の初期化です。
//
NN_ALIGNAS(4096) char g_MallocBuffer[0x200000];
extern "C" void nninitStartup()
{
    nn::os::SetMemoryHeapSize( 0x200000 );
    nn::init::InitializeAllocator(g_MallocBuffer, sizeof(g_MallocBuffer));
}


//-----------------------------------------------------------------------------
//  メイン関数
//
void LibraryAppletCabinetMain(const nn::ae::LibraryAppletSelfInfo& info)
{
    NN_LOG("LA2: Launched an LibraryApplet2.\n");

    // 現在のシーケンスではアプリからのみ起動
    auto mainAppletIdentityInfo = nn::ae::GetMainAppletIdentityInfo();
    NN_ABORT_UNLESS_EQUAL(nn::applet::AppletId_Application, mainAppletIdentityInfo.appletId);

    auto callerAppletIdentityInfo = nn::ae::GetCallerAppletIdentityInfo();
    NN_ABORT_UNLESS_EQUAL(nn::applet::AppletId_Application, callerAppletIdentityInfo.appletId);

    // TODO: floating 起動の場合、現在 applicationId を取ることができず 0 が返る
    if (mainAppletIdentityInfo.applicationId.value != 0)
    {
        NN_ABORT_UNLESS_EQUAL(appletSequenceOeApplication.value, mainAppletIdentityInfo.applicationId.value);
    }

    //-------------------------------------------------------------------------
    // ライブラリアプレットの初期化
    nn::os::SystemEventType event;
    nn::ae::InitializeNotificationMessageEvent(&event);

    // 部分 foreground 起動なので、最初に FG 状態が通知される
    NN_LOG("LA2: Wait nn::ae::Message_ChangeIntoForeground.\n");
    CheckAndProcessMessage("LA2", &event, nn::ae::Message_ChangeIntoForeground );

    // アプリの Audio 制御
    NN_LOG("LA2: Control application's volume.\n");
    ControlMainAppletAudio();

    //-------------------------------------------------------------------------
    // 要求コマンドの待機と受信
    NN_LOG("LA2: Wait nn::ae::GetPopFromInteractiveInChannelEvent.\n");
    nn::os::WaitSystemEvent(nn::ae::GetPopFromInteractiveInChannelEvent());
    NN_LOG("LA2: Wait nn::ae::TryPopFromInteractiveInChannel.\n");
    {
        nn::applet::StorageHandle storageHandle;
        NN_ABORT_UNLESS(nn::ae::TryPopFromInteractiveInChannel(&storageHandle));
        NN_ABORT_UNLESS(GetStorageSize(storageHandle) == sizeof(g_RequestCommandBuffer));
        NN_ABORT_UNLESS_RESULT_SUCCESS(ReadFromStorage(storageHandle, 0, g_RequestCommandBuffer, sizeof(g_RequestCommandBuffer)));
        ReleaseStorage(storageHandle);
        CheckMemory(g_RequestCommandBuffer, sizeof(g_RequestCommandBuffer), 'G');
    }

    //-------------------------------------------------------------------------
    // 要求コマンドを契機に、SA に対して HOME ボタン短押しを発行
    NN_LOG("LA2: Press Home Button Short.\n");
    nn::ae::NotifyShortPressingHomeButtonForDebug();

    // 部分 foreground 起動なので、SA に FG が移り BG 状態になる
    NN_LOG("LA2: Wait nn::ae::Message_ChangeIntoBackground.\n");
    CheckAndProcessMessage("LA2", &event, nn::ae::Message_ChangeIntoBackground );

    // SA へメッセージを送る
    NN_LOG("LA2: PushToSystemGeneralChannel('N')\n");
    {
        static char buffer[2048];
        FillMemory(buffer, sizeof(buffer), 'N');
        nn::applet::StorageHandle storageHandle;
        NN_ABORT_UNLESS_RESULT_SUCCESS(CreateStorage(&storageHandle, sizeof(buffer)));
        NN_ABORT_UNLESS_RESULT_SUCCESS(WriteToStorage(storageHandle, 0, buffer, sizeof(buffer)));
        nn::applet::PushToSystemGeneralChannel(storageHandle);
    }

    // 部分 foreground 起動なので、SA から再度 FG 状態が通知される
    NN_LOG("LA2: Wait nn::ae::Message_ChangeIntoForeground.\n");
    CheckAndProcessMessage("LA2", &event, nn::ae::Message_ChangeIntoForeground );

    //-------------------------------------------------------------------------
    // 返答メッセージの送信
    NN_LOG("LA2: Invoke nn::ae::PushToInteractiveOutChannel().\n");
    {
        FillMemory(g_ReplayMessageBuffer, sizeof(g_ReplayMessageBuffer), 'H');
        nn::applet::StorageHandle storageHandle;
        NN_ABORT_UNLESS_RESULT_SUCCESS(CreateStorage(&storageHandle, sizeof(g_ReplayMessageBuffer)));
        NN_ABORT_UNLESS_RESULT_SUCCESS(WriteToStorage(storageHandle, 0, g_ReplayMessageBuffer, sizeof(g_ReplayMessageBuffer)));
        nn::ae::PushToInteractiveOutChannel(storageHandle);
    }

    //-------------------------------------------------------------------------
    // 終了要求の待機
    NN_LOG("LA2: Wait nn::ae::GetPopFromInteractiveInChannelEvent.\n");
    nn::os::WaitSystemEvent(nn::ae::GetPopFromInteractiveInChannelEvent());
    NN_LOG("LA2: Wait nn::ae::TryPopFromInteractiveInChannel.\n");
    {
        nn::applet::StorageHandle storageHandle;
        NN_ABORT_UNLESS(nn::ae::TryPopFromInteractiveInChannel(&storageHandle));
        NN_ABORT_UNLESS(GetStorageSize(storageHandle) == 0);
        ReleaseStorage(storageHandle);
    }
    NN_LOG("LA2: Wait nn::ae::TryPopFromInteractiveInChannel.\n");
}

extern "C" void nnMain()
{
    NN_LOG("LA2: Invoke nn::ae::InvokeLibraryAppletMain().\n");

    nn::ae::InvokeLibraryAppletMain(LibraryAppletCabinetMain);
}

