﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstdlib>
#include <cstring>

#include <nn/init.h>
#include <nn/os.h>
#include <nn/ae.h>
#include <nn/hid.h>
#include <nn/fs.h>
#include <nn/applet/applet.h>
#include <nn/apm/apm.h>

#include <nn/nn_Assert.h>
#include <nn/nn_Log.h>

#include "../Common/Common.h"
#include "Graphics.h"
#include "../Common/AudioRenderer.h"
#include "../Common/ConvNpad.h"


namespace {

// スレッドのスタックサイズ
const size_t ThreadStackSize = 32 * 1024;

// OE メッセージ処理スレッドのスタック
NN_OS_ALIGNAS_THREAD_STACK char g_MessageThreadStack[ThreadStackSize];

// グラフィックススレッドのスタック
NN_OS_ALIGNAS_THREAD_STACK char g_GraphicsThreadStack[ThreadStackSize];

// デバッグコントローラスレッドのスタック
NN_OS_ALIGNAS_THREAD_STACK char g_DebugPadThreadStack[ThreadStackSize];

// Audio スレッドのスタック
NN_OS_ALIGNAS_THREAD_STACK char g_AudioThreadStack[ThreadStackSize];

// ライブラリアプレット終了対応スレッドのスタック
NN_OS_ALIGNAS_THREAD_STACK char g_LibraryApplet1ThreadStack[ThreadStackSize];

// タイマースレッドのスタック
NN_OS_ALIGNAS_THREAD_STACK char g_TimerThreadStack[ThreadStackSize];

nn::os::ThreadType  g_MessageThread;
nn::os::ThreadType  g_GraphicsThread;
nn::os::ThreadType  g_DebugPadThread;
nn::os::ThreadType  g_AudioThread;
nn::os::ThreadType  g_LibraryApplet1Thread;
nn::os::ThreadType  g_TimerThread;

// ライブラリアプレットハンドル
nn::applet::LibraryAppletHandle g_LibraryAppletHandle;

// アプレットパラメータ
char g_ParamBuffer[ nn::applet::StartupParamSizeMax ];

// メッセージバッファ
char g_MessageBuffer[ nn::applet::StartupParamSizeMax ] = { 0 };

// タイマー時間
const size_t TimerTimeSeconds = 20;

// Home Button 押し待ち状態
bool g_IsWaitPressHomeButton = false;

// Audio が有効かどうか
bool g_IsAudioEnabled = false;

enum AudioState
{
    AudioState_Disabled,
    AudioState_Enabled,
    AudioState_Stress,
};

enum StarterState
{
    StarterState_ForeGround,
    StarterState_BackGround,
};

StarterState g_State = StarterState_ForeGround;
AudioState g_AudioState = AudioState_Disabled;

// スレッド制御
ThreadControl g_ThreadControl;

void* Allocate( size_t size )
{
    return malloc( size );
}

void Deallocate( void* ptr, size_t )
{
    free( ptr );
}

void Initialize()
{
    // fs 用のアロケータをセット
    nn::fs::SetAllocator(Allocate, Deallocate);

    // AE ライブラリの初期化
    nn::ae::InitializeAsSystemApplication();

    // グラフィックス関連の初期化
    Graphics::Rgba clearColor = { 0.4f, 0.1f, 0.4f, 1.0f };
    Graphics::InitializeGraphics( clearColor, "Starter" );

    // パラメータの取得 (AppletISA から起動されていれば取得できる)
    nn::applet::StorageHandle handle;
    if ( nn::applet::TryPopFromApplicationParameterChannel( &handle, nn::applet::LaunchParameterKind_User ) &&
         sizeof(g_ParamBuffer) == nn::applet::GetStorageSize(handle) )
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS( nn::applet::ReadFromStorage(handle, 0, g_ParamBuffer, sizeof(g_ParamBuffer) ) );
        nn::applet::ReleaseStorage(handle);

        std::strcpy( g_MessageBuffer, g_ParamBuffer );
    }
    else
    {
        std::strcpy( g_MessageBuffer, "" );
    }
    std::strcat( g_MessageBuffer, "STA " );
    Graphics::SetMessage( g_MessageBuffer );

    // Audio の初期化
    nnt::applet::audiorenderer::InitializeAudio();
    nnt::applet::audiorenderer::LoadBgm( nnt::applet::audiorenderer::Bgm_B );

    // DebugPad の初期化
    nnt::applet::hid::InitializeConvNpad();
}

void Finalize()
{
    // グラフィックス関連の終了処理
    Graphics::FinalizeGraphics();
}

} // namespace

//-----------------------------------------------------------------------------
// メッセージハンドリング
//
void MessageThreadFunction(void *arg) NN_NOEXCEPT
{
    nn::os::SystemEventType event;
    nn::ae::InitializeNotificationMessageEvent(&event);

    for (;;)
    {
        auto message = nn::ae::WaitForNotificationMessage(&event);
        switch (message)
        {
        case nn::ae::Message_ChangeIntoForeground:
            NN_LOG("STA: Received Message_ChangeIntoForeground\n");
            nn::ae::AcquireForegroundRights();
            g_State = StarterState_ForeGround;
            NN_LOG("STA: Done AcquireForegroundRights()\n");
            g_ThreadControl.SetInFocusState(true);
            break;

        case nn::ae::Message_ChangeIntoBackground:
            NN_LOG("STA: Received Message_ChangeIntoBackground\n");
            nn::ae::ReleaseForegroundRights();
            g_State = StarterState_BackGround;
            NN_LOG("STA: Done ReleaseForegroundRights()\n");
            g_ThreadControl.SetInFocusState(false);
            break;

        case nn::ae::Message_OperationModeChanged:
            NN_LOG("STA: Received Message_OperationModeChanged\n");
            NN_LOG("STA: Currently Operation Mode is %s Mode\n", nn::ae::GetOperationMode() == nn::ae::OperationMode_Handheld ? "Handheld" : "Console");
            break;

        case nn::ae::Message_PerformanceModeChanged:
            NN_LOG("STA: Received Message_PerformanceModeChanged\n");
            break;

        case nn::ae::Message_Exit:
            NN_LOG("STA: Received Message_Exit\n");
            g_ThreadControl.RequestExit();
            return;

        default:
            NN_LOG("STA: Received unknown message= 0x%08x\n", message);
            break;
        }
    }
}

//-----------------------------------------------------------------------------

const char* GetLibraryAppletExitReasonString(nn::applet::LibraryAppletExitReason reason) NN_NOEXCEPT
{
    switch (reason)
    {
    case nn::applet::LibraryAppletExitReason_Normal: return "LibraryAppletExitReason_Normal";
    case nn::applet::LibraryAppletExitReason_Canceled : return "LibraryAppletExitReason_Canceled";
    case nn::applet::LibraryAppletExitReason_Abnormal: return "LibraryAppletExitReason_Abnormal";
    case nn::applet::LibraryAppletExitReason_Unexpected: return "LibraryAppletExitReason_Unexpected";
    default: NN_UNEXPECTED_DEFAULT;
    }
}

void WaitLibraryAppletExited()
{
    nn::applet::JoinLibraryApplet( g_LibraryAppletHandle );
    auto exitReason = nn::applet::GetLibraryAppletExitReason(g_LibraryAppletHandle);
    NN_LOG("STA: JoinLibraryApplet: %s\n", GetLibraryAppletExitReasonString(exitReason));
    if (exitReason == nn::applet::LibraryAppletExitReason_Normal)
    {
        NN_LOG("STA: Received PopFromOutChannelEvent\n" );
        nn::applet::StorageHandle storageHandle;
        NN_ABORT_UNLESS( TryPopFromOutChannel( &storageHandle, g_LibraryAppletHandle ) );
        NN_ABORT_UNLESS( GetStorageSize(storageHandle) == sizeof(g_ParamBuffer) );
        NN_ABORT_UNLESS_RESULT_SUCCESS( ReadFromStorage( storageHandle, 0, g_ParamBuffer, sizeof(g_ParamBuffer) ) );
        ReleaseStorage( storageHandle );
        std::strcpy( g_MessageBuffer, g_ParamBuffer );
    }
    nn::applet::CloseLibraryApplet( g_LibraryAppletHandle );
    g_LibraryAppletHandle = nn::applet::InvalidLibraryAppletHandle;
}

nn::applet::LibraryAppletHandle PrepareLibraryApplet( nn::applet::AppletId id, const char* inData, size_t inDataSize, bool isAllForeground ) NN_NOEXCEPT
{
    // LA 作成
    nn::applet::LibraryAppletHandle appletHandle;
    NN_ABORT_UNLESS_RESULT_SUCCESS( nn::applet::CreateLibraryApplet( &appletHandle, id, isAllForeground ? nn::applet::LibraryAppletMode_AllForeground : nn::applet::LibraryAppletMode_PartialForeground ) );

    // 入力データの push
    {
        nn::applet::StorageHandle storageHandle;
        NN_ABORT_UNLESS_RESULT_SUCCESS( nn::applet::CreateStorage(&storageHandle, inDataSize) );
        NN_ABORT_UNLESS_RESULT_SUCCESS( nn::applet::WriteToStorage(storageHandle, 0, inData, inDataSize) );
        nn::applet::PushToInChannel( appletHandle, storageHandle );
    }
    return appletHandle;
}

bool LaunchLibraryApplet1(bool isAllForeground) NN_NOEXCEPT
{
    // LA 呼出元の画像をキャプチャしておく
    nn::applet::TakeScreenShotOfCallerApplet();

    std::strcpy( g_ParamBuffer, g_MessageBuffer );
    g_LibraryAppletHandle = PrepareLibraryApplet( nn::applet::AppletId_LibraryAppletAuth, g_ParamBuffer, sizeof(g_ParamBuffer), isAllForeground );
    if (g_LibraryAppletHandle == nn::applet::InvalidLibraryAppletHandle)
    {
        NN_LOG("STA: Request exit failed: Cannot create LA1 handle\n");
        return false;
    }
    NN_ABORT_UNLESS_RESULT_SUCCESS( nn::applet::StartLibraryApplet( g_LibraryAppletHandle, NULL ) );
    return true;
}

//-----------------------------------------------------------------------------
// LA 制御用のワーカースレッド
//
enum WorkerCommand
{
    WorkerCommand_None,
    WorkerCommand_LaunchLA,
    WorkerCommand_LaunchLAWithPartialFG,
};

const int CommandBufferCount = 8;
uintptr_t g_CommandBuffer[CommandBufferCount];
nn::os::MessageQueue g_CommandQueue(g_CommandBuffer, CommandBufferCount);

void LibraryApplet1HandleThread(void *arg)
{
    NN_UNUSED(arg);

    bool isLibraryAppletLaunched = false;

    while (g_ThreadControl.WaitForInFocusOrExitRequested())
    {
        uintptr_t cmd;
        g_CommandQueue.Receive(&cmd);
        switch (cmd)
        {
            // LA1 の通常起動
            case WorkerCommand_LaunchLA:
            {
                NN_LOG("STA: Invoke nn::applet::CreateLibraryApplet() (LA1)\n");
                isLibraryAppletLaunched = LaunchLibraryApplet1(true);
                break;
            }
            // LA1 の Partial-FG 起動
            case WorkerCommand_LaunchLAWithPartialFG:
            {
                NN_LOG("STA: Invoke nn::applet::CreateLibraryApplet() (LA1, Partial FG)\n");
                isLibraryAppletLaunched = LaunchLibraryApplet1(false);
                break;
            }
            default:
            {
                NN_LOG("STA: Unknown internal command: 0x%08x\n", cmd);
                break;
            }
        }

        // LA1 の終了を待機
        if (isLibraryAppletLaunched)
        {
            WaitLibraryAppletExited();
            isLibraryAppletLaunched = false;
        }
    }
}

//-----------------------------------------------------------------------------
//  Pad 入力ハンドリングスレッド
//
void DebugPadThreadFunction(void *arg) NN_NOEXCEPT
{
    NN_UNUSED(arg);

    nn::hid::DebugPadState debugPadState;
    nn::hid::DebugPadButtonSet pressed;
    nn::hid::DebugPadButtonSet trigger;

    while ( g_ThreadControl.WaitForInFocusOrExitRequested() )
    {
        // 最後の HOME ボタン押下待ちならボタン入力受付を終了する
        if ( g_IsWaitPressHomeButton )
        {
            return;
        }

        nnt::applet::hid::GetConvNpadState( &debugPadState );

        // ボタン情報の更新
        trigger = ~pressed & debugPadState.buttons;
        pressed = debugPadState.buttons;

        // Partial Foreground な LA が起動していたら何もしない
        if ( g_LibraryAppletHandle != nn::applet::InvalidLibraryAppletHandle )
        {
            Graphics::WaitVsync();
            continue;
        }

        // DebugPad の Y が押された
        if ( (trigger & nn::hid::DebugPadButton::Y::Mask).IsAnyOn() &&
             (g_LibraryAppletHandle == nn::applet::InvalidLibraryAppletHandle) )
        {
            // UP + Y なら Partial Foreground で LA1 起動
            if ( (pressed & nn::hid::DebugPadButton::Up::Mask).IsAnyOn() )
            {
                g_CommandQueue.Send(WorkerCommand_LaunchLAWithPartialFG);
            }
            else
            {
                g_CommandQueue.Send(WorkerCommand_LaunchLA);
            }
        }

        // DebugPad の ZR で BGM ON/OFF
        if ((trigger & nn::hid::DebugPadButton::ZR::Mask).IsAnyOn())
        {
            if( g_AudioState == AudioState_Stress )
            {
                NN_LOG( "STA: Disable Audio\n" );
                g_IsAudioEnabled = false;
                g_AudioState = AudioState_Disabled;
                Graphics::SetAudioStatus( g_IsAudioEnabled );
                nnt::applet::audiorenderer::PauseBgm();
                nnt::applet::audiorenderer::PauseSine();
            }
            else if( g_AudioState == AudioState_Disabled )
            {
                NN_LOG( "STA: Enable Audio\n" );
                g_IsAudioEnabled = true;
                g_AudioState = AudioState_Enabled;
                Graphics::SetAudioStatus( g_IsAudioEnabled );
                nnt::applet::audiorenderer::PlayBgm();
            }
            else
            {
                NN_LOG( "STA: Add Audio\n" );
                g_IsAudioEnabled = true;
                g_AudioState = AudioState_Stress;
                Graphics::SetAudioStatus( g_IsAudioEnabled );
                nnt::applet::audiorenderer::PlaySine();
            }
        }

        Graphics::WaitVsync();
    }
}

//-----------------------------------------------------------------------------
//  グラフィックスレンダリング
//
void GraphicsThreadFunction(void *arg)
{
    NN_UNUSED(arg);

    // 毎フレームのレンダリング
    while ( g_ThreadControl.WaitForInFocusOrExitRequested() )
    {
        Graphics::GraphicsRenderer();
    }
}

//-----------------------------------------------------------------------------
//  オーディオ再生スレッド
//
void AudioThreadFunction(void *arg)
{
    NN_UNUSED(arg);

    while ( g_ThreadControl.WaitForInFocusOrExitRequested() )
    {
        nnt::applet::audiorenderer::WaitAudio();
        nnt::applet::audiorenderer::UpdateAudio();
        Graphics::WaitVsync();
    }
}

void TimerThreadFunction( void *arg ) NN_NOEXCEPT
{
    nn::os::TimerEvent timer( nn::os::EventClearMode_AutoClear );

    timer.StartOneShot( nn::TimeSpan::FromSeconds(TimerTimeSeconds) );
    timer.Wait();

    g_IsWaitPressHomeButton = true;
    Graphics::SetWaitPressHomeButtonMode();
}

NN_ALIGNAS(4096) uint8_t  g_MallocBuffer[256 * 1024 * 1024];
extern "C" void nninitStartup()
{
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::ae::SetMemoryHeapSizeWithRetry(0x400000));
    nn::init::InitializeAllocator( g_MallocBuffer, sizeof(g_MallocBuffer) );
}

extern "C" void nnMain()
{
    NN_LOG( "STA: start.\n" );

    // 起動時点で InFocus 状態である
    g_ThreadControl.SetInFocusState(true);

    // 初期化処理
    Initialize();
    {
        NN_LOG("STA: My own ARUID= 0x%016llx\n", nn::applet::GetAppletResourceUserId());
        nn::applet::AppletResourceUserId aruid = {};
        auto result = nn::applet::GetAppletResourceUserIdOfCallerApplet(&aruid);
        NN_LOG("STA: Parent ARUID= 0x%016llx (result=0x%08x)\n", aruid, result.GetInnerValueForDebug());
    }

    // スレッドを生成する
    NN_ABORT_UNLESS_RESULT_SUCCESS( nn::os::CreateThread( &g_AudioThread, AudioThreadFunction, NULL, g_AudioThreadStack, ThreadStackSize, AppletPriorityForAudio ) );
    NN_ABORT_UNLESS_RESULT_SUCCESS( nn::os::CreateThread( &g_MessageThread, MessageThreadFunction, NULL, g_MessageThreadStack, ThreadStackSize, AppletPriorityForLimit1msec) );
    NN_ABORT_UNLESS_RESULT_SUCCESS( nn::os::CreateThread( &g_DebugPadThread, DebugPadThreadFunction, NULL, g_DebugPadThreadStack, ThreadStackSize, AppletPriorityForLimit1msec + 1) );
    NN_ABORT_UNLESS_RESULT_SUCCESS( nn::os::CreateThread( &g_LibraryApplet1Thread, LibraryApplet1HandleThread, NULL, g_LibraryApplet1ThreadStack, ThreadStackSize, AppletPriorityForLimit10msec) );
    NN_ABORT_UNLESS_RESULT_SUCCESS( nn::os::CreateThread( &g_GraphicsThread, GraphicsThreadFunction, NULL, g_GraphicsThreadStack, ThreadStackSize, AppletPriorityForOver10msec) );
    NN_ABORT_UNLESS_RESULT_SUCCESS( nn::os::CreateThread( &g_TimerThread, TimerThreadFunction, NULL, g_TimerThreadStack, ThreadStackSize, AppletPriorityForOver10msec + 1) );

    // スレッドの実行を開始する
    nn::os::StartThread( &g_MessageThread );
    nn::os::StartThread( &g_GraphicsThread );
    nn::os::StartThread( &g_DebugPadThread );
    nn::os::StartThread( &g_AudioThread );
    nn::os::StartThread( &g_LibraryApplet1Thread );
    nn::os::StartThread( &g_TimerThread );

    // API が呼べることだけ確認しておく
    nn::ae::SetScreenShotPermission(nn::ae::ScreenShotPermission_Forbid);
    nn::ae::SetScreenShotPermission(nn::ae::ScreenShotPermission_Permit);
    nn::ae::SetScreenShotPermission(nn::ae::ScreenShotPermission_Inherit);

    // スレッドの終了を待機
    nn::os::WaitThread( &g_MessageThread );
    nn::os::WaitThread( &g_GraphicsThread );
    nn::os::WaitThread( &g_DebugPadThread );
    nn::os::WaitThread( &g_AudioThread );
    nn::os::WaitThread( &g_LibraryApplet1Thread );
    nn::os::WaitThread( &g_TimerThread );

    // スレッドを破棄する
    nn::os::DestroyThread( &g_MessageThread );
    nn::os::DestroyThread( &g_GraphicsThread );
    nn::os::DestroyThread( &g_DebugPadThread );
    nn::os::DestroyThread( &g_AudioThread );
    nn::os::DestroyThread( &g_LibraryApplet1Thread );
    nn::os::DestroyThread( &g_TimerThread );

    // 終了処理
    Finalize();

    NN_LOG( "STA: end.\n" );

    return;
}
