﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstdlib>
#include <cstring>
#include <mutex>

#include <nn/init.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/os.h>
#include <nn/ns/ns_InitializationApi.h>
#include <nn/ns/ns_ApplicationControlDataApi.h>
#include <nn/ae.h>
#include <nn/ae/ae_ApplicationLaunchPropertyApi.h>
#include <nn/hid.h>
#include <nn/fs.h>
#include <nn/applet/applet.h>
#include <nn/account/account_Api.h>
#include <nn/account/account_ApiForAdministrators.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/vi/fbshare/vi_SharedLayerWindowGfx.h>

#include <nn/nn_Assert.h>
#include <nn/nn_Log.h>

#include "../Common/applet_Configs.h"
#include "../Common/Common.h"
#include "../Common/Graphics.h"
#include "../Common/AudioOut.h"
#include "../Common/ConvNpad.h"


namespace {

// スレッドのスタックサイズ
const size_t ThreadStackSize = 32 * 1024;

// AE メッセージ処理スレッドのスタック
NN_OS_ALIGNAS_THREAD_STACK char g_MessageThreadStack[ThreadStackSize];

// アプレットイベント処理スレッドのスタック
NN_OS_ALIGNAS_THREAD_STACK char g_LibraryApplet1HandleThreadStack[ThreadStackSize];

// グラフィックススレッドのスタック
NN_OS_ALIGNAS_THREAD_STACK char g_GraphicsThreadStack[ThreadStackSize];

// デバッグコントローラスレッドのスタック
NN_OS_ALIGNAS_THREAD_STACK char g_DebugPadThreadStack[ThreadStackSize];

// Foreground キャプチャスレッド
NN_OS_ALIGNAS_THREAD_STACK char g_CaptureThreadStack[ThreadStackSize];

// Audio スレッドのスタック
NN_OS_ALIGNAS_THREAD_STACK char g_AudioThreadStack[ThreadStackSize];

nn::os::ThreadType  g_MessageThread;
nn::os::ThreadType  g_LibraryApplet1HandleThread;
nn::os::ThreadType  g_GraphicsThread;
nn::os::ThreadType  g_DebugPadThread;
nn::os::ThreadType  g_CaptureThread;
nn::os::ThreadType  g_AudioThread;

nn::os::SystemEventType g_MessageSystemEvent;
nn::ae::ApplicationHandle g_ApplicationHandle;

enum SystemAppletState
{
    SystemAppletState_Init,
    SystemAppletState_ForeGround,
    SystemAppletState_BackGround
};

enum AudioState
{
    AudioState_Disabled,
    AudioState_Enabled,
    AudioState_Stress,
};

struct SystemAppletInfo
{
    SystemAppletState state;
    bool isWaitForLaunch;
    bool isAppLaunched;
    bool isAppStarter;

    bool isAudioEnabled;
    bool isInMediaPlayback;
};

SystemAppletInfo g_Info = { SystemAppletState_Init, false, false, false, false, false };

// ライブラリアプレットハンドル
nn::applet::LibraryAppletHandle g_LibraryAppletHandle;

// コールバックによって通知されたハンドル
nn::applet::LibraryAppletHandle g_AddedLibraryAppletHandle;

// アプレットパラメータ
char g_ParamBuffer[ nn::applet::StartupParamSizeMax ] = { 0 };

// メッセージバッファ
char g_MessageBuffer[ nn::applet::StartupParamSizeMax ] = { 0 };

// 起動するアプリケーション ID
const nn::ncm::ApplicationId ApplicationId = appletIntegrationApplicationId;

// 起動する Starter ID
const nn::ncm::SystemApplicationId StarterId = { 0x0100000000001012 };

int g_AutoSleepForbiddenMode = 0;
bool g_InSleepSequence = false;
AudioState audioState = AudioState_Disabled;

// スレッド制御
ThreadControl g_ThreadControl;

// キャプチャバッファの Map 用 Mutex
nn::os::Mutex g_MutexForMapLastApplication{false};
nn::os::Mutex g_MutexForMapLastForeground{false};
nn::os::Mutex g_MutexForMapCallerApplet{false};

// アプリ終了時などにキャプチャバッファをクリアする
void ClearLastApplicationCaptureBuffer()
{
    auto pBuffer = Graphics::GetLinearTextureBuffer( 0 );
    auto dataSize = Graphics::GetLinearTextureSize( 0 );
    std::memset( pBuffer, 0, dataSize );
}

NN_ALIGNAS(4096) char g_CaptureImage[nn::ae::CaptureBufferSize];
void MapAllCaptureBufferForDebug()
{
    // キャプチャバッファ３つを同時に Map する
    std::lock_guard<nn::os::Mutex> lock1(g_MutexForMapLastApplication);
    bool isMapped1 = false;
    for (;;)
    {
        bool isScreenShotEnabled;
        void* mappedAddress;
        auto result = nn::ae::MapLastApplicationCaptureBuffer(&isScreenShotEnabled, &mappedAddress);
        if (result.IsSuccess())
        {
            if (!isScreenShotEnabled)
            {
                NN_LOG("SA: MapLastApplicationCaptureBuffer() => ScreenShot is Disabled\n");
            }
            isMapped1 = true;
            break;
        }
        else if (result <= nn::ae::ResultCaptureBufferBusy())
        {
            nn::os::SleepThread( nn::TimeSpan::FromMilliSeconds(10) );
            continue;
        }
        else
        {
            NN_LOG("SA: MapLastApplicationCaptureBuffer() failed %X(%d-%d)\n", result, result.GetModule(), result.GetDescription());
            break;
        }
    }
    NN_UTIL_SCOPE_EXIT
    {
        if(isMapped1)
        {
            nn::ae::UnmapLastApplicationCaptureBuffer();
            isMapped1 = false;
        }
    };

    std::lock_guard<nn::os::Mutex> lock2(g_MutexForMapLastForeground);
    bool isMapped2 = false;
    for (;;)
    {
        bool isScreenShotEnabled;
        void* mappedAddress;
        auto result = nn::ae::MapLastForegroundCaptureBuffer(&isScreenShotEnabled, &mappedAddress);
        if (result.IsSuccess())
        {
            if (!isScreenShotEnabled)
            {
                NN_LOG("SA: MapLastForegroundCaptureBuffer() => ScreenShot is Disabled\n");
            }
            isMapped2 = true;
            break;
        }
        else if (result <= nn::ae::ResultCaptureBufferBusy())
        {
            nn::os::SleepThread( nn::TimeSpan::FromMilliSeconds(10) );
            continue;
        }
        else
        {
            NN_LOG("SA: MapLastForegroundCaptureBuffer() failed %X(%d-%d)\n", result, result.GetModule(), result.GetDescription());
            break;
        }
    }
    NN_UTIL_SCOPE_EXIT
    {
        if(isMapped2)
        {
            nn::ae::UnmapLastForegroundCaptureBuffer();
            isMapped2 = false;
        }
    };

    std::lock_guard<nn::os::Mutex> lock3(g_MutexForMapCallerApplet);
    bool isMapped3 = false;
    for (;;)
    {
        bool isScreenShotEnabled;
        void* mappedAddress;
        auto result = nn::ae::MapCallerAppletCaptureBuffer(&isScreenShotEnabled, &mappedAddress);
        if (result.IsSuccess())
        {
            if (!isScreenShotEnabled)
            {
                NN_LOG("SA: MapCallerAppletCaptureBuffer() => ScreenShot is Disabled\n");
            }
            isMapped3 = true;
            break;
        }
        else if (result <= nn::ae::ResultCaptureBufferBusy())
        {
            nn::os::SleepThread( nn::TimeSpan::FromMilliSeconds(10) );
            continue;
        }
        else
        {
            NN_LOG("SA: MapCallerAppletCaptureBuffer() failed %X(%d-%d)\n", result, result.GetModule(), result.GetDescription());
            break;
        }
    }
    NN_UTIL_SCOPE_EXIT
    {
        if(isMapped3)
        {
            nn::ae::UnmapCallerAppletCaptureBuffer();
            isMapped3 = false;
        }
    };

    {
        bool isScreenShotEnabled;
        NN_LOG("nn::ae::GetLastApplicationCaptureBuffer() result=0x%08x capture=%s\n", nn::ae::GetLastApplicationCaptureBuffer(&isScreenShotEnabled, g_CaptureImage, sizeof(g_CaptureImage)).GetInnerValueForDebug(), isScreenShotEnabled ? "true" : "false");
        NN_LOG("nn::ae::GetLastForegroundCaptureBuffer()  result=0x%08x capture=%s\n", nn::ae::GetLastForegroundCaptureBuffer(&isScreenShotEnabled, g_CaptureImage, sizeof(g_CaptureImage)).GetInnerValueForDebug(), isScreenShotEnabled ? "true" : "false");
        NN_LOG("nn::ae::GetCallerAppletCaptureBuffer()    result=0x%08x capture=%s\n", nn::ae::GetCallerAppletCaptureBuffer(&isScreenShotEnabled, g_CaptureImage, sizeof(g_CaptureImage)).GetInnerValueForDebug(), isScreenShotEnabled ? "true" : "false");
    }

}// NOLINT(impl/function_size)

void UpdateLastApplicationCaptureBuffer()
{
    std::lock_guard<nn::os::Mutex> lock(g_MutexForMapLastApplication);

    void* mappedAddress = nullptr;
    for (;;)
    {
        bool isScreenShotEnabled;
        auto result = nn::ae::MapLastApplicationCaptureBuffer(&isScreenShotEnabled, &mappedAddress);
        if (result.IsSuccess())
        {
            if (!isScreenShotEnabled)
            {
                NN_LOG("SA: MapLastApplicationCaptureBuffer() => ScreenShot is Disabled\n");
            }
            break;
        }
        else if (result <= nn::ae::ResultCaptureBufferBusy())
        {
            nn::os::SleepThread( nn::TimeSpan::FromMilliSeconds(10) );
            continue;
        }
        else
        {
            NN_LOG("SA: MapLastApplicationCaptureBuffer() failed %X(%d-%d)\n", result, result.GetModule(), result.GetDescription());
            break;
        }
    }

    if(mappedAddress)
    {
        auto dstAddr = Graphics::GetLinearTextureBuffer( 0 );
        auto dstSize = Graphics::GetLinearTextureSize( 0 );
        std::memcpy( dstAddr, mappedAddress, dstSize );
        nn::ae::UnmapLastApplicationCaptureBuffer();
    }
}

void UpdateLastForegroundCaptureBuffer()
{
    std::lock_guard<nn::os::Mutex> lock(g_MutexForMapLastForeground);

    void* mappedAddress = nullptr;
    for (;;)
    {
        bool isScreenShotEnabled;
        auto result = nn::ae::MapLastForegroundCaptureBuffer(&isScreenShotEnabled, &mappedAddress);
        if (result.IsSuccess())
        {
            if (!isScreenShotEnabled)
            {
                NN_LOG("SA: MapLastForegroundCaptureBuffer() => ScreenShot is Disabled\n");
            }
            break;
        }
        else if (result <= nn::ae::ResultCaptureBufferBusy())
        {
            nn::os::SleepThread( nn::TimeSpan::FromMilliSeconds(10) );
            continue;
        }
        else
        {
            NN_LOG("SA: MapLastForegroundCaptureBuffer() failed %X(%d-%d)\n", result, result.GetModule(), result.GetDescription());
            break;
        }
    }

    if(mappedAddress)
    {
        auto dstAddr = Graphics::GetLinearTextureBuffer( 1 );
        auto dstSize = Graphics::GetLinearTextureSize( 1 );
        std::memcpy( dstAddr, mappedAddress, dstSize );
        nn::ae::UnmapLastForegroundCaptureBuffer();
    }
}

void StartSleepSequenceImpl()
{
    if (g_InSleepSequence)
    {
        NN_LOG("SA: sleep sequence is already running\n");
        return;
    }
    g_InSleepSequence = true;
    // スリープ要求
    NN_LOG("SA: Invoke nn::ae::StartSleepSequence()\n");
    nn::ae::StartSleepSequence();
}

void PrintCradleFirmwareVersion()
{
    nn::ae::CradleFwVersion cradleFwVersion;
    if (nn::ae::GetCradleFwVersion(&cradleFwVersion))
    {
        NN_LOG("SA: Cradle Fw Version (%u, %u, %u, %u)\n", cradleFwVersion.pdcH, cradleFwVersion.pdcA, cradleFwVersion.mcu, cradleFwVersion.dp2hdmi);
    }
    else
    {
        NN_LOG("SA: Cradle Fw Version is not available.\n");
    }
}

void PrintCaptureBufferInfo(bool isForeground)
{
    auto pSharedWindow = Graphics::GetSharedLayerWindow();
    if(pSharedWindow == nullptr)
    {
        return;
    }

    nn::Result result;
    bool isScreenshotPermitted;
    nn::gfx::Texture* pTexture;

    const char* header = isForeground ? "SA" : "SA(BG)";

    result = pSharedWindow->AcquireLastForegroundTexture(&isScreenshotPermitted, &pTexture);
    if(result.IsSuccess())
    {
        int index;
        NN_ABORT_UNLESS_RESULT_SUCCESS(pSharedWindow->GetInternalTextureIndex(&index, pTexture));
        NN_LOG("%s: LastForegroundTexture (index=%d, screenshot=%s)\n", header, index, isScreenshotPermitted ? "y" : "n");
        pSharedWindow->ReleaseLastForegroundTexture();
    }
    else
    {
        NN_LOG("%s: LastForegroundTexture -> failed %08X(%d-%d)\n", header, result, result.GetModule(), result.GetDescription());
    }

    result = pSharedWindow->AcquireLastApplicationTexture(&isScreenshotPermitted, &pTexture);
    if(result.IsSuccess())
    {
        int index;
        NN_ABORT_UNLESS_RESULT_SUCCESS(pSharedWindow->GetInternalTextureIndex(&index, pTexture));
        NN_LOG("%s: LastApplicationTexture (index=%d, screenshot=%s)\n", header, index, isScreenshotPermitted ? "y" : "n");
        pSharedWindow->ReleaseLastApplicationTexture();
    }
    else
    {
        NN_LOG("%s: LastApplicationTexture -> failed %08X(%d-%d)\n", header, result, result.GetModule(), result.GetDescription());
    }

    result = pSharedWindow->AcquireCallerAppletTexture(&isScreenshotPermitted, &pTexture);
    if(result.IsSuccess())
    {
        int index;
        NN_ABORT_UNLESS_RESULT_SUCCESS(pSharedWindow->GetInternalTextureIndex(&index, pTexture));
        NN_LOG("%s: CallerAppletTexture (index=%d, screenshot=%s)\n", header, index, isScreenshotPermitted ? "y" : "n");
        pSharedWindow->ReleaseCallerAppletTexture();
    }
    else
    {
        NN_LOG("%s: CallerAppletTexture -> failed %08X(%d-%d)\n", header, result, result.GetModule(), result.GetDescription());
    }
}

//-----------------------------------------------------------------------------
//  アプリの ControlProperty と LaunchProperty のチェック
//  Integration/App/App.*.nmeta の内容と同期させておくこと。
//
nn::ns::ApplicationControlProperty g_ControlProperty;
nn::arp::ApplicationLaunchProperty g_LaunchProperty;

const char* g_StorageName[] =
{
    "None",
    "PC-Host",
    "GameCard",
    "Built-In-System",
    "Built-In-User",
    "SD-Card",
    "Any",
};

const char* GetStorageName(nn::ncm::StorageId id)
{
    if (nn::ncm::StorageId::None <= id && id <= nn::ncm::StorageId::Any)
    {
        return g_StorageName[static_cast<int>(id)];
    }
    return "Unknown";
}

void CheckApplicationProperties(nn::ae::ApplicationHandle handle)
{
    auto isControlPropertyValid = nn::ae::GetApplicationControlProperty(&g_ControlProperty, handle).IsSuccess();
    auto isLaunchPropertyValid = nn::ae::GetApplicationLaunchProperty(&g_LaunchProperty, handle).IsSuccess();

    NN_LOG("SA: GetApplicationControlProperty() = %s\n", isControlPropertyValid ? "true" : "false");
    if (isControlPropertyValid)
    {
        NN_LOG("SA:   LogoHandling = %s\n", g_ControlProperty.logoHandling == nn::ns::LogoHandling::Auto ? "Auto" : "Manual");
        NN_LOG("SA:   Screenshot   = %s\n", g_ControlProperty.screenShot == nn::ns::Screenshot::Allow ? "Allow" : "Deny");
        NN_LOG("SA:   VideoCapture = %s\n", g_ControlProperty.videoCapture == nn::ns::VideoCapture::Disable ? "Disable" : g_ControlProperty.videoCapture == nn::ns::VideoCapture::Enable ? "Enable" : "Manual");
        NN_LOG("\n");

        NN_ABORT_UNLESS(g_ControlProperty.logoHandling == nn::ns::LogoHandling::Manual);
        NN_ABORT_UNLESS(g_ControlProperty.screenShot   == nn::ns::Screenshot::Allow);
        NN_ABORT_UNLESS(g_ControlProperty.videoCapture == nn::ns::VideoCapture::Enable);
    }

    NN_LOG("SA: GetApplicationLaunchProperty() = %s\n", isLaunchPropertyValid ? "true" : "false");
    if (isLaunchPropertyValid)
    {
        NN_LOG("SA:   ApplicationId   = 0x%016llx\n", g_LaunchProperty.id);
        NN_LOG("SA:   Version         = 0x%08x\n", g_LaunchProperty.version);
        NN_LOG("SA:   LaunchStorageId = %d [%s]\n", g_LaunchProperty.storageId, GetStorageName(g_LaunchProperty.storageId));
        NN_LOG("SA:   PatchStorageId  = %d [%s]\n", g_LaunchProperty.patchStorageId, GetStorageName(g_LaunchProperty.patchStorageId));
        NN_LOG("\n");

        NN_ABORT_UNLESS(g_LaunchProperty.id == ApplicationId);
    }
}

//-----------------------------------------------------------------------------

char g_ControlDataBuffer[144 * 1024];

nn::Result GetApplicationControlPropertyImpl(const nn::ns::ApplicationControlProperty** pOutPointer, const nn::ncm::ApplicationId& applicationId) NN_NOEXCEPT
{
    // アプリ起動前の ApplicationControlData の取得
    size_t controlDataSize = 0;
    auto result = nn::ns::GetApplicationControlData(
                            &controlDataSize,
                            g_ControlDataBuffer,
                            sizeof(g_ControlDataBuffer),
                            nn::ns::ApplicationControlSource::StorageOnly,
                            applicationId);
    if (!result.IsSuccess())
    {
        NN_LOG("SA: Failed to ns::GetApplicationControlData() (result: 0x%08x)\n", result.GetInnerValueForDebug() );
        return result;
    }

    // アプリ起動前の ApplicationControlProperty の取得
    nn::ns::ApplicationControlDataAccessor accessor( g_ControlDataBuffer, sizeof(g_ControlDataBuffer) );
    *pOutPointer = &accessor.GetProperty();
    NN_RESULT_SUCCESS;
}

// 選択したユーザアカウント情報をアプリ起動引数としてプッシュ
nn::Result PushUserAccountInfoOnLaunchApplication(const nn::account::Uid& uid, const nn::ae::ApplicationHandle& applicationHandle) NN_NOEXCEPT
{
    auto info = nn::account::MakePreselectionInfo(uid);

    nn::applet::StorageHandle storageHandle;
    NN_RESULT_DO(nn::applet::CreateStorage(&storageHandle, sizeof(info)));

    bool isSuccess = false;
    NN_UTIL_SCOPE_EXIT
    {
        if (!isSuccess)
        {
            nn::applet::ReleaseStorage(storageHandle);
        }
    };

    NN_RESULT_DO(nn::applet::WriteToStorage(storageHandle, 0, &info, sizeof(info)));

    nn::ae::PushApplicationLaunchParameter(applicationHandle, nn::applet::LaunchParameterKind_Account, storageHandle);
    isSuccess = true;
    NN_RESULT_SUCCESS;
}

// アプリ起動前のユーザ選択が必要なら自動的にユーザを１つ選択する
bool SelectUserAccountAutoIfRequired(const nn::ae::ApplicationHandle applicationHandle) NN_NOEXCEPT
{
    const auto applicationId = nn::ae::GetApplicationId(applicationHandle);
    NN_ABORT_UNLESS(applicationId == appletIntegrationApplicationId);

    const nn::ns::ApplicationControlProperty* p;
    if (!GetApplicationControlPropertyImpl(&p, applicationId).IsSuccess())
    {
        return false;
    }

    NN_LOG("SA: SelectUserAccountAutoIfRequired() startupUserAccount=%d\n", p->startupUserAccount);
    if (nn::ns::StartupUserAccount::None == p->startupUserAccount)
    {
        // ユーザアカウント選択不要
        return true;
    }

    // ユーザアカウントを選択
    nn::account::Uid uid = {};
    {
        // 全ユーザアカウントのうち適当に１つだけリストアップ
        int length = 0;
        auto result = nn::account::ListAllUsers(&length, &uid, 1);
        if (!(result.IsSuccess() && length >= 1 && uid))
        {
            NN_LOG("SA: Failed to pre-select user account (result=0x%08x)\n", result.GetInnerValueForDebug());
            return false;
        }
    }
    NN_LOG("SA: Selected user account: uid= { 0x%016llx, 0x%016llx }\n", uid._data[0], uid._data[1]);

    // 選択したユーザアカウント情報をアプリ起動引数としてプッシュ
    {
        auto result = PushUserAccountInfoOnLaunchApplication(uid, applicationHandle);
        if (!result.IsSuccess())
        {
            NN_LOG("SA: Failed to push account info on launch application. result=0x%08x\n", result.GetInnerValueForDebug());
            return false;
        }
    }

    // 全て成功
    return true;
}

void PrintApplicationLaunchRequestInfo(nn::ae::ApplicationHandle handle)
{
    auto info = nn::ae::GetApplicationLaunchRequestInfo(handle);

    NN_LOG("SA: ae::GetApplicationLaunchRequestInfo():\n");
    NN_LOG("SA:     IsApplicationChain()=%s  IsFloatingApplication()=%s\n", info.IsApplicationChain() ? "true" : "false", info.IsFloatingApplication() ? "true" : "false");
    NN_LOG("SA:     requestingAppletId    : %d\n", info.requestingAppletId);
    NN_LOG("SA:     requestingMainAppletId: %d\n", info.requestingMainAppletId);
    NN_LOG("SA:     applicationId         : 0x%016llx\n", info.applicationId);
}

//-----------------------------------------------------------------------------

bool RequestExitLibraryApplet1() NN_NOEXCEPT;

//-----------------------------------------------------------------------------
// メッセージハンドリング
//
void HandleAeMessage(void* arg)
{
    NN_UNUSED(arg);

    for (;;)
    {
        auto message = nn::ae::WaitForNotificationMessage( &g_MessageSystemEvent );
        switch (message)
        {
        case nn::ae::Message_ChangeIntoForeground:
            NN_LOG("SA: Received Message_ChangeIntoForeground\n");
            nn::ae::AcquireForegroundRights();

            // キャプチャバッファの更新
            if ( g_Info.state == SystemAppletState_BackGround )
            {
                MapAllCaptureBufferForDebug();
                UpdateLastApplicationCaptureBuffer();
                UpdateLastForegroundCaptureBuffer();
            }
            g_Info.state = SystemAppletState_ForeGround;
            g_ThreadControl.SetInFocusState(true);
            break;

        case nn::ae::Message_ChangeIntoBackground:
            NN_LOG("SA: Received Message_ChangeIntoBackground\n");
            nn::ae::ReleaseForegroundRights();
            g_ThreadControl.SetInFocusState(false);
            g_Info.state = SystemAppletState_BackGround;
            break;

        case nn::ae::Message_RequestToPrepareSleep:
            NN_LOG("SA: Received Message_RequestToPrepareSleep\n");
            nn::ae::DisableHandlingForRequestToDisplay();
            nn::ae::EnterEntranceProcessSection();
            nn::ae::LockForeground();
            nn::ae::AllowToEnterSleepAndWait();
            break;

        case nn::ae::Message_FinishedSleepSequence:
            NN_LOG("SA: Received Message_FinishedSleepSequence\n");
            nn::ae::UnlockForeground();
            nn::ae::LeaveEntranceProcessSection();
            nn::ae::EnableHandlingForRequestToDisplay();
            g_InSleepSequence = false;
            break;

        case nn::ae::Message_Exit:
            NN_LOG("SA: Received Message_Exit\n");
            g_ThreadControl.RequestExit();
            return;

        case nn::ae::Message_RequestToGoBackQuestMenu:
            NN_LOG("SA: Received Message_RequestToGoBackQuestMenu\n");
            if ( g_Info.isAppLaunched == true )
            {
                NN_LOG("SA: Invoke nn::ae::RequestApplicationExit()\n");
                nn::ae::RequestApplicationExit( g_ApplicationHandle );
            }
            break;

        case nn::ae::Message_FloatingApplicationDetected:
            NN_LOG("SA: Received Message_FloatingApplicationDetected\n");
            {
                auto applicationHandle = nn::ae::PopFloatingApplicationHandle();
                PrintApplicationLaunchRequestInfo(applicationHandle);

                // ユーザアカウントを選択
                if (!SelectUserAccountAutoIfRequired(applicationHandle))
                {
                    NN_LOG("======================================================================\n");
                    NN_LOG("SA: Gave up to start application (ApplicationId=0x%016llx)\n", nn::ae::GetApplicationId(applicationHandle));
                    NN_LOG("======================================================================\n");
                    nn::ae::RequestApplicationExit(applicationHandle);
                    nn::ae::CloseApplicationHandle(applicationHandle);
                    break;
                }

                CheckApplicationProperties( applicationHandle );
                g_Info.isWaitForLaunch = false;
                g_Info.isAppLaunched = true;
                g_ApplicationHandle = applicationHandle;
                nn::ae::RequestApplicationGetForeground(applicationHandle);
            }
            break;

        case nn::ae::Message_ApplicationExited:
        {
            nn::ae::EnterInterruptSceneProcessSection();
            NN_LOG("SA: Received Message_ApplicationExited\n");
            g_Info.isAppLaunched = false;
            if (g_ApplicationHandle.IsValid())
            {
                // アプリ動作中に SA を強制再起動したときの対策
                auto applicationResult = nn::ae::GetApplicationResult(g_ApplicationHandle);
                NN_UNUSED(applicationResult);
                NN_LOG("SA: ApplicationResult: %d\n", static_cast<int>(applicationResult));
                NN_LOG("SA: ApplicationResultForDebug: 0x%08x\n", GetApplicationResultRawForDebug(g_ApplicationHandle).GetInnerValueForDebug());
                // アプリケーションハンドルを閉じる
                nn::ae::CloseApplicationHandle( g_ApplicationHandle );
                g_ApplicationHandle = nn::ae::ApplicationHandle::GetInvalidHandle();
            }
            ClearLastApplicationCaptureBuffer();
            nn::ae::LeaveInterruptSceneProcessSection();
            break;
        }

        case nn::ae::Message_LaunchApplicationRequested:
        {
            NN_LOG("SA: Received Message_LaunchApplicationRequested\n");

            // SA チェインの LA を終了させる
            RequestExitLibraryApplet1();

            // 現在のアプリを終了させる
            if (g_Info.isAppLaunched)
            {
                NN_LOG("SA: Invoke nn::ae::RequestApplicationExit()\n");
                nn::ae::RequestApplicationExit( g_ApplicationHandle );

                // アプリの終了を待つ
                for (;;)
                {
                    if (nn::ae::Message_ApplicationExited == nn::ae::WaitForNotificationMessage( &g_MessageSystemEvent ))
                    {
                        break;
                    }
                }
                nn::ae::CloseApplicationHandle( g_ApplicationHandle );
                g_Info.isWaitForLaunch = false;
                g_Info.isAppLaunched = false;
                g_ApplicationHandle = nn::ae::ApplicationHandle::GetInvalidHandle();
            }

            // 起動要求のアプリハンドルを取得
            nn::ae::ApplicationHandle handle;
            NN_ABORT_UNLESS(nn::ae::TryPopLaunchRequestedApplication(&handle));
            NN_ABORT_UNLESS(nn::ae::GetApplicationId(handle) == appletIntegrationApplicationId);
            PrintApplicationLaunchRequestInfo(handle);

            // ユーザアカウントを選択
            if (!SelectUserAccountAutoIfRequired(handle))
            {
                NN_LOG("======================================================================\n");
                NN_LOG("SA: Gave up to start application (ApplicationId=0x%016llx)\n", nn::ae::GetApplicationId(handle));
                NN_LOG("======================================================================\n");
                nn::ae::CloseApplicationHandle(handle);
                break;
            }

            // アプリの起動
            g_Info.isWaitForLaunch = false;
            g_Info.isAppLaunched = true;
            g_ApplicationHandle = handle;
            NN_LOG("SA: Invoke nn::ae::StartApplication()\n");
            nn::ae::StartApplication( g_ApplicationHandle );
            nn::ae::RequestApplicationGetForeground( g_ApplicationHandle );
            break;
        }

        case nn::ae::Message_DetectShortPressingHomeButton:
        {
            NN_LOG("SA: Received Message_DetectShortPressingHomeButton\n");
            if (!nn::ae::TryEnterHomeButtonProcessSection())
            {
                NN_LOG("SA: HomeButton Prohibited.\n");
                break;
            }
            NN_UTIL_SCOPE_EXIT
            {
                nn::ae::LeaveHomeButtonProcessSection();
            };
            if (g_LibraryAppletHandle != nn::applet::InvalidLibraryAppletHandle)
            {
                // BG 状態でない場合は全 LA の終了要求を諦める
                if (!g_ThreadControl.TryWaitForNotInFocus())
                {
                    break;
                }
                nn::applet::RequestExitLibraryAppletAndWait(g_LibraryAppletHandle);
            }
            if ( g_Info.isAppLaunched == true )
            {
                // アプリが起動している
                if ( g_Info.isAppLaunched )
                {
                    if ( g_Info.state == SystemAppletState_BackGround )
                    {
                        if ( g_Info.isAppStarter )
                        {
                            // Starter に終了要求
                            NN_LOG("SA: Invoke nn::ae::RequestApplicationExit()\n");
                            nn::ae::RequestApplicationExit( g_ApplicationHandle );
                            g_Info.isAppStarter = false;
                        }
                        else
                        {
                            // アプリに BG 遷移要求
                            NN_LOG("SA: Invoke nn::ae::RequestToGetForeground()\n");
                            nn::ae::RequestToGetForeground();
                        }
                    }
                    // アプリに FG 遷移要求
                    else if ( g_Info.state == SystemAppletState_ForeGround )
                    {
                        NN_LOG("SA: Invoke nn::ae::RequestApplicationGetForeground()\n");
                        nn::ae::RequestApplicationGetForeground( g_ApplicationHandle );
                    }
                }
            }
            else
            {
                NN_LOG("SA: Application doesn't launched. Do nothing.\n");
            }
            break;
        }

        case nn::ae::Message_RequestToDisplay:
            NN_LOG("SA: Received Message_RequestToDisplay\n");
            NN_LOG("SA: Invoke nn::ae::ApproveToDisplay()\n");
            // TORIAEZU: すぐに表示許可
            nn::ae::ApproveToDisplay();
            break;

        case nn::ae::Message_DetectLongPressingHomeButton:
            NN_LOG("SA: Received Message_DetectLongPressingHomeButton\n");
            break;

        case nn::ae::Message_DetectShortPressingPowerButton:
            NN_LOG("SA: Received Message_DetectShortPressingPowerButton\n");
            StartSleepSequenceImpl();
            break;

        case nn::ae::Message_DetectLongPressingPowerButton:
            NN_LOG("SA: Received Message_DetectLongPressingPowerButton\n");
            NN_LOG("SA: Invoke nn::ae::StartShutdownSequence()\n");
            nn::ae::StartShutdownSequence();
            nn::os::SleepThread( nn::TimeSpan::FromDays(1) );
            break;

        case nn::ae::Message_OperationModeChanged:
            NN_LOG("SA: Received Message_OperationModeChanged\n");
            NN_LOG("SA: Currently Operation Mode is %s Mode\n", nn::ae::GetOperationMode() == nn::ae::OperationMode_Handheld ? "Handheld" : "Console");
            PrintCradleFirmwareVersion();
            break;

        case nn::ae::Message_VrModeChanged:
            NN_LOG("SA: Received Message_VrModeChanged\n");
            NN_LOG("SA: ae::IsVrMode() returns %s.\n", nn::ae::IsVrMode() ? "true" : "false");
            break;

        case nn::ae::Message_VrModeCurtainRequired:
            NN_LOG("SA: Received Message_VrModeCurtainRequired\n");
            break;

        case nn::ae::Message_SleepRequiredByHighTemperature:
            NN_LOG("SA: Received Message_SleepRequiredByHighTemperature\n");
            break;

        case nn::ae::Message_SleepRequiredByLowBattery:
            NN_LOG("SA: Received Message_SleepRequiredByLowBattery\n");
            break;

        case nn::ae::Message_AutoPowerDown:
            NN_LOG("SA: Received Message_AutoPowerDown\n");
            if (g_AutoSleepForbiddenMode == 2)
            {
                // SIGLO-70011
                NN_LOG("SA: But auto sleep is forbidden.\n");
                NN_LOG("SA: Invoke nn::ae::ReportUserIsActive()\n");
                nn::ae::ReportUserIsActive();
            }
            else
            {
                StartSleepSequenceImpl();
            }
            break;

        case nn::ae::Message_DetectReceivingCecSystemStandby:
            NN_LOG("SA: Received Message_DetectReceivingCecSystemStandby\n");
            StartSleepSequenceImpl();
            break;

        case nn::ae::Message_SdCardRemoved:
            nn::ae::EnterInterruptSceneProcessSection();
            NN_LOG("SA: Received Message_SdCardRemoved\n");
            nn::ae::LeaveInterruptSceneProcessSection();
            break;

        case nn::ae::Message_ControllerFirmwareUpdateSectionChanged:
            NN_LOG("SA: Received Message_ControllerFirmwareUpdateSectionChanged\n");
            NN_LOG("SA: ae::IsInControllerFirmwareUpdateSection() returns %s.\n", nn::ae::IsInControllerFirmwareUpdateSection() ? "true" : "false");
            break;

        case nn::ae::Message_RequestToShutdown:
            NN_LOG("SA: Received Message_RequestToShutdown\n");
            nn::ae::StartShutdownSequence();
            break;

        case nn::ae::Message_RequestToReboot:
            NN_LOG("SA: Received Message_RequestToReboot\n");
            nn::ae::StartRebootSequence();
            break;

        default:
            NN_LOG("SA: Received unknown message= 0x%08x\n", message);
            break;
        }
    }
} // NOLINT(impl/function_size)

//-----------------------------------------------------------------------------

void HandlePopFromOutChannelEvent()
{
    NN_LOG("SA: Received PopFromOutChannelEvent\n" );
    nn::applet::StorageHandle storageHandle;
    if ( TryPopFromOutChannel( &storageHandle, g_LibraryAppletHandle ) )
    {
        NN_ABORT_UNLESS( GetStorageSize(storageHandle) == sizeof(g_ParamBuffer) );
        NN_ABORT_UNLESS_RESULT_SUCCESS( ReadFromStorage( storageHandle, 0, g_ParamBuffer, sizeof(g_ParamBuffer) ) );
        ReleaseStorage( storageHandle );
    }
    std::strcpy( g_MessageBuffer, g_ParamBuffer );
}

const char* GetLibraryAppletExitReasonString(nn::applet::LibraryAppletExitReason reason) NN_NOEXCEPT
{
    switch (reason)
    {
    case nn::applet::LibraryAppletExitReason_Normal: return "LibraryAppletExitReason_Normal";
    case nn::applet::LibraryAppletExitReason_Canceled : return "LibraryAppletExitReason_Canceled";
    case nn::applet::LibraryAppletExitReason_Abnormal: return "LibraryAppletExitReason_Abnormal";
    case nn::applet::LibraryAppletExitReason_Unexpected: return "LibraryAppletExitReason_Unexpected";
    default: NN_UNEXPECTED_DEFAULT;
    }
}

void WaitLibraryAppletExited()
{
    // 後で Join するので必ずしもこれは必要ではない
    auto event = nn::applet::GetLibraryAppletExitEvent(g_LibraryAppletHandle);
    nn::os::WaitSystemEvent(event);

    nn::applet::JoinLibraryApplet( g_LibraryAppletHandle );
    auto exitReason = nn::applet::GetLibraryAppletExitReason(g_LibraryAppletHandle);
    NN_LOG("SA: JoinLibraryApplet: %s\n", GetLibraryAppletExitReasonString(exitReason));
    if (exitReason == nn::applet::LibraryAppletExitReason_Normal)
    {
        HandlePopFromOutChannelEvent();
    }
    nn::applet::CloseLibraryApplet( g_LibraryAppletHandle );
    g_LibraryAppletHandle = nn::applet::InvalidLibraryAppletHandle;
}

nn::applet::LibraryAppletHandle PrepareLibraryApplet( nn::applet::AppletId id, const char* inData, size_t inDataSize, bool isAllForeground ) NN_NOEXCEPT
{
    // LA の作成
    nn::applet::LibraryAppletHandle appletHandle;
    NN_ABORT_UNLESS_RESULT_SUCCESS( nn::applet::CreateLibraryApplet( &appletHandle, id, isAllForeground ? nn::applet::LibraryAppletMode_AllForeground : nn::applet::LibraryAppletMode_PartialForeground ) );

    // 入力データの push
    {
        nn::applet::StorageHandle storageHandle;
        NN_ABORT_UNLESS_RESULT_SUCCESS( nn::applet::CreateStorage(&storageHandle, inDataSize) );
        NN_ABORT_UNLESS_RESULT_SUCCESS( nn::applet::WriteToStorage(storageHandle, 0, inData, inDataSize) );
        nn::applet::PushToInChannel( appletHandle, storageHandle );
    }
    return appletHandle;
}

bool LaunchLibraryApplet1(bool isAllForeground) NN_NOEXCEPT
{
    // アプリが起動していれば、事前にアプリ起因の LA を終了させておく
    if (g_ApplicationHandle.IsValid())
    {
        NN_LOG("SA: Request exit to the application's LAs\n");
        auto exitNormally = nn::ae::RequestExitLibraryAppletOfApplicationOrTerminate(g_ApplicationHandle, nn::TimeSpan::FromSeconds(5));
        if (!exitNormally)
        {
            NN_LOG("SA: Request exit failed: Terminate Application's LAs\n");
            return false;
        }
    }

    // LA 呼出元の画像をキャプチャしておく
    nn::applet::TakeScreenShotOfCallerApplet();

    std::strcpy( g_ParamBuffer, g_MessageBuffer );
    g_LibraryAppletHandle = PrepareLibraryApplet( nn::applet::AppletId_LibraryAppletAuth, g_ParamBuffer, sizeof(g_ParamBuffer), isAllForeground );
    if (g_LibraryAppletHandle == nn::applet::InvalidLibraryAppletHandle)
    {
        NN_LOG("SA: Request exit failed: Cannot create LA1 handle\n");
        return false;
    }
    NN_ABORT_UNLESS_RESULT_SUCCESS( nn::applet::StartLibraryApplet( g_LibraryAppletHandle, NULL ) );
    return true;
}

bool RequestExitLibraryApplet1() NN_NOEXCEPT
{
    // LA が起動していれば、その LA に終了を要求する
    if (g_LibraryAppletHandle != nn::applet::InvalidLibraryAppletHandle)
    {
        NN_LOG("SA: RequestExitLibraryAppletAndWaitOrTerminate( LA1 )\n");
        return nn::applet::RequestExitLibraryAppletAndWaitOrTerminate( g_LibraryAppletHandle, nn::TimeSpan::FromSeconds(5) );
    }
    return true;
}


//-----------------------------------------------------------------------------
// LA 制御用のワーカースレッド
//
enum WorkerCommand
{
    WorkerCommand_None,
    WorkerCommand_LaunchLA,
    WorkerCommand_LaunchLAWithPartialFG,
};

const int CommandBufferCount = 8;
uintptr_t g_CommandBuffer[CommandBufferCount];
nn::os::MessageQueue g_CommandQueue(g_CommandBuffer, CommandBufferCount);

void LibraryApplet1HandleThread(void *arg)
{
    NN_UNUSED(arg);

    bool isLibraryAppletLaunched = false;

    while (g_ThreadControl.WaitForInFocusOrExitRequested())
    {
        uintptr_t cmd;
        g_CommandQueue.Receive(&cmd);
        switch (cmd)
        {
            // LA1 の通常起動
            case WorkerCommand_LaunchLA:
            {
                NN_LOG("SA: Invoke nn::applet::CreateLibraryApplet() (LA1)\n");
                isLibraryAppletLaunched = LaunchLibraryApplet1(true);
                break;
            }
            // LA1 の Partial-FG 起動
            case WorkerCommand_LaunchLAWithPartialFG:
            {
                NN_LOG("SA: Invoke nn::applet::CreateLibraryApplet() (LA1, Partial FG)\n");
                isLibraryAppletLaunched = LaunchLibraryApplet1(false);
                break;
            }
            default:
            {
                NN_LOG("SA: Unknown internal command: 0x%08x\n", cmd);
                break;
            }
        }

        // LA1 の終了を待機
        if (isLibraryAppletLaunched)
        {
            WaitLibraryAppletExited();
            isLibraryAppletLaunched = false;
        }
    }
}


//-----------------------------------------------------------------------------
//  Pad 入力ハンドリングスレッド
//
void DebugPadThreadFunction(void *arg)
{
    NN_UNUSED(arg);

    nn::hid::DebugPadState debugPadState;
    nn::hid::DebugPadButtonSet pressed;
    nn::hid::DebugPadButtonSet trigger;

    while (g_ThreadControl.WaitForInFocusOrExitRequested())
    {
        Graphics::WaitVsync();

        // ボタン情報の更新
        nnt::applet::hid::GetConvNpadState( &debugPadState );
        trigger = ~pressed & debugPadState.buttons;
        pressed = debugPadState.buttons;

        // Partial Foreground な LA が起動していたら何もしない
        if ( g_LibraryAppletHandle != nn::applet::InvalidLibraryAppletHandle )
        {
            continue;
        }

        // DebugPad の ZL/ZR/L/R ボタンを 4 つ同時押しながら～の操作
        if ((pressed & nn::hid::DebugPadButton::ZL::Mask).IsAnyOn() &&
            (pressed & nn::hid::DebugPadButton::ZR::Mask).IsAnyOn() &&
            (pressed & nn::hid::DebugPadButton::L::Mask).IsAnyOn() &&
            (pressed & nn::hid::DebugPadButton::R::Mask).IsAnyOn())
        {
            // ZL/ZR/L/R 同時押し ＋ SELECT(-) 押しでシャットダウンを開始
            if ((trigger & nn::hid::DebugPadButton::Select::Mask).IsAnyOn())
            {
                NN_LOG("APP: Invoke nn::ae::StartShutdownSequence()\n");
                nn::ae::StartShutdownSequence();
            }

            // ZL/ZR/L/R 同時押し ＋ START(+) 押しで再起動を開始
            if ((trigger & nn::hid::DebugPadButton::Start::Mask).IsAnyOn())
            {
                NN_LOG("APP: Invoke nn::ae::StartRebootSequence()\n");
                nn::ae::StartRebootSequence();
            }
            continue;
        }

        // DebugPad の A が押された
        if ( (trigger & nn::hid::DebugPadButton::A::Mask).IsAnyOn() )
        {
            bool isSystemApplication = false;
            nn::ae::ApplicationHandle applicationHandle;
            if ( g_Info.isAppLaunched == false )
            {
                if ( (pressed & nn::hid::DebugPadButton::Up::Mask).IsAnyOn() )
                {
                    NN_LOG("SA: Launch Starter\n");
                    applicationHandle = nn::ae::CreateSystemApplication( StarterId );
                    g_Info.isAppStarter = true;
                    isSystemApplication = true;
                }
                else
                {
                    NN_LOG("SA: Launch application\n");
                    applicationHandle = nn::ae::CreateApplication( ApplicationId );
                    //nn::applet::SetNextApplicationExitTimer(0, 0);
                }
                PrintApplicationLaunchRequestInfo(applicationHandle);

                {
                    std::strcpy( g_ParamBuffer, g_MessageBuffer );
                    nn::applet::StorageHandle storageHandle;
                    NN_ABORT_UNLESS_RESULT_SUCCESS( nn::applet::CreateStorage( &storageHandle, sizeof(g_ParamBuffer) ) );
                    NN_ABORT_UNLESS_RESULT_SUCCESS( nn::applet::WriteToStorage( storageHandle, 0, g_ParamBuffer, sizeof(g_ParamBuffer) ) );
                    nn::ae::PushApplicationLaunchParameter(applicationHandle, nn::applet::LaunchParameterKind_User, storageHandle);
                }

                // ユーザアカウントを選択
                if (!isSystemApplication && !SelectUserAccountAutoIfRequired(applicationHandle))
                {
                    NN_LOG("======================================================================\n");
                    NN_LOG("SA: Gave up to start application (ApplicationId=0x%016llx)\n", nn::ae::GetApplicationId(applicationHandle));
                    NN_LOG("======================================================================\n");
                    nn::ae::CloseApplicationHandle(applicationHandle);
                    continue;
                }

                g_Info.isWaitForLaunch = false;
                g_Info.isAppLaunched = true;
                g_ApplicationHandle = applicationHandle;

                // アプリ起動前に LastFG キャプチャバッファをクリア（念のため）
                nn::ae::ClearLastForegroundCaptureBuffer(true);

                // アプリの起動
                nn::ae::StartApplication( g_ApplicationHandle );
                CheckApplicationProperties( g_ApplicationHandle );
                nn::ae::RequestApplicationGetForeground( g_ApplicationHandle );
            }
            else
            {
                NN_LOG("SA: Application already exist. Do nothing\n");
            }
        }
        // DebugPad の X が押された
        else if ( (trigger & nn::hid::DebugPadButton::X::Mask).IsAnyOn() )
        {
            if ( (pressed & nn::hid::DebugPadButton::Up::Mask).IsAnyOn() )
            {
                if ( g_Info.isAppLaunched )
                {
                    NN_LOG("SA: Invoke nn::ae::RequestExitLibraryAppletOfApplication()\n");
                    nn::ae::RequestExitLibraryAppletOfApplication( g_ApplicationHandle );
                }
            }
            else if ( (pressed & nn::hid::DebugPadButton::Down::Mask).IsAnyOn() )
            {
                NN_LOG("SA: Invoke nn::applet::RequestExitLibraryAppletAndWait()\n");
                if (g_LibraryAppletHandle != nn::applet::InvalidLibraryAppletHandle)
                {
                    nn::applet::RequestExitLibraryAppletAndWait(g_LibraryAppletHandle);
                }
            }
            // アプリの終了要求
            else if ( g_Info.isAppLaunched == true )
            {
                NN_LOG("SA: Invoke nn::ae::RequestApplicationExit()\n");
                nn::ae::RequestApplicationExit( g_ApplicationHandle );
            }
            else
            {
                NN_LOG("SA: Application doesn't launched. Do nothing.\n");
            }
        }
        // DebugPad の Y が押された
        else if ( (trigger & nn::hid::DebugPadButton::Y::Mask).IsAnyOn() &&
                  (g_LibraryAppletHandle == nn::applet::InvalidLibraryAppletHandle) )
        {
            // UP + Y なら Partial Foreground で LA1 起動
            if ( (pressed & nn::hid::DebugPadButton::Up::Mask).IsAnyOn() )
            {
                g_CommandQueue.Send(WorkerCommand_LaunchLAWithPartialFG);
            }
            else
            {
                g_CommandQueue.Send(WorkerCommand_LaunchLA);
            }
        }
        // DebugPad の B が押された
        else if ( (trigger & nn::hid::DebugPadButton::B::Mask).IsAnyOn() )
        {
            if ( (pressed & nn::hid::DebugPadButton::Up::Mask).IsAnyOn() )
            {
                // UP + B ならオートスリープ時間をデフォルトに戻す
                // オートスリープ禁止（無視）も解除
                NN_LOG( "SA: Invoke nn::applet::OverrideAutoSleepTimeAndDimmingTime(0, 0, 0, 0)\n" );
                g_AutoSleepForbiddenMode = 0;
                nn::applet::OverrideAutoSleepTimeAndDimmingTime(0, 0, 0, 0);
            }
            else if ( (pressed & nn::hid::DebugPadButton::Down::Mask).IsAnyOn() )
            {
                // DOWN + B ならオートスリープ時間を 10 秒に設定する
                // もう一度 DOWN + B でオートスリープを禁止（無視）する
                switch (g_AutoSleepForbiddenMode)
                {
                    case 0: // 1 入力目
                    {
                        g_AutoSleepForbiddenMode = 1;
                        NN_LOG( "SA: Invoke nn::applet::OverrideAutoSleepTimeAndDimmingTime(10, 10, 5, 5)\n" );
                        nn::applet::OverrideAutoSleepTimeAndDimmingTime(10, 10, 5, 5);
                        break;
                    }
                    case 1: // 2 入力目（SIGLO-70011）
                    {
                        g_AutoSleepForbiddenMode = 2;
                        NN_LOG( "SA: Forbid auto power sleep\n" );
                        break;
                    }
                    default: // それ以降
                    {
                        break;
                    }
                }
            }
            else if ( (pressed & nn::hid::DebugPadButton::Left::Mask).IsAnyOn() )
            {
                // LEFT + B ならメディア再生中フラグをトグル
                g_Info.isInMediaPlayback = !g_Info.isInMediaPlayback;
                NN_LOG( "SA: Invoke nn::applet::SetMediaPlaybackState(%s)\n", g_Info.isInMediaPlayback ? "true" : "false" );
                nn::applet::SetMediaPlaybackState(g_Info.isInMediaPlayback);
            }
            else if ( (pressed & nn::hid::DebugPadButton::Right::Mask).IsAnyOn() )
            {
                // RIGHT + B で、HOME 長押し時間を 3s に
                NN_LOG( "SA: Invoke nn::ae::SetDefaultHomeButtonLongPressTime(nn::TimeSpan::FromSeconds(3))\n");
                nn::ae::SetDefaultHomeButtonLongPressTime(nn::TimeSpan::FromSeconds(3));
            }
        }

        // DebugPad の ZR を押しながら～の操作
        if ((pressed & nn::hid::DebugPadButton::ZR::Mask).IsAnyOn())
        {
            // ZR + 右 で BGM の on/off
            if ((trigger & nn::hid::DebugPadButton::Right::Mask).IsAnyOn())
            {
                if( audioState == AudioState_Stress )
                {
                    NN_LOG( "SA: Disable Audio\n" );
                    g_Info.isAudioEnabled = false;
                    audioState = AudioState_Disabled;
                    Graphics::SetAudioStatus( g_Info.isAudioEnabled );
                    nnt::applet::audioout::PauseBgm();
                    nnt::applet::audioout::PauseSine();
                }
                else if( audioState == AudioState_Disabled )
                {
                    NN_LOG( "SA: Enable Audio\n" );
                    g_Info.isAudioEnabled = true;
                    audioState = AudioState_Enabled;
                    Graphics::SetAudioStatus( g_Info.isAudioEnabled );
                    nnt::applet::audioout::PlayBgm();
                }
                else
                {
                    NN_LOG( "SA: Add Audio\n" );
                    g_Info.isAudioEnabled = true;
                    audioState = AudioState_Stress;
                    Graphics::SetAudioStatus( g_Info.isAudioEnabled );
                    nnt::applet::audioout::PlaySine();
                }
            }
        }

        // DebugPad の R で負荷を増やす
        if ((pressed & nn::hid::DebugPadButton::R::Mask).IsAnyOn())
        {
            // 上ボタンを同時に押した場合 CPU の負荷を増加
            if ((pressed & nn::hid::DebugPadButton::Up::Mask).IsAnyOn())
            {
                Graphics::SetSleepCount(Graphics::GetSleepCount() + 1);
            }
            else if ((pressed & nn::hid::DebugPadButton::Down::Mask).IsAnyOn())
            {
                Graphics::SetLiteObjectCount(Graphics::GetLiteObjectCount() + 1);
            }
            else
            {
                Graphics::SetObjectCount(Graphics::GetObjectCount() + 1);
            }
        }
        // DebugPad の L で負荷を減らす
        if ((pressed & nn::hid::DebugPadButton::L::Mask).IsAnyOn())
        {
            // 上ボタンを同時に押した場合 CPU の負荷を増加
            if ((pressed & nn::hid::DebugPadButton::Up::Mask).IsAnyOn())
            {
                Graphics::SetSleepCount(Graphics::GetSleepCount() - 1);
            }
            else if ((pressed & nn::hid::DebugPadButton::Down::Mask).IsAnyOn())
            {
                Graphics::SetLiteObjectCount(Graphics::GetLiteObjectCount() - 1);
            }
            else
            {
                Graphics::SetObjectCount(Graphics::GetObjectCount() - 1);
            }
        }
    }
}   // NOLINT(readability/fn_size)

//-----------------------------------------------------------------------------
//  グラフィックスレンダリング
//
void GraphicsThreadFunction(void *arg)
{
    NN_UNUSED(arg);

    // 毎フレームのレンダリング
    while (g_ThreadControl.WaitForInFocusOrExitRequested())
    {
        Graphics::GraphicsRenderer();
    }
}

//-----------------------------------------------------------------------------
//  周期キャプチャ機能用スレッド
//
void CaptureThreadFunction(void *arg)
{
    NN_UNUSED(arg);

    int timer = 0;

    while (!g_ThreadControl.IsExitRequested())
    {
        nn::os::SleepThread( nn::TimeSpan::FromMilliSeconds( 50 ) );
        timer += 50;

        if ( timer >= 1000 )
        {
            timer = 0;
            bool isForeground = g_ThreadControl.TryWaitForInFocus();

            if(isForeground)
            {
                // LastForegroundCaptureBuffer の更新
                UpdateLastForegroundCaptureBuffer();
            }

            // インデックスの取得はいつでもできる
            PrintCaptureBufferInfo(isForeground);
        }
    }
}

//-----------------------------------------------------------------------------
//  オーディオ再生スレッド
//
void AudioThreadFunction(void *arg)
{
    NN_UNUSED(arg);

    while (g_ThreadControl.WaitForInFocusOrExitRequested())
    {
        nnt::applet::audioout::WaitAudio();
        nnt::applet::audioout::UpdateAudio();
        Graphics::WaitVsync();
    }
}

//-----------------------------------------------------------------------------

void* Allocate( size_t size )
{
    return malloc( size );
}

void Deallocate( void* ptr, size_t )
{
    free( ptr );
}

void Initialize()
{
    // fs 用のアロケータをセット
    nn::fs::SetAllocator(Allocate, Deallocate);

    // グラフィックス関連の初期化
    Graphics::Rgba clearColor = { 0.3f, 0.1f, 0.1f, 1.0f };
    Graphics::FrameworkMode frameworkMode = Graphics::FrameworkMode_DeferredExecution;
    Graphics::InitializeGraphics( clearColor, "SystemApplet", frameworkMode, false );
    Graphics::EnableLinearTextures("LastApplicationCaptureBuffer",
                                   "LastForegroundCaptureBuffer");

    std::strcpy( g_MessageBuffer, "SA " );
    Graphics::SetMessage( g_MessageBuffer );

    // アプレット通知メッセージ用イベントの初期化
    nn::ae::InitializeNotificationMessageEvent( &g_MessageSystemEvent );

    // スリープの事前ハンドリングを行う
    nn::ae::BeginSleepHandling();

    // Audio の初期化
    nnt::applet::audioout::InitializeAudio();
    nnt::applet::audioout::LoadBgm( nnt::applet::audioout::Bgm_A );

    // DebugPad の初期化
    nnt::applet::hid::InitializeConvNpad();
}

void Finalize()
{
    // Audio 関連の終了処理
    nnt::applet::audioout::FinalizeAudio();

    // グラフィックス関連の終了処理
    Graphics::FinalizeGraphics();
}

} // namespace

void SystemAppletMenuMain(nn::ae::SystemAppletParameters* param)
{
    NN_LOG("SA: Operating Core is #%d.\n", nn::os::GetCurrentCoreNumber());
    {
        NN_LOG("SA: My own ARUID= 0x%016llx\n", nn::applet::GetAppletResourceUserId());
        nn::applet::AppletResourceUserId aruid = {};
        auto result = nn::applet::GetAppletResourceUserIdOfCallerApplet(&aruid);
        NN_LOG("SA: Parent ARUID= 0x%016llx (result=0x%08x)\n", aruid, result.GetInnerValueForDebug());
    }

    // キャプチャバッファのクリア（API 呼出確認）
    nn::ae::ClearLastApplicationCaptureBuffer(true, 0xffff0000);  // Blue
    nn::ae::ClearLastForegroundCaptureBuffer(true, 0x00000000);   // Transparent
    nn::ae::ClearCallerAppletCaptureBuffer(true, 0xff00ff00);     // Green
    nn::ae::ClearAppletTransitionTextureBuffer();

    // 出画前ハンドリングの宣言
    nn::ae::EnableHandlingForRequestToDisplay();

    // 初期化処理
    Initialize();

    // スレッドを生成する
    NN_ABORT_UNLESS_RESULT_SUCCESS( nn::os::CreateThread( &g_AudioThread, AudioThreadFunction, NULL, g_AudioThreadStack, ThreadStackSize, AppletPriorityForAudio ) );
    NN_ABORT_UNLESS_RESULT_SUCCESS( nn::os::CreateThread( &g_MessageThread, HandleAeMessage, NULL, g_MessageThreadStack, ThreadStackSize, AppletPriorityForLimit1msec ) );
    NN_ABORT_UNLESS_RESULT_SUCCESS( nn::os::CreateThread( &g_DebugPadThread, DebugPadThreadFunction, NULL, g_DebugPadThreadStack, ThreadStackSize, AppletPriorityForLimit1msec + 1 ) );
    NN_ABORT_UNLESS_RESULT_SUCCESS( nn::os::CreateThread( &g_LibraryApplet1HandleThread, LibraryApplet1HandleThread, NULL, g_LibraryApplet1HandleThreadStack, ThreadStackSize, AppletPriorityForLimit10msec ) );
    NN_ABORT_UNLESS_RESULT_SUCCESS( nn::os::CreateThread( &g_GraphicsThread, GraphicsThreadFunction, NULL, g_GraphicsThreadStack, ThreadStackSize, AppletPriorityForOver10msec ) );
    NN_ABORT_UNLESS_RESULT_SUCCESS( nn::os::CreateThread( &g_CaptureThread, CaptureThreadFunction, NULL, g_CaptureThreadStack, ThreadStackSize, AppletPriorityForOver10msec + 1 ) );

    // スレッドの実行を開始する
    nn::os::StartThread( &g_MessageThread );
    nn::os::StartThread( &g_LibraryApplet1HandleThread );
    nn::os::StartThread( &g_GraphicsThread );
    nn::os::StartThread( &g_DebugPadThread );
    nn::os::StartThread( &g_CaptureThread );
    nn::os::StartThread( &g_AudioThread );

    // API が呼べることだけ確認しておく
    nn::ae::SetScreenShotPermission(nn::ae::ScreenShotPermission_Forbid);
    nn::ae::SetScreenShotPermission(nn::ae::ScreenShotPermission_Permit);
    nn::ae::SetScreenShotPermission(nn::ae::ScreenShotPermission_Inherit);

    nn::ae::LoadAndApplyIdlePolicySettings();

    nn::applet::OverrideAutoSleepTimeAndDimmingTime(60, 60, 10, 10);
    nn::applet::SetMediaPlaybackState(true);
    nn::applet::SetMediaPlaybackState(false);
    nn::applet::OverrideAutoSleepTimeAndDimmingTime(0, 0, 0, 0);

    nn::applet::SetInputDetectionSourceSet( nn::applet::InputSource_Pad |
                                            nn::applet::InputSource_Sensor |
                                            nn::applet::InputSource_DebugPad |
                                            nn::applet::InputSource_TouchScreen |
                                            nn::applet::InputSource_Keyboard |
                                            nn::applet::InputSource_Mouse |
                                            nn::applet::InputSource_UsbPort |
                                            nn::applet::InputSource_SdCardSlot |
                                            nn::applet::InputSource_GameCardSlot |
                                            0 );
    nn::applet::SetInputDetectionSourceSet( nn::applet::InputSource_All );

    // スレッドの終了を待機
    nn::os::WaitThread( &g_MessageThread );
    nn::os::WaitThread( &g_LibraryApplet1HandleThread );
    nn::os::WaitThread( &g_GraphicsThread );
    nn::os::WaitThread( &g_DebugPadThread );
    nn::os::WaitThread( &g_CaptureThread );
    nn::os::WaitThread( &g_AudioThread );

    // スレッドを破棄する
    nn::os::DestroyThread( &g_MessageThread );
    nn::os::DestroyThread( &g_LibraryApplet1HandleThread );
    nn::os::DestroyThread( &g_GraphicsThread );
    nn::os::DestroyThread( &g_DebugPadThread );
    nn::os::DestroyThread( &g_CaptureThread );
    nn::os::DestroyThread( &g_AudioThread );

    // 終了処理
    Finalize();
}

NN_ALIGNAS(4096) uint8_t  g_MallocBuffer[52 * 1024 * 1024];
extern "C" void nninitStartup()
{
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::ae::SetMemoryHeapSizeWithRetry(0x400000));
    nn::init::InitializeAllocator( g_MallocBuffer, sizeof(g_MallocBuffer) );
}

extern "C" void nnMain()
{
    nn::ns::Initialize();
    nn::account::InitializeForAdministrator();

    nn::applet::SetOnAddLibraryApplet([] (nn::applet::LibraryAppletHandle handle)
    {
        NN_ABORT_UNLESS(g_LibraryAppletHandle == handle);
        NN_ABORT_UNLESS(g_AddedLibraryAppletHandle == nn::applet::InvalidLibraryAppletHandle);
        g_AddedLibraryAppletHandle = handle;
    });
    nn::applet::SetOnRemoveLibraryApplet([] (nn::applet::LibraryAppletHandle handle)
    {
        NN_ABORT_UNLESS(g_LibraryAppletHandle == handle);
        NN_ABORT_UNLESS(g_AddedLibraryAppletHandle == handle);
        g_AddedLibraryAppletHandle = nn::applet::InvalidLibraryAppletHandle;
    });
    nn::ae::InvokeSystemAppletMain( nn::ae::AppletId_SystemAppletMenu, SystemAppletMenuMain );
}
