﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstdlib>
#include <cstring>
#include <mutex>

#include <nn/nn_Assert.h>
#include <nn/nn_Log.h>

#include <nn/init.h>
#include <nn/os.h>
#include <nn/fs.h>
#include <nn/vi.h>
#include <nn/gfx.h>
#include <nn/gfx/util/gfx_DebugFontTextWriter.h>
#include <nn/result/result_ErrorResult.h>
#include <nn/util/util_Color.h>

#define NN_PERF_PROFILE_ENABLED
#include <nn/perf.h>

#include <nns/gfx/gfx_GraphicsFramework.h>
#include <nns/gfx/gfx_PrimitiveRenderer.h>
#include <nns/gfx/gfx_PrimitiveRendererMeshRes.h>
#include <nns/gfx/gfx_PrimitiveRendererMeterDrawer.h>

#include "Common.h"
#include "Graphics.h"

#include <nn/fs/fs_SdCardForDebug.h>

#define ANIMATION_ENABLED

// 自動でアプリが使用可能な最大 GPU 時間を計測します。
//#define AUTO_CHECK_MAXGPURESOURCE_ENABLED

// PERF にカスタム属性 (CpuCached|GpuUncached) のメモリプールを使用します
// コメントアウトした場合は GraphicsFramework::MemoryPoolType_Data (CpuCached|GpuCached) を使用します
#define PERF_CUSTOM_MEMORYPOOL_ENABLE

namespace Graphics {

namespace {

// クリアカラー
Rgba g_ClearColor;

// タイトル
std::string g_Title;

// フレームワークモード
FrameworkMode g_CurrentFrameworkMode;

// partial かどうか
bool g_IsPartialForeground = false;

// 線形テクスチャを有効にするか
bool g_IsEnabledLinearTextures = false;
const char* g_pCaptureBufferName1 = "";
const char* g_pCaptureBufferName2 = "";

// 線形テクスチャの数
const size_t g_LinearTextureNum = 2;

// VSync イベント
// 複数スレッドの Wait に対応する
class VsyncEvent
{
private:
    nn::os::Mutex m_Mutex{false};
    nn::os::ConditionVariable m_Cond;
    uint64_t m_SignalCount{0};

public:
    // Wait() している全てのスレッドを動作再開する
    void Signal() NN_NOEXCEPT
    {
        std::lock_guard<nn::os::Mutex> lock(m_Mutex);

        ++m_SignalCount;
        m_Cond.Broadcast();
    }

    // Signal() されるまで待機する（複数スレッドが待機可能）
    void Wait() NN_NOEXCEPT
    {
        std::lock_guard<nn::os::Mutex> lock(m_Mutex);

        auto currentSignalCount = m_SignalCount;
        while (currentSignalCount == m_SignalCount)
        {
            m_Cond.Wait(m_Mutex);
        }
    }
};
VsyncEvent g_VsyncEvent;

// メッセージ
const char *g_pMessage = NULL;

// 描画するオブジェクト数
int g_ObjectCount = 0;
int g_LiteObjectCount = 0;

// スリープ時間 g_SleepCount * 200us
int g_SleepCount = 0;

// バッファ数 (2 以上を指定してください)
const int g_BufferCount = 2;

// Audio が再生中かどうか
bool g_IsAudioEnabled = false;

// Home Button 押し待ちモード
bool g_IsWaitPressHomeButton = false;
unsigned int g_FrameCount = 0;
unsigned int g_SavedFrameCount = 0;

//-----------------------------------------------------------------------------
int g_TextureDescriptorBaseIndex = 0;
int g_SamplerDescriptorBaseIndex = 0;

// グラフィックスフレームワークを初期化
nns::gfx::GraphicsFramework g_GraphicsFramework;
nns::gfx::GraphicsFramework::DebugFontTextWriter g_Writer;
static int g_BufferIndex = 0;
void InitializeGraphicsFramework(bool isColorBufferEnabled)
{
    nns::gfx::GraphicsFramework::FrameworkInfo fwInfo;
    fwInfo.SetDefault();
    fwInfo.SetDisplayHeight(720);
    fwInfo.SetDisplayWidth(1280);
    fwInfo.SetColorBufferEnabled(isColorBufferEnabled);
    fwInfo.SetDepthStencilBufferEnabled(false);
    fwInfo.SetMemoryPoolSize(nns::gfx::GraphicsFramework::MemoryPoolType_CommandBuffer, 8 * 1024 * 1024);
    fwInfo.SetMemoryPoolSize(nns::gfx::GraphicsFramework::MemoryPoolType_RenderTarget, (isColorBufferEnabled ? 12 : 8) * 1024 * 1024);
    fwInfo.SetMemoryPoolSize(nns::gfx::GraphicsFramework::MemoryPoolType_Shader, 0 * 1024 * 1024);
    fwInfo.SetMemoryPoolSize(nns::gfx::GraphicsFramework::MemoryPoolType_Data, 1 * 1024 * 1024);
    fwInfo.SetRootCommandBufferCommandMemorySize(2 * 1024 * 1024);
    fwInfo.SetBufferCount(g_BufferCount);
    fwInfo.SetSwapChainBufferCount(g_BufferCount);
    g_GraphicsFramework.Initialize(fwInfo);
}

// グラフィックスフレームワークを破棄
void FinalizeGraphicsFramework()
{
    g_GraphicsFramework.Finalize();
}

// 線形テクスチャを初期化
nn::gfx::Texture g_LinearTexture[g_LinearTextureNum];
ptrdiff_t g_LinearTextureOffset[g_LinearTextureNum];
ptrdiff_t g_TextureMemoryPoolOffset[g_LinearTextureNum];
void InitializeLinearTextures()
{
    nn::gfx::Texture::InfoType info;
    info.SetDefault();
    info.SetGpuAccessFlags( nn::gfx::GpuAccess_Texture );
    info.SetWidth( g_GraphicsFramework.GetDisplayWidth() );
    info.SetHeight( g_GraphicsFramework.GetDisplayHeight() );
    info.SetImageStorageDimension( nn::gfx::ImageStorageDimension_2d );
    info.SetImageFormat( nn::gfx::ImageFormat_R8_G8_B8_A8_Unorm );
    info.SetMipCount( 1 );
    info.SetTileMode( nn::gfx::TileMode_Linear );

    for (int i = 0; i < g_LinearTextureNum; i++)
    {
        g_LinearTextureOffset[i] = 0;

        size_t alignment = nn::gfx::Texture::CalculateMipDataAlignment(g_GraphicsFramework.GetDevice(), info );
        size_t size = nn::gfx::Texture::CalculateMipDataSize(g_GraphicsFramework.GetDevice(), info );
        size_t pitch = nn::gfx::Texture::GetRowPitch(g_GraphicsFramework.GetDevice(), info );
        g_TextureMemoryPoolOffset[i] = g_GraphicsFramework.AllocatePoolMemory(nns::gfx::GraphicsFramework::MemoryPoolType_ConstantBuffer, size, alignment);
        nn::gfx::MemoryPool* memoryPool = g_GraphicsFramework.GetMemoryPool(nns::gfx::GraphicsFramework::MemoryPoolType_ConstantBuffer);
        uint8_t* pixels = nn::util::BytePtr(memoryPool->Map(), g_TextureMemoryPoolOffset[i]).Get< uint8_t >();
        for ( int y = 0; y < info.GetHeight(); ++y )
        {
            uint8_t* row = pixels + y * pitch;
            for ( int x = 0; x < info.GetWidth(); ++x )
            {
                uint8_t* pixel = row + x * 4;
                pixel[ 0 ] = static_cast< uint8_t >( x );
                pixel[ 1 ] = static_cast< uint8_t >( y );
                pixel[ 2 ] = 0x00;
                pixel[ 3 ] = 0xFF;
            }
        }
        g_LinearTexture[i].Initialize(g_GraphicsFramework.GetDevice(), info, memoryPool, g_TextureMemoryPoolOffset[i], size );
        g_LinearTextureOffset[i] = g_TextureMemoryPoolOffset[i];
    }
};

// 線形テクスチャを破棄
void FinalizeLinearTextures()
{
    for (int i = 0; i < g_LinearTextureNum; i++)
    {
        g_GraphicsFramework.FreePoolMemory(nns::gfx::GraphicsFramework::MemoryPoolType_ConstantBuffer, g_TextureMemoryPoolOffset[i]);
        g_LinearTexture[i].Finalize(g_GraphicsFramework.GetDevice());
    }
}

// 線形テクスチャビューを初期化
nn::gfx::TextureView g_LinearTextureView[g_LinearTextureNum];
void InitializeLinearTextureViews()
{
    for (int i = 0; i < g_LinearTextureNum; i++)
    {
        nn::gfx::TextureView::InfoType info;
        info.SetDefault();
        info.SetImageDimension( nn::gfx::ImageDimension_2d );
        info.SetImageFormat( nn::gfx::ImageFormat_R8_G8_B8_A8_Unorm );
        info.SetTexturePtr( &g_LinearTexture[i] );
        info.SetChannelMapping( nn::gfx::ChannelMapping_Red, nn::gfx::ChannelMapping_Green,
            nn::gfx::ChannelMapping_Blue, nn::gfx::ChannelMapping_Alpha );
        g_LinearTextureView[i].Initialize(g_GraphicsFramework.GetDevice(), info );
    }
}

// 線形テクスチャビューを破棄
void FinalizeLinearTextureViews()
{
    for (int i = 0; i < g_LinearTextureNum; i++)
    {
        g_LinearTextureView[i].Finalize(g_GraphicsFramework.GetDevice());
    }
}

// サンプラを初期化
nn::gfx::Sampler g_Sampler;
void InitializeSampler()
{
    nn::gfx::Sampler::InfoType info;
    info.SetDefault();
    info.SetFilterMode(nn::gfx::FilterMode_MinLinear_MagLinear_MipPoint);
    info.SetAddressU(nn::gfx::TextureAddressMode_Mirror);
    info.SetAddressV(nn::gfx::TextureAddressMode_Mirror);
    info.SetAddressW(nn::gfx::TextureAddressMode_Mirror);
    g_Sampler.Initialize(g_GraphicsFramework.GetDevice(), info);
}

// サンプラを破棄
void FinalizeSampler()
{
    g_Sampler.Finalize(g_GraphicsFramework.GetDevice());
}

// プリミティブレンダラの初期化
nn::mem::StandardAllocator  g_PrimitiveRendererAllocator;
nns::gfx::PrimitiveRenderer::Renderer* g_pPrimitiveRenderer;
void InitializePrimitiveRenderer()
{
    nns::gfx::PrimitiveRenderer::RendererInfo info;
    info.SetDefault();
    info.SetAllocator(g_GraphicsFramework.DefaultAllocateFunction, NULL);
    info.SetAdditionalBufferSize(1024 * 4);
    info.SetDrawCallCountMax(1024 * 4);
    info.SetViewFunctionCallCountMax(1024 * 4);
    info.SetMultiBufferQuantity(g_BufferCount);

    g_pPrimitiveRenderer = nns::gfx::PrimitiveRenderer::CreateRenderer(g_GraphicsFramework.GetDevice(), info);
    g_pPrimitiveRenderer->SetScreenWidth(g_GraphicsFramework.GetDisplayWidth());
    g_pPrimitiveRenderer->SetScreenHeight(g_GraphicsFramework.GetDisplayHeight());
}

// プリミティブレンダラの破棄
void FinalizePrimitiveRenderer()
{
    nns::gfx::PrimitiveRenderer::DestroyRenderer(g_pPrimitiveRenderer, g_GraphicsFramework.GetDevice(), g_GraphicsFramework.DefaultFreeFunction, NULL);
}

// パフォーマンス計測クラスの初期化
nns::gfx::PrimitiveRenderer::MeterDrawer g_MeterDrawer;
void*  g_MeterMemory;
#ifdef PERF_CUSTOM_MEMORYPOOL_ENABLE
nn::gfx::MemoryPool g_MeterMemoryPool;
void* g_pMeterPoolMemory;
#endif
ptrdiff_t g_MeterMemoryPoolOffset;
void InitializeLoadMeter()
{
    nn::perf::LoadMeterCenterInfo info;
    info.SetCoreCount(4);
    if ((g_CurrentFrameworkMode == FrameworkMode_QueueFinish) || (g_CurrentFrameworkMode == FrameworkMode_Immediate))
    {
        info.SetGpuBufferCount(2);
    }
    else
    {
        info.SetGpuBufferCount(3);
    }
    info.SetCpuSectionCountMax(16);
    info.SetGpuSectionCountMax(500);

    // 計測で使用するメモリの確保
    size_t memorySize = NN_PERF_GET_BUFFER_SIZE(info);
    size_t memoryAlignment = NN_PERF_GET_BUFFER_ALIGNMENT();
    g_MeterMemory = g_GraphicsFramework.AllocateMemory(memorySize, memoryAlignment);

    // メモリプールの確保
#ifdef PERF_CUSTOM_MEMORYPOOL_ENABLE
    nn::gfx::MemoryPool::InfoType memoryPoolInfo;
    memoryPoolInfo.SetDefault();
    memoryPoolInfo.SetMemoryPoolProperty(nn::gfx::MemoryPoolProperty_CpuCached | nn::gfx::MemoryPoolProperty_GpuUncached);

    size_t memoryPoolSize = NN_PERF_GET_MEMORY_POOL_SIZE(g_GraphicsFramework.GetDevice(), info);
    size_t alignment = NN_PERF_GET_MEMORY_POOL_ALIGNMENT(g_GraphicsFramework.GetDevice(), info);
    alignment = nn::util::align_up(alignment, nn::gfx::MemoryPool::GetPoolMemoryAlignment(g_GraphicsFramework.GetDevice(), memoryPoolInfo));
    memoryPoolSize = nn::util::align_up(memoryPoolSize, nn::gfx::MemoryPool::GetPoolMemorySizeGranularity(g_GraphicsFramework.GetDevice(), memoryPoolInfo));
    g_pMeterPoolMemory = g_GraphicsFramework.AllocateMemory(memoryPoolSize, alignment);
    NN_ASSERT_NOT_NULL(g_pMeterPoolMemory);
    memoryPoolInfo.SetPoolMemory(g_pMeterPoolMemory, memoryPoolSize);

    g_MeterMemoryPool.Initialize(g_GraphicsFramework.GetDevice(), memoryPoolInfo);
    g_MeterMemoryPoolOffset = 0;

    NN_PERF_INITIALIZE_METER(g_GraphicsFramework.GetDevice(), info,
        g_MeterMemory, memorySize,
        &g_MeterMemoryPool, g_MeterMemoryPoolOffset, memoryPoolSize);
#else
    size_t memoryPoolSize = NN_PERF_GET_MEMORY_POOL_SIZE(g_GraphicsFramework.GetDevice(), info);
    g_MeterMemoryPoolOffset = g_GraphicsFramework.AllocatePoolMemory(nns::gfx::GraphicsFramework::MemoryPoolType_Data, memoryPoolSize, NN_PERF_GET_MEMORY_POOL_ALIGNMENT(g_GraphicsFramework.GetDevice(), info));
    nn::gfx::MemoryPool* meterMemoryPool = g_GraphicsFramework.GetMemoryPool(nns::gfx::GraphicsFramework::MemoryPoolType_Data);
    NN_PERF_INITIALIZE_METER(g_GraphicsFramework.GetDevice(), info,
        g_MeterMemory, memorySize,
        meterMemoryPool, g_MeterMemoryPoolOffset, memoryPoolSize);
    NN_ASSERT(g_MeterMemoryPoolOffset != nn::gfx::util::MemoryPoolAllocator::InvalidOffset, "memoryPool shortage");
#endif
}

// パフォーマンス計測クラスの破棄
void FinalizeLoadMeter()
{
    NN_PERF_FINALIZE_METER(g_GraphicsFramework.GetDevice());
#ifdef PERF_CUSTOM_MEMORYPOOL_ENABLE
    g_MeterMemoryPool.Finalize(g_GraphicsFramework.GetDevice());
    g_GraphicsFramework.FreeMemory(g_pMeterPoolMemory);
#else
    g_GraphicsFramework.FreePoolMemory(nns::gfx::GraphicsFramework::MemoryPoolType_Data, g_MeterMemoryPoolOffset);
#endif
    g_GraphicsFramework.FreeMemory(g_MeterMemory);
}

nns::gfx::GraphicsFramework::CommandBuffer g_CommandBufferBegin[g_BufferCount];
nns::gfx::GraphicsFramework::CommandBuffer g_CommandBufferEnd[g_BufferCount];
void InitializeQueryCommandBuffer()
{
    nn::gfx::CommandBuffer::InfoType info;
    info.SetDefault();
    info.SetQueueCapability(nn::gfx::QueueCapability_Graphics);
    info.SetCommandBufferType(nn::gfx::CommandBufferType_Direct);
    size_t commandMemorySize = 1024 * 1024 * 1;
    size_t controlMemorySize = 256;

    for (int bufferIndex = 0; bufferIndex < g_BufferCount; bufferIndex++)
    {
        g_GraphicsFramework.InitializeCommandBuffer(&g_CommandBufferBegin[bufferIndex], info, commandMemorySize, controlMemorySize);
        g_GraphicsFramework.InitializeCommandBuffer(&g_CommandBufferEnd[bufferIndex], info, commandMemorySize, controlMemorySize);
    }
}

void FinalizeQueryCommandBuffer()
{
    for (int bufferIndex = 0; bufferIndex < g_BufferCount; bufferIndex++)
    {
        g_GraphicsFramework.FinalizeCommandBuffer(&g_CommandBufferBegin[bufferIndex]);
        g_GraphicsFramework.FinalizeCommandBuffer(&g_CommandBufferEnd[bufferIndex]);
    }
}

// コマンド作成
void MakeCommand( unsigned int frame )
{
#ifdef AUTO_CHECK_MAXGPURESOURCE_ENABLED
    // 最大使用可能 GPU 時間の自動計測
    if (g_ObjectCount < 25)
    {
        g_ObjectCount++;
    }
    else
    {
        static int objectCountMax = 160;
        static int stableFrameCount = 0;
        static int64_t gpuLoadSum = 0;

        nn::perf::CpuMeter* pCpuMeter = NN_PERF_GET_FRAME_METER();
        float fps = 1000.f / (static_cast<float>(pCpuMeter->GetLastTotalSpan().GetNanoSeconds()) / 1000000.0f);

        if (fps < 45 || stableFrameCount >= 1800)
        {
            if (g_LiteObjectCount > 10)
            {
                if (stableFrameCount == 0)
                {
                    NN_LOG("objectCountMax=%d(act %d), stableFrameCount=%d\n", objectCountMax, g_LiteObjectCount, stableFrameCount);
                }
                else
                {
                    NN_LOG("objectCountMax=%d(act %d), stableFrameCount=%d, gpuAverage=%dus\n", objectCountMax, g_LiteObjectCount, stableFrameCount, gpuLoadSum / stableFrameCount);
                }
                gpuLoadSum = 0;
                g_LiteObjectCount = 0;
                stableFrameCount = 0;
                objectCountMax--;
            }
        }

        if (g_LiteObjectCount < objectCountMax)
        {
            g_LiteObjectCount++;
        }
        else
        {
            nn::perf::GpuMeter* pGpuMeter = NN_PERF_GET_GPU_METER();
            nn::perf::LoadMeterBase::Section mainSection = pGpuMeter->GetLastResult(0);
            gpuLoadSum += mainSection.end.ToTimeSpan().GetMicroSeconds() - mainSection.begin.ToTimeSpan().GetMicroSeconds();
            stableFrameCount++;
        }
    }
#endif

    g_FrameCount = frame;

    g_pPrimitiveRenderer->Update(g_BufferIndex);

    nn::gfx::CommandBuffer* pCommandBuffer = g_GraphicsFramework.GetRootCommandBuffer(g_BufferIndex);
    g_GraphicsFramework.BeginFrame(g_BufferIndex);
    {
        //NN_PERF_SET_COLOR_GPU(nn::util::Color4u8::Red());
        //NN_PERF_BEGIN_MEASURE_NAME_GPU(pCommandBuffer, "All");

        int scanBufferIndex = g_GraphicsFramework.GetNextScanBufferIndex();
        nn::gfx::ColorTargetView* pTarget = g_IsPartialForeground ? g_GraphicsFramework.GetColorTargetView() : g_GraphicsFramework.GetScanBufferView(scanBufferIndex);

        pCommandBuffer->ClearColor(pTarget, g_ClearColor.r, g_ClearColor.g, g_ClearColor.b, g_ClearColor.a, NULL);
        //pCommandBuffer->ClearDepthStencil(g_GraphicsFramework.GetDepthStencilView(), 1.0f, 0, nn::gfx::DepthStencilClearMode_DepthStencil, nullptr);

        pCommandBuffer->SetRenderTargets(1, &pTarget, g_GraphicsFramework.GetDepthStencilView());
        pCommandBuffer->SetViewportScissorState(g_GraphicsFramework.GetViewportScissorState());
        pCommandBuffer->SetRasterizerState(g_GraphicsFramework.GetRasterizerState(nns::gfx::GraphicsFramework::RasterizerStateType::RasterizerStateType_FillSolid_CullNone));

        pCommandBuffer->InvalidateMemory(nn::gfx::GpuAccess_Texture | nn::gfx::GpuAccess_IndexBuffer
            | nn::gfx::GpuAccess_ConstantBuffer | nn::gfx::GpuAccess_VertexBuffer);

        g_pPrimitiveRenderer->SetDefaultParameters();
        nn::util::Matrix4x3fType viewMatrix;
        nn::util::Matrix4x4fType projectionMatrix;
        nn::util::Matrix4x3f modelMatrix;
        nn::util::MatrixIdentity(&modelMatrix);
        nn::util::Vector3f translate;
        nn::util::VectorSet(&translate, 0.f, 0.f, 1.f);
        nn::util::MatrixSetAxisW(&modelMatrix, translate);

        // Blend
        pCommandBuffer->SetBlendState(g_pPrimitiveRenderer->GetBlendState(nns::gfx::PrimitiveRenderer::BlendType::BlendType_Normal));

        // ビューとプロジェクションを設定
        int count = g_IsWaitPressHomeButton ? g_SavedFrameCount : frame;
        float radius = 20.f;
#if defined(ANIMATION_ENABLED)
        float x      = radius * sin(count / 500.f );
        float z      = radius * cos(count / 500.f );
#else
        float x = radius * 1.0f;
        float z = radius * 0.0f;
#endif
                nn::util::Vector3fType camPos = { x, 10.f, z };
        nn::util::Vector3fType camTarget = { 0.f, 0.f, 0.f };
        nn::util::Vector3fType camUp = { 0.f, 1.f, 0.f };
        nn::util::MatrixLookAtRightHanded(&viewMatrix, camPos, camTarget, camUp);

        const float fovy = nn::util::FloatPi / 3.0f;
        const float aspect = static_cast< float >(g_GraphicsFramework.GetDisplayWidth()) / static_cast< float >(g_GraphicsFramework.GetDisplayHeight());
        nn::util::MatrixPerspectiveFieldOfViewRightHanded(&projectionMatrix, fovy, aspect, 0.1f, 1000.f);

        g_pPrimitiveRenderer->SetViewMatrix(&viewMatrix);
        g_pPrimitiveRenderer->SetProjectionMatrix(&projectionMatrix);


        // // Depth Enable
        // g_pPrimitiveRenderer->SetDepthStencilState(pCommandBuffer, nns::gfx::PrimitiveRenderer::DepthStencilType::DepthStencilType_DepthWriteTest);
        // Depth Disable
        g_pPrimitiveRenderer->SetDepthStencilState(pCommandBuffer, nns::gfx::PrimitiveRenderer::DepthStencilType::DepthStencilType_DepthNoWriteTest);

        // 軸を描画
        float interval = -10.f;
        nn::util::Vector3fType begin;
        nn::util::Vector3fType end;
        g_pPrimitiveRenderer->SetLineWidth(1.f);
        for (int i = 0; i < 21; i++)
        {
            nn::util::VectorSet(&begin, -10.f, 0.f, interval);
            nn::util::VectorSet(&end, 10.f, 0.f, interval);
            g_pPrimitiveRenderer->SetColor(nn::util::Color4u8::White());
            g_pPrimitiveRenderer->DrawLine(pCommandBuffer, begin, end);
            nn::util::VectorSet(&begin, interval, 0.f, -10.f);
            nn::util::VectorSet(&end, interval, 0.f, 10.f);
            g_pPrimitiveRenderer->DrawLine(pCommandBuffer, begin, end);
            interval += 1.0f;
        }

        // オブジェクトを描画
        nn::util::Vector3fType center = { 0.f, 0.f, 0.f };
        nn::util::Vector3fType size = { 50.f, 50.f, 50.f };
        for (int i = 0; i < g_ObjectCount; ++i)
        {
            pCommandBuffer->InvalidateMemory(nn::gfx::GpuAccess_IndirectBuffer);
            pCommandBuffer->FlushMemory(nn::gfx::GpuAccess_IndirectBuffer);
            nn::util::VectorSet(&translate, 0, 0, 0);
            nn::util::MatrixSetAxisW(&modelMatrix, translate);
            g_pPrimitiveRenderer->SetColor(nn::util::Color4u8(0, 0, 0, 0));
            g_pPrimitiveRenderer->SetModelMatrix(&modelMatrix);
            g_pPrimitiveRenderer->SetLineWidth(14.f);
            g_pPrimitiveRenderer->DrawCone(pCommandBuffer, nns::gfx::PrimitiveRenderer::Surface::Surface_Solid, center, size);
        }

        for (int i = 0; i < g_LiteObjectCount * 2; ++i)
        {
            pCommandBuffer->InvalidateMemory(nn::gfx::GpuAccess_IndirectBuffer);
            pCommandBuffer->FlushMemory(nn::gfx::GpuAccess_IndirectBuffer);
            g_Writer.object.Print(" ");
        }
        // Depth Disable
        g_pPrimitiveRenderer->SetDepthStencilState(pCommandBuffer, nns::gfx::PrimitiveRenderer::DepthStencilType::DepthStencilType_DepthNoWriteTest );

        // タイトルを描画
        g_Writer.object.SetTextColor( nn::util::Color4u8::White() );
        g_Writer.object.SetScale( 3, 3 );
        g_Writer.object.SetCursor( 30, 150 );
        g_Writer.object.Print( g_Title.c_str() );

        // メッセージ描画
        if (g_pMessage)
        {
            g_Writer.object.SetCursor(30, 210);
            g_Writer.object.Print("%s", g_pMessage);
        }

        // ビューをリセット
        g_pPrimitiveRenderer->SetDefaultParameters();

        // 2D Rect 描画
        if ( g_IsEnabledLinearTextures )
        {
            // 線形テクスチャのデスクリプタを取得
            nn::gfx::DescriptorSlot rectDescriptor[g_LinearTextureNum];
            nn::gfx::DescriptorSlot samplerDescriptor;
            g_GraphicsFramework.GetDescriptorPool(nn::gfx::DescriptorPoolType_TextureView)->GetDescriptorSlot(&rectDescriptor[0], g_TextureDescriptorBaseIndex);
            g_GraphicsFramework.GetDescriptorPool(nn::gfx::DescriptorPoolType_TextureView)->GetDescriptorSlot(&rectDescriptor[1], g_TextureDescriptorBaseIndex + 1);
            g_GraphicsFramework.GetDescriptorPool(nn::gfx::DescriptorPoolType_Sampler)->GetDescriptorSlot(&samplerDescriptor, g_SamplerDescriptorBaseIndex);

            g_Writer.object.SetScale( 2, 2 );

            g_Writer.object.SetCursor( 760, 5 );
            g_Writer.object.Print( g_pCaptureBufferName1 );
            g_pPrimitiveRenderer->Draw2DRect(pCommandBuffer,
                760, 53, 480, 270, rectDescriptor[0], samplerDescriptor );

            g_Writer.object.SetCursor( 760, 335 );
            g_Writer.object.Print( g_pCaptureBufferName2 );
            g_pPrimitiveRenderer->Draw2DRect(pCommandBuffer,
                760, 383, 480, 270, rectDescriptor[1], samplerDescriptor );
        }

        // 処理負荷メータを描画
        g_Writer.object.SetScale(1, 1);
        nn::perf::CpuMeter* pFrameMeter = NN_PERF_GET_FRAME_METER();
        g_MeterDrawer.SetVisibleNoResult(false);
        nn::util::Float2 pos = NN_UTIL_FLOAT_2_INITIALIZER(32.f, g_GraphicsFramework.GetDisplayHeight() - g_MeterDrawer.GetHeight(pFrameMeter) - 10);
        g_MeterDrawer.SetDebugFontTextWriter(&g_Writer.object);
        g_MeterDrawer.SetPosition(pos);
        g_MeterDrawer.SetScaleMax(15);
        g_MeterDrawer.SetWidth(g_GraphicsFramework.GetDisplayWidth() - 64.f);
        g_MeterDrawer.Draw(pCommandBuffer, g_pPrimitiveRenderer, pFrameMeter);

        g_Writer.object.SetScale(1.5f, 1.5f);

        // Audio ステータス
        g_Writer.object.SetCursor(30, 280);
        g_Writer.object.SetTextColor(nn::util::Color4u8(255, 255, 255, 255));
        g_Writer.object.Print("Audio : %s", g_IsAudioEnabled ? "Enabled" : "Disabled");

        // フレームカウントを描画
        g_Writer.object.SetCursor(30, 310);
        g_Writer.object.Print("Frame : %d", frame);

        // FPS
        float fps = 1000.f / (static_cast<float>(pFrameMeter->GetLastTotalSpan().GetNanoSeconds()) / 1000000.0f);
        g_Writer.object.SetCursor(30, 340);
        g_Writer.object.Print("FPS : %4.1f", fps);

        // オブジェクト数を描画
        //g_Writer.object.SetCursor(30, 400);
        //g_Writer.object.Print("Object : %d", g_ObjectCount);

        // CPU 負荷 [AcquireTexture]
        {
            nn::TimeSpan t = NN_PERF_GET_ELAPSED_TIME_CPU(NULL, "AcquireTexture", 0);
            g_Writer.object.SetCursor(30, 370);
            g_Writer.object.SetTextColor(nn::util::Color4u8(64, 64, 255, 255));
            g_Writer.object.Print("ACQ : %lldus", t.GetMicroSeconds());
        }

        // CPU 負荷 [RecordCommand]
        {
            nn::TimeSpan t = NN_PERF_GET_ELAPSED_TIME_CPU(NULL, "RecordCommand", 0);
            g_Writer.object.SetCursor(30, 400);
            g_Writer.object.SetTextColor(nn::util::Color4u8(64, 255, 64, 255));
            g_Writer.object.Print("REC : %lldus", t.GetMicroSeconds());
        }

        // CPU 負荷 [PresentTexture]
        {
            nn::TimeSpan t = NN_PERF_GET_ELAPSED_TIME_CPU(NULL, "PresentTexture", 0);
            g_Writer.object.SetCursor(30, 430);
            g_Writer.object.SetTextColor(nn::util::Color4u8(255, 255, 64, 255));
            g_Writer.object.Print("PRE : %lldus", t.GetMicroSeconds());
        }

        // GPU 負荷を描画
        {
            nn::perf::GpuMeter* pGpuMeter = NN_PERF_GET_GPU_METER();
            nn::perf::LoadMeterBase::Section mainSection = pGpuMeter->GetLastResult(0);
            g_Writer.object.SetCursor(30, 510);
            g_Writer.object.SetTextColor(nn::util::Color4u8(255, 64, 64, 255));
            g_Writer.object.Print("GPU : %lldus", mainSection.end.ToTimeSpan().GetMicroSeconds() - mainSection.begin.ToTimeSpan().GetMicroSeconds());
        }


        if (g_IsWaitPressHomeButton)
        {
            g_pPrimitiveRenderer->SetColor(nn::util::Color4u8(0, 0, 0, 128));
            g_pPrimitiveRenderer->Draw2DRect(pCommandBuffer, 440, 210, 400, 300);

            g_Writer.object.SetTextColor(nn::util::Color4u8(255, 255, 255, 255));
            g_Writer.object.SetScale(1.75f, 1.75f);
            g_Writer.object.SetCursor(600, 310);
            g_Writer.object.Print("Press");
            g_Writer.object.SetCursor(550, 360);
            g_Writer.object.Print("Home Button");
        }

        // まとめて文字を書き出す
        g_Writer.object.Draw(pCommandBuffer);

        // カラーバッファからスキャンバッファへコピー
        {
            nn::gfx::TextureCopyRegion destRegion;
            destRegion.SetDefault();
            destRegion.SetWidth(480);
            destRegion.SetHeight(270);
            destRegion.SetOffsetU(260);
            destRegion.SetOffsetV(68);
            nn::gfx::TextureCopyRegion srcRegion;
            srcRegion.SetDefault();
            srcRegion.SetWidth(g_GraphicsFramework.GetDisplayWidth());
            srcRegion.SetHeight(g_GraphicsFramework.GetDisplayHeight());

            if (g_IsPartialForeground)
            {
                int scanBufferIndex = g_GraphicsFramework.GetNextScanBufferIndex();
                nn::gfx::ColorTargetView* pTarget = g_GraphicsFramework.GetScanBufferView(scanBufferIndex);
                pCommandBuffer->ClearColor(pTarget, 0.0f, 0.0f, 0.0f, 0.0f, NULL);
                pCommandBuffer->BlitImage(g_GraphicsFramework.GetScanBuffer(scanBufferIndex), destRegion, g_GraphicsFramework.GetColorBuffer(), srcRegion, 0);
            }
        }

        //NN_PERF_END_MEASURE_GPU(pCommandBuffer);
    }
    pCommandBuffer->FlushMemory( nn::gfx::GpuAccess_QueryBuffer );
    pCommandBuffer->End();
} // NOLINT(impl/function_size)

} // namespace

// グラフィックスの初期処理
void InitializeGraphics( Rgba clearColor, std::string title, FrameworkMode frameworkMode, bool isPartialForeground )
{
    g_ClearColor = clearColor;
    g_Title = title;
    g_CurrentFrameworkMode = frameworkMode;
    g_IsPartialForeground = isPartialForeground;

    const size_t graphicsSystemMemorySize = 8 * 1024 * 1024;
    nns::gfx::GraphicsFramework::InitializeGraphicsSystem(graphicsSystemMemorySize);

    // グラフィックスフレームワークの初期化
    InitializeGraphicsFramework(isPartialForeground);

    // テクスチャ初期化
    InitializeLinearTextures();
    InitializeLinearTextureViews();
    g_TextureDescriptorBaseIndex = g_GraphicsFramework.AllocateDescriptorSlot(nn::gfx::DescriptorPoolType_TextureView, 2);
    g_GraphicsFramework.SetTextureViewToDescriptorPool(g_TextureDescriptorBaseIndex, &g_LinearTextureView[0]);
    g_GraphicsFramework.SetTextureViewToDescriptorPool(g_TextureDescriptorBaseIndex + 1, &g_LinearTextureView[1]);

    //サンプラ初期化
    InitializeSampler();
    g_SamplerDescriptorBaseIndex = g_GraphicsFramework.AllocateDescriptorSlot(nn::gfx::DescriptorPoolType_Sampler, 1);
    g_GraphicsFramework.SetSamplerToDescriptorPool(g_SamplerDescriptorBaseIndex, &g_Sampler);

    // プリミティブレンダラの初期化
    InitializePrimitiveRenderer();

    // デバッグフォントの初期化
    g_GraphicsFramework.InitializeDebugFontTextWriter(&g_Writer, 2560, g_BufferCount);

    // 負荷計測の初期化
    InitializeLoadMeter();

    InitializeQueryCommandBuffer();
}

// グラフィックスの終了処理
void FinalizeGraphics()
{
    FinalizeQueryCommandBuffer();

    // 負荷計測の破棄
    FinalizeLoadMeter();

    // デバッグフォントの破棄
    g_GraphicsFramework.FinalizeDebugFontTextWriter(&g_Writer);

    // プリミティブレンダラの破棄
    FinalizePrimitiveRenderer();

    // サンプラ破棄
    g_GraphicsFramework.FreeDescriptorSlot(nn::gfx::DescriptorPoolType_Sampler, g_SamplerDescriptorBaseIndex);
    FinalizeSampler();

    // テクスチャ破棄
    g_GraphicsFramework.FreeDescriptorSlot(nn::gfx::DescriptorPoolType_TextureView, g_TextureDescriptorBaseIndex);
    FinalizeLinearTextures();
    FinalizeLinearTextureViews();

    // グラフィックスフレームワークの破棄
    FinalizeGraphicsFramework();
}

nn::vi::Layer* GetViLayer()
{
    return g_GraphicsFramework.GetLayer();
}

void EnableLinearTextures(const char* name1, const char* name2)
{
    g_IsEnabledLinearTextures = true;
    g_pCaptureBufferName1 = name1;
    g_pCaptureBufferName2 = name2;
}

void DisableLinearTextures()
{
    g_IsEnabledLinearTextures = false;
}

void SetMessage( const char* pMessage )
{
    g_pMessage = pMessage;
}

char* GetLinearTextureBuffer( int index )
{
    if ( index >= g_LinearTextureNum )
    {
        return NULL;
    }
    return nn::util::BytePtr(g_GraphicsFramework.GetMemoryPool(nns::gfx::GraphicsFramework::MemoryPoolType_ConstantBuffer)->Map(), g_LinearTextureOffset[index] ).Get< char >();
}

size_t GetLinearTextureSize( int index )
{
    if ( index >= g_LinearTextureNum )
    {
        return 0;
    }

    // TORIAEZU
    return g_GraphicsFramework.GetDisplayWidth() * g_GraphicsFramework.GetDisplayHeight() * 4;
}

size_t GetLinearTextureWidth( int index )
{
    if ( index >= g_LinearTextureNum )
    {
        return 0;
    }

    // TORIAEZU
    return g_GraphicsFramework.GetDisplayWidth();
}

size_t GetLinearTextureHeight( int index )
{
    if ( index >= g_LinearTextureNum )
    {
        return 0;
    }

    // TORIAEZU
    return g_GraphicsFramework.GetDisplayHeight();
}

int GetObjectCount()
{
    return g_ObjectCount;
}

void SetObjectCount(int objectCount)
{
    g_ObjectCount = std::min(objectCount, 400);
    g_ObjectCount = std::max(g_ObjectCount, 0);
}

int GetLiteObjectCount()
{
    return g_LiteObjectCount;
}

void SetLiteObjectCount(int objectCount)
{
    g_LiteObjectCount = std::min(objectCount, 400 * 4);
    g_LiteObjectCount = std::max(g_LiteObjectCount, 0);
}

int GetSleepCount()
{
    return g_SleepCount;
}

void SetSleepCount(int sleepCount)
{
    g_SleepCount = std::max(sleepCount, 0);
}

void SetAudioStatus(bool isEnable)
{
    g_IsAudioEnabled = isEnable;
}

void SetWaitPressHomeButtonMode()
{
    if (!g_IsWaitPressHomeButton)
    {
        g_IsWaitPressHomeButton = true;
        g_SavedFrameCount = g_FrameCount;
    }
}

void QueueFlush()
{
    nn::gfx::Queue* queue = g_GraphicsFramework.GetQueue();
    queue->Flush();
}

void WindowAcquireTexture()
{
    NN_PERF_SET_COLOR(nn::util::Color4u8::Blue());
    NN_PERF_AUTO_MEASURE_NAME("AcquireTexture");
    g_GraphicsFramework.AcquireTexture(g_BufferIndex);
}

void QueueWaitSync()
{
    nn::gfx::Queue* queue = g_GraphicsFramework.GetQueue();

    NN_PERF_SET_COLOR(nn::util::Color4u8::Green());
    NN_PERF_AUTO_MEASURE();
    queue->SyncSemaphore(g_GraphicsFramework.GetDisplaySemaphore(g_BufferIndex));
}

void QueueAcquireTexture()
{
    WindowAcquireTexture();
    QueueWaitSync();
    QueueFlush();
}

void QueueFinish()
{
    g_GraphicsFramework.QueueFinish();
}

void WaitDisplaySync()
{
    NN_PERF_SET_COLOR(nn::util::Color4u8::White());
    NN_PERF_AUTO_MEASURE();
    g_GraphicsFramework.WaitDisplaySync(g_BufferIndex, nn::TimeSpan::FromSeconds(2));
}

void UpdateBufferIndex()
{
    g_BufferIndex = (g_BufferIndex + 1) % g_BufferCount;
}

void MakeAppCommands(unsigned int frame)
{
    NN_PERF_SET_COLOR(nn::util::Color4u8::Green());
    NN_PERF_AUTO_MEASURE_NAME("RecordCommand");
    MakeCommand(frame);

    // MakeBeginTimeStampQueryCommand
    g_GraphicsFramework.ResetCommandBuffer(&g_CommandBufferBegin[g_BufferIndex]);
    g_CommandBufferBegin[g_BufferIndex].object.Begin();
    NN_PERF_SET_COLOR_GPU(nn::util::Color4u8::Red());
    NN_PERF_BEGIN_MEASURE_GPU(&g_CommandBufferBegin[g_BufferIndex].object);
    g_CommandBufferBegin[g_BufferIndex].object.FlushMemory(nn::gfx::GpuAccess_QueryBuffer);
    g_CommandBufferBegin[g_BufferIndex].object.End();

    // MakeEndTimeStampQueryCommand
    g_GraphicsFramework.ResetCommandBuffer(&g_CommandBufferEnd[g_BufferIndex]);
    g_CommandBufferEnd[g_BufferIndex].object.Begin();
    NN_PERF_END_MEASURE_GPU(&g_CommandBufferEnd[g_BufferIndex].object);
    g_CommandBufferEnd[g_BufferIndex].object.FlushMemory(nn::gfx::GpuAccess_QueryBuffer);
    g_CommandBufferEnd[g_BufferIndex].object.End();
}

void QueueAppCommands()
{
    nn::gfx::Queue* queue = g_GraphicsFramework.GetQueue();

    NN_PERF_SET_COLOR(nn::util::Color4u8::Green());
    NN_PERF_AUTO_MEASURE_NAME("RecordCommand");
    queue->ExecuteCommand(g_GraphicsFramework.GetRootCommandBuffer(g_BufferIndex), NULL);
}

void QueuePresentTexture()
{
    nn::gfx::Queue* queue = g_GraphicsFramework.GetQueue();

    NN_PERF_SET_COLOR(nn::util::Color4u8::Yellow());
    NN_PERF_AUTO_MEASURE_NAME("PresentTexture");
    queue->Present(g_GraphicsFramework.GetSwapChain(), 1);
}

void QueueBeginTimeStampQueryCommand()
{
    nn::gfx::Queue* queue = g_GraphicsFramework.GetQueue();

    NN_PERF_SET_COLOR(nn::util::Color4u8::Green());
    NN_PERF_BEGIN_MEASURE();
    queue->ExecuteCommand(&g_CommandBufferBegin[g_BufferIndex].object, NULL);
    NN_PERF_END_MEASURE();
}

void QueueEndTimeStampQueryCommand()
{
    nn::gfx::Queue* queue = g_GraphicsFramework.GetQueue();

    NN_PERF_SET_COLOR(nn::util::Color4u8::Green());
    NN_PERF_BEGIN_MEASURE();
    queue->ExecuteCommand(&g_CommandBufferEnd[g_BufferIndex].object, NULL);
    NN_PERF_END_MEASURE();
}

void SleepCpu()
{
    NN_PERF_SET_COLOR(nn::util::Color4u8::Black());
    NN_PERF_AUTO_MEASURE();
    nn::os::SleepThread(nn::TimeSpan::FromMicroSeconds(g_SleepCount * 200));
}

void GraphicsRenderer()
{
    static unsigned int frame = 0;

    // フレーム全体の計測
    NN_PERF_BEGIN_FRAME();
    {
        if ((g_CurrentFrameworkMode == FrameworkMode_AppletOption1) || (g_CurrentFrameworkMode == FrameworkMode_QueueFinish))
        {
            // Update bufferIndex
            UpdateBufferIndex();

            if (frame == 0)
            {
                // AcquireTexture
                QueueAcquireTexture();
            }

            // sleep to increase CPU load
            SleepCpu();

            // Make commands
            MakeAppCommands(frame++);

            // AddBeginTimeStampQueryCommand
            QueueBeginTimeStampQueryCommand();

            // AddCommand
            QueueAppCommands();

            // AddPresentCommand
            QueuePresentTexture();

            // AddEndTimeStampQueryCommand
            QueueEndTimeStampQueryCommand();

            // AcquireTexture
            QueueAcquireTexture();

            if (g_CurrentFrameworkMode == FrameworkMode_QueueFinish)
            {
                // QueueFinish
                QueueFinish();
            }
        }
        else if (g_CurrentFrameworkMode == FrameworkMode_DeferredExecution)
        {
            // Update bufferIndex
            UpdateBufferIndex();

            // WindowAcquireTexture
            WindowAcquireTexture();

            // Add sync command to queue
            QueueWaitSync();

            // sleep to increase CPU load
            SleepCpu();

            // Make commands
            MakeAppCommands(frame++);

            // AddBeginTimeStampQueryCommand
            QueueBeginTimeStampQueryCommand();

            // AddCommand
            QueueAppCommands();

            // PresentTexture
            QueuePresentTexture();

            // AddEndTimeStampQueryCommand
            QueueEndTimeStampQueryCommand();

            QueueFlush();

            // Wait for vsync by waiting for the release fence from previous acquire.
            // That is, when above commands start executing on the GPU.
            WaitDisplaySync();
        }
        else if (g_CurrentFrameworkMode == FrameworkMode_DeferredSubmission)
        {
            // PresentTexture & flush
            if (frame > 0)
            {
                // AddBeginTimeStampQueryCommand
                QueueBeginTimeStampQueryCommand();

                // AddCommand
                QueueAppCommands();

                // PresentTexture
                QueuePresentTexture();

                // AddEndTimeStampQueryCommand
                QueueEndTimeStampQueryCommand();

                QueueFlush();
            }

            // Update bufferIndex
            UpdateBufferIndex();

            // WindowAcquireTexture
            WindowAcquireTexture();

            // Add sync command to queue
            QueueWaitSync();

            // sleep to increase CPU load
            SleepCpu();

            // Make and add commands
            MakeAppCommands(frame++);

            // Wait for vsync by waiting for the release fence from previous acquire.
            // That is, when above commands start executing on the GPU.
            WaitDisplaySync();
        }
        else if (g_CurrentFrameworkMode == FrameworkMode_Immediate)
        {
            if (frame == 0)
            {
                // Update bufferIndex
                UpdateBufferIndex();

                // WindowAcquireTexture
                WindowAcquireTexture();

                // Add sync command to queue
                //QueueWaitSync();

                WaitDisplaySync();
            }

            // sleep to increase CPU load
            SleepCpu();

            // Make and add commands
            MakeAppCommands(frame++);

            // AddBeginTimeStampQueryCommand
            QueueBeginTimeStampQueryCommand();

            // AddCommand
            QueueAppCommands();

            // PresentTexture
            QueuePresentTexture();

            // AddEndTimeStampQueryCommand
            QueueEndTimeStampQueryCommand();

            QueueFlush();

            // Update bufferIndex
            UpdateBufferIndex();

            // WindowAcquireTexture
            WindowAcquireTexture();

            // Add sync command to queue
            //QueueWaitSync();

            // Wait for vsync by waiting for the release fence from previous acquire.
            // That is, when above commands start executing on the GPU.
            WaitDisplaySync();
        }

        g_VsyncEvent.Signal();
    }
    NN_PERF_END_FRAME();
} // NOLINT(impl/function_size)

void WaitVsync()
{
    g_VsyncEvent.Wait();
}

} // Graphics
