﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/os.h>

namespace {

//-----------------------------------------------------------------------------
// スレッド優先度については SIGLO-38575 を参照（2017.2.17 時点）
// priority_e.pdf から更に NINTENDOSDK-2818 にて pri=30～35 の変更あり。
//-----------------------------------------------------------------------------

const int AppletPriorityForAudio       = 1;
const int AppletPriorityForLimit1msec  = 7;
const int AppletPriorityForLimit10msec = 12;
const int AppletPriorityForOver10msec  = 22;
const int AppletPriorityForNonRealtime = 34;

//-----------------------------------------------------------------------------
// FG/BG 遷移時のスレッド同期制御用
//-----------------------------------------------------------------------------
class ThreadControl
{
public:
    // InFocus 状態になるまで待機し true を返す。
    // ただし、終了要求が来たら false を返す。
    bool WaitForInFocusOrExitRequested() NN_NOEXCEPT
    {
        auto index = nn::os::WaitAny( /* index=0 */ m_ExitRequested.GetBase(),
                                      /* index=1 */ m_InFocus.GetBase());
        return index > 0;
    }

    // InFocus 状態になるまで待機する
    bool TryWaitForInFocus() NN_NOEXCEPT
    {
        return m_InFocus.TryWait();
    }

    // InFocus 状態でなくなるまで待機する
    bool TryWaitForNotInFocus() NN_NOEXCEPT
    {
        return m_NotInFocus.TryWait();
    }

    // 終了要求の有無を返す
    bool IsExitRequested() NN_NOEXCEPT
    {
        return m_ExitRequested.TryWait();
    }

    // InFocus ステートか否かを設定する
    void SetInFocusState(bool isInFocusState) NN_NOEXCEPT
    {
        if (isInFocusState)
        {
            m_NotInFocus.Clear();
            m_InFocus.Signal();
        }
        else
        {
            m_InFocus.Clear();
            m_NotInFocus.Signal();
        }
    }

    // 終了を要求する
    void RequestExit() NN_NOEXCEPT
    {
        m_ExitRequested.Signal();
    }

private:
    nn::os::Event m_InFocus{nn::os::EventClearMode_ManualClear};
    nn::os::Event m_NotInFocus{nn::os::EventClearMode_ManualClear};
    nn::os::Event m_ExitRequested{nn::os::EventClearMode_ManualClear};
};

}
