﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Log.h>
#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/nn_Abort.h>
#include <nn/util/util_ScopeExit.h>

#include <nvn/nvn.h>
#include <nvn/nvn_FuncPtrInline.h>

extern "C" PFNNVNGENERICFUNCPTRPROC NVNAPIENTRY nvnBootstrapLoader(const char *name);

namespace nnt{ namespace album{

    class NvnUtility
    {
    public:
        static void InitializeNvnAndDevice(NVNdevice& device) NN_NOEXCEPT
        {
            PFNNVNDEVICEGETPROCADDRESSPROC getProcAddress = (PFNNVNDEVICEGETPROCADDRESSPROC) ((nvnBootstrapLoader)("nvnDeviceGetProcAddress"));
            nvnLoadCProcs(nullptr, getProcAddress);

            int flags =
                NVN_DEVICE_FLAG_DEBUG_ENABLE_BIT |
                NVN_DEVICE_FLAG_DEBUG_ENABLE_LEVEL_0_BIT |
                NVN_DEVICE_FLAG_DEBUG_ENABLE_LEVEL_1_BIT |
                NVN_DEVICE_FLAG_DEBUG_ENABLE_LEVEL_2_BIT |
                NVN_DEVICE_FLAG_DEBUG_ENABLE_LEVEL_3_BIT |
                NVN_DEVICE_FLAG_DEBUG_ENABLE_LEVEL_4_BIT;
            NVNdeviceBuilder builder;
            nvnDeviceBuilderSetDefaults(&builder);
            nvnDeviceBuilderSetFlags(&builder, flags);
            NN_ABORT_UNLESS(nvnDeviceInitialize(&device, &builder));

            // load procs
            nvnLoadCProcs(&device, getProcAddress);

            // check version
            int majorVersion;
            int minorVersion;
            nvnDeviceGetInteger(&device, NVN_DEVICE_INFO_API_MAJOR_VERSION, &majorVersion);
            nvnDeviceGetInteger(&device, NVN_DEVICE_INFO_API_MINOR_VERSION, &minorVersion);
            NN_ABORT_UNLESS_EQUAL(majorVersion, NVN_API_MAJOR_VERSION);
            NN_ABORT_UNLESS_GREATER_EQUAL(minorVersion, NVN_API_MINOR_VERSION);

            nvnDeviceSetWindowOriginMode(&device, NVN_WINDOW_ORIGIN_MODE_UPPER_LEFT);
        }

        static void InitializeCommandMemoryPool(NVNmemoryPool& commandPool, NVNdevice& device, void* pMemory, size_t size) NN_NOEXCEPT
        {
            NVNmemoryPoolBuilder builder;
            nvnMemoryPoolBuilderSetDevice(&builder, &device);
            nvnMemoryPoolBuilderSetDefaults(&builder);
            nvnMemoryPoolBuilderSetFlags(&builder, NVN_MEMORY_POOL_FLAGS_CPU_UNCACHED_BIT | NVN_MEMORY_POOL_FLAGS_GPU_CACHED_BIT);
            nvnMemoryPoolBuilderSetStorage(&builder, pMemory, size);
            NN_ABORT_UNLESS(nvnMemoryPoolInitialize(&commandPool, &builder));
        }

        static void InitializeTextureMemoryPool(NVNmemoryPool& texturePool, NVNdevice& device, void* pMemory, size_t size) NN_NOEXCEPT
        {
            NVNmemoryPoolBuilder builder;
            nvnMemoryPoolBuilderSetDevice(&builder, &device);
            nvnMemoryPoolBuilderSetDefaults(&builder);
            nvnMemoryPoolBuilderSetFlags(&builder, NVN_MEMORY_POOL_FLAGS_CPU_NO_ACCESS_BIT | NVN_MEMORY_POOL_FLAGS_GPU_CACHED_BIT | NVN_MEMORY_POOL_FLAGS_COMPRESSIBLE_BIT);
            nvnMemoryPoolBuilderSetStorage(&builder, pMemory, size);
            NN_ABORT_UNLESS(nvnMemoryPoolInitialize(&texturePool, &builder));
        }

        static void InitializeQueue(NVNqueue& queue, NVNdevice& device) NN_NOEXCEPT
        {
            NVNqueueBuilder builder;
            nvnQueueBuilderSetDefaults(&builder);
            nvnQueueBuilderSetDevice(&builder, &device);
            NN_ABORT_UNLESS(nvnQueueInitialize(&queue, &builder));
        }

        static void InitializeCommandBuffer(NVNcommandBuffer& commandBuffer, NVNdevice& device) NN_NOEXCEPT
        {
            NN_ABORT_UNLESS(nvnCommandBufferInitialize(&commandBuffer, &device));
        }

        static void InitializeFrameBufferTexture(NVNtexture* pTexutures, int count, int width, int height, NVNdevice& device, NVNmemoryPool& texturePool, ptrdiff_t offset, size_t size) NN_NOEXCEPT
        {
            ptrdiff_t pos = offset;
            ptrdiff_t posEnd = offset + size;
            NVNtextureBuilder builder;
            nvnTextureBuilderSetDevice(&builder, &device);
            nvnTextureBuilderSetDefaults(&builder);
            nvnTextureBuilderSetSize2D(&builder, width, height);
            nvnTextureBuilderSetFlags(&builder, NVN_TEXTURE_FLAGS_COMPRESSIBLE_BIT | NVN_TEXTURE_FLAGS_DISPLAY_BIT | NVN_TEXTURE_FLAGS_IMAGE_BIT);
            nvnTextureBuilderSetFormat(&builder, NVN_FORMAT_RGBA8_SRGB);
            nvnTextureBuilderSetTarget(&builder, NVN_TEXTURE_TARGET_2D);

            size_t texSize = nvnTextureBuilderGetStorageSize(&builder);
            size_t texAlign = nvnTextureBuilderGetStorageAlignment(&builder);

            pos = ((pos + texAlign - 1) / texAlign) * texAlign;
            NN_ABORT_UNLESS_GREATER_EQUAL(posEnd - pos, texSize * count);

            for(int i = 0; i < count; i++)
            {
                nvnTextureBuilderSetStorage(&builder, &texturePool, pos);
                NN_ABORT_UNLESS(nvnTextureInitialize(&pTexutures[i], &builder));
                pos += texSize;
            }
                    }

        static void InitializeWindow(NVNwindow& window, NVNtexture*const* pTextureList, int count, NVNnativeWindow nativeWindow, NVNdevice& device) NN_NOEXCEPT
        {
            NVNwindowBuilder builder;
            nvnWindowBuilderSetDevice(&builder, &device);
            nvnWindowBuilderSetDefaults(&builder);
            nvnWindowBuilderSetNativeWindow(&builder, nativeWindow);
            nvnWindowBuilderSetTextures(&builder, count, pTextureList);
            NN_ABORT_UNLESS(nvnWindowInitialize(&window, &builder));

        }

        static void InitializeTextureViewForFrameBuffer(NVNtextureView& texView) NN_NOEXCEPT
        {
            nvnTextureViewSetDefaults(&texView);
            nvnTextureViewSetFormat(&texView, NVN_FORMAT_RGBA8_SRGB);
            nvnTextureViewSetTarget(&texView, NVN_TEXTURE_TARGET_2D);
        }
    };

}}
