﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Abort.h>
#include <nn/nn_Common.h>
#include <nn/account/account_ApiForAdministrators.h>
#include <nn/account/account_Result.h>

#include "testAccount_TestOperator.h"

namespace nnt { namespace account {

template <typename ContextType>
struct Executor
{
    nn::account::NetworkServiceAccountAdministrator* _pAdmin;
    OperatorBase<ContextType>* _pOperator;
    bool _cancelRequired;
    int64_t _waitMicroSeconds;

    static nn::Result Execute(Executor<ContextType>* pObj) NN_NOEXCEPT
    {
        typename OperatorBase<ContextType>::Context ctx;
        NN_RESULT_DO(pObj->_pOperator->Create(&ctx, *pObj->_pAdmin));

        nn::os::SystemEvent e;
        NN_ABORT_UNLESS_RESULT_SUCCESS(ctx.GetSystemEvent(&e));
        if (pObj->_cancelRequired)
        {
            if (!e.TimedWait(nn::TimeSpan::FromMicroSeconds(pObj->_waitMicroSeconds)))
            {
                NN_ABORT_UNLESS_RESULT_SUCCESS(ctx.Cancel());
                e.Wait();
            }
        }
        else
        {
            e.Wait();
        }

        auto r = ctx.GetResult();
        if (!r.IsSuccess())
        {
            NN_RESULT_THROW_UNLESS(nn::account::ResultCancelled::Includes(r), r);
        }
        NN_RESULT_SUCCESS;
    }

    template <typename OperatorType>
    static Executor<ContextType> Get(OperatorType& op, nn::account::NetworkServiceAccountAdministrator& admin) NN_NOEXCEPT
    {
        Executor<ContextType> e = {&admin, &op, false, 0};
        return e;
    }

    template <typename OperatorType>
    static Executor<ContextType> GetForCancel(OperatorType& op, nn::account::NetworkServiceAccountAdministrator& admin, int64_t waitMicroSeconds) NN_NOEXCEPT
    {
        Executor<ContextType> e = {&admin, &op, true, waitMicroSeconds};
        return e;
    }
};

}} // ~namespace nnt::account
