﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/account/account_Api.h>
#include <nn/account/account_ApiForSystemServices.h>

#include "detail/account_CacheUtil.h"
#include "detail/account_UuidUtil.h"
#include "nas/account_NasCredentialHolder.h"
#include <nn/account/baas/account_BaasUserInfoHolder.h>
#include <nn/account/nas/account_NasUserResourceCache.h>

#include "testAccount_Util.h"
#include "testAccount_Mounter.h"

#include <nn/nn_Abort.h>
#include <nn/nn_Log.h>
#include <nn/os/os_SystemEvent.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_FormatString.h>
#include <nn/util/util_ScopeExit.h>

namespace a = nn::account;
namespace t = nnt::account;

#if defined(NN_BUILD_CONFIG_OS_HORIZON)
#define NNT_ACCOUNT_ENABLE_BAAS_USER_INFO
#endif

#if defined(NNT_ACCOUNT_ENABLE_BAAS_USER_INFO)

namespace
{
void CreateNasInfoCache(
    a::nas::NasCredentialCache* pCred,
    const a::NintendoAccountId& na,
    const a::http::CodeVerifier& ver,
    a::nas::NasUserResourceCache& resourceCache,
    const a::detail::AbstractLocalStorage& storage) NN_NOEXCEPT
{
    NN_UNUSED(ver);

    NN_ABORT_UNLESS_RESULT_SUCCESS(a::detail::CacheUtil::StoreCacheFile(&pCred->refreshTokenCacheId, "xyz", sizeof("xyz"), storage));
    NN_ABORT_UNLESS_RESULT_SUCCESS(a::detail::CacheUtil::StoreCacheFile(&pCred->idTokenCacheId, "xyz", sizeof("xyz"), storage));

    a::detail::Uuid userCache;
    NN_ABORT_UNLESS_RESULT_SUCCESS(a::detail::CacheUtil::StoreCacheFile(&userCache, "xyz", sizeof("xyz"), storage));
    resourceCache.Store(na, std::move(userCache));
}

} // ~namespace <anonymous>

TEST(AccountNintendoAccount, UserInfo)
{
    // ストレージ
    t::DefaultTestStorage storage;
    NN_ABORT_UNLESS_RESULT_SUCCESS(storage.Mount());
    storage.Clear();
    NN_ABORT_UNLESS_RESULT_SUCCESS(storage.Setup());
    a::nas::NasUserResourceCache resourceCache;
    NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(resourceCache.Initialize(storage));

    // BaasUserInfoHolder
    a::baas::BaasUserInfoHolder holder;
    NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(holder.Initialize(resourceCache, storage));
    holder.lock();
    NN_UTIL_SCOPE_EXIT
    {
        holder.unlock();
    };
    auto lock = storage.AcquireWriterLock();

    // 認証情報類
    a::Uid user = a::detail::ConvertToUid(storage.GenerateUuidWithContext());
    a::NetworkServiceAccountId nsaId = {0x10};
    a::baas::BaasCredential nsaCredential = {0x20};
    std::memset(nsaCredential .loginPassword, '0', sizeof(nsaCredential.loginPassword));
    a::NintendoAccountId naId = {0x30};
    a::nas::NasCredentialCache nasCredential;
    a::http::CodeVerifier nasVerifier = {"0"};

    //
    t::Buffer buffer(1024);
    a::NintendoAccountId na;
    nn::os::SystemEventType* pEvent;
    NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(holder.GetBaasUserAvailabilityChangeNotifier(&pEvent));
    NN_UTIL_SCOPE_EXIT
    {
        holder.ReleaseSystemEvent(pEvent);
    };

    // 初期状態
    NN_LOG("[Initiali state]\n");
    NNT_ACCOUNT_ASSERT_RESULT_INCLUDED(a::ResultNetworkServiceAccountRegistrationRequired, holder.CheckAvailability(user));
    ASSERT_FALSE(nn::os::TryWaitSystemEvent(pEvent));
    // - NA 連携不可
    NNT_ACCOUNT_ASSERT_RESULT_INCLUDED(
        a::ResultNetworkServiceAccountRegistrationRequired,
        holder.InitializeNintendoAccountLinkageState(user, naId, nasCredential, nasVerifier, buffer.GetAddress(), buffer.GetSize()));
    ASSERT_FALSE(a::nas::NasCredentialHolder::HasCredential(naId, storage));
    ASSERT_FALSE(resourceCache.IsAvailable(naId));

    // 認証情報の作成
    CreateNasInfoCache(&nasCredential, naId, nasVerifier, resourceCache, storage);

    // NSA 登録
    NN_LOG("[holder.Register()]\n");
    NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(holder.Register(user, nsaId, nsaCredential));
    NNT_ACCOUNT_ASSERT_RESULT_INCLUDED(a::ResultNintendoAccountLinkageRequired, holder.CheckAvailability(user));
    NNT_ACCOUNT_ASSERT_RESULT_INCLUDED(a::ResultNintendoAccountLinkageRequired, holder.GetLinkedNintendoAccountId(&na, user));
    ASSERT_FALSE(nn::os::TryWaitSystemEvent(pEvent));
    // - 登録不可
    NNT_ACCOUNT_ASSERT_RESULT_INCLUDED(
        a::ResultNetworkServiceAccountAlreadyRegistered,
        holder.Register(user, nsaId, nsaCredential));

    // NA リンク開始
    NN_LOG("[holder.InitializeNintendoAccountLinkageState()]\n");
    ASSERT_FALSE(a::nas::NasCredentialHolder::HasCredential(naId, storage));
    ASSERT_TRUE(resourceCache.IsAvailable(naId));
    NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(
        holder.InitializeNintendoAccountLinkageState(user, naId, nasCredential, nasVerifier, buffer.GetAddress(), buffer.GetSize()));
    NNT_ACCOUNT_ASSERT_RESULT_INCLUDED(a::ResultNintendoAccountLinkageBroken, holder.CheckAvailability(user));
    NNT_ACCOUNT_ASSERT_RESULT_INCLUDED(a::ResultNintendoAccountInvalidState, holder.CheckAvailability(user));
    NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(holder.GetLinkedNintendoAccountId(&na, user));
    ASSERT_EQ(naId, na);
    ASSERT_TRUE(a::nas::NasCredentialHolder::HasCredential(naId, storage));
    ASSERT_TRUE(resourceCache.IsAvailable(naId));
    ASSERT_FALSE(nn::os::TryWaitSystemEvent(pEvent));
    // - 登録不可
    NNT_ACCOUNT_ASSERT_RESULT_INCLUDED(
        a::ResultNintendoAccountAlreadyLinked,
        holder.InitializeNintendoAccountLinkageState(user, naId, nasCredential, nasVerifier, buffer.GetAddress(), buffer.GetSize()));

    // NA リンク確定
    NN_LOG("[holder.CompleteNintendoAccountLinkageState()]\n");
    NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(holder.CompleteNintendoAccountLinkageState(user));
    NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(holder.CheckAvailability(user));
    NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(holder.GetLinkedNintendoAccountId(&na, user));
    ASSERT_EQ(naId, na);
    ASSERT_TRUE(a::nas::NasCredentialHolder::HasCredential(naId, storage));
    ASSERT_TRUE(resourceCache.IsAvailable(naId));
    ASSERT_TRUE(nn::os::TryWaitSystemEvent(pEvent));
    nn::os::ClearSystemEvent(pEvent);
    // - 登録不可
    NNT_ACCOUNT_ASSERT_RESULT_INCLUDED(
        a::ResultNintendoAccountAlreadyLinked,
        holder.InitializeNintendoAccountLinkageState(user, naId, nasCredential, nasVerifier, buffer.GetAddress(), buffer.GetSize()));

    // NA ユーザー状態異常 (user[0])
    NN_LOG("[holder.DegradeNintendoAccountUserState()]\n");
    NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(holder.DegradeNintendoAccountUserState(user, a::baas::NintendoAccountUserState_Banned));
    NNT_ACCOUNT_ASSERT_RESULT_INCLUDED(a::ResultNintendoAccountStateBanned, holder.CheckAvailability(user));
    NNT_ACCOUNT_ASSERT_RESULT_INCLUDED(a::ResultNintendoAccountStateInteractionRequired, holder.CheckAvailability(user));
    NNT_ACCOUNT_ASSERT_RESULT_INCLUDED(a::ResultNintendoAccountInvalidState, holder.CheckAvailability(user));
    ASSERT_TRUE(a::nas::NasCredentialHolder::HasCredential(naId, storage));
    ASSERT_TRUE(resourceCache.IsAvailable(naId));
    ASSERT_TRUE(nn::os::TryWaitSystemEvent(pEvent));
    nn::os::ClearSystemEvent(pEvent);
    // - 登録不可
    NNT_ACCOUNT_ASSERT_RESULT_INCLUDED(
        a::ResultNintendoAccountAlreadyLinked,
        holder.InitializeNintendoAccountLinkageState(user, naId, nasCredential, nasVerifier, buffer.GetAddress(), buffer.GetSize()));

    // NA ユーザー状態復旧 (認証情報更新なし)
    NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(holder.RecoveryNintendoAccountUserState(user));
    ASSERT_TRUE(nn::os::TryWaitSystemEvent(pEvent));
    nn::os::ClearSystemEvent(pEvent);

    // 再度異常にする
    NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(holder.DegradeNintendoAccountUserState(user, a::baas::NintendoAccountUserState_Deleted));
    NNT_ACCOUNT_ASSERT_RESULT_INCLUDED(a::ResultNintendoAccountStateDeleted, holder.CheckAvailability(user));
    ASSERT_TRUE(nn::os::TryWaitSystemEvent(pEvent));
    nn::os::ClearSystemEvent(pEvent);

    // その他の状態異常
    NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(holder.DegradeNintendoAccountUserState(user, a::baas::NintendoAccountUserState_Suspended));
    NNT_ACCOUNT_ASSERT_RESULT_INCLUDED(a::ResultNintendoAccountStateSuspended, holder.CheckAvailability(user));
    NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(holder.DegradeNintendoAccountUserState(user, a::baas::NintendoAccountUserState_Withdrawn));
    NNT_ACCOUNT_ASSERT_RESULT_INCLUDED(a::ResultNintendoAccountStateWithdrawn, holder.CheckAvailability(user));
    NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(holder.DegradeNintendoAccountUserState(user, a::baas::NintendoAccountUserState_ReauthorizationRequired));
    NNT_ACCOUNT_ASSERT_RESULT_INCLUDED(a::ResultNintendoAccountStateReauthorizationRequired, holder.CheckAvailability(user));
    NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(holder.DegradeNintendoAccountUserState(user, a::baas::NintendoAccountUserState_OtherButInteractionRequired));
    NNT_ACCOUNT_ASSERT_RESULT_INCLUDED(a::ResultNintendoAccountStateOtherButInteractionRequired, holder.CheckAvailability(user));
    NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(holder.DegradeNintendoAccountUserState(user, a::baas::NintendoAccountUserState_TermsAgreementRequired));
    NNT_ACCOUNT_ASSERT_RESULT_INCLUDED(a::ResultNintendoAccountStateTermsAgreementRequired, holder.CheckAvailability(user));
    ASSERT_FALSE(nn::os::TryWaitSystemEvent(pEvent));

    // NA ユーザー状態復旧
    NN_LOG("[holder.RecoveryNintendoAccountUserState()]\n");
    NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(holder.RecoveryNintendoAccountUserState(user, naId, nasCredential, nasVerifier, buffer.GetAddress(), buffer.GetSize()));
    NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(holder.CheckAvailability(user));
    NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(holder.GetLinkedNintendoAccountId(&na, user));
    ASSERT_EQ(naId, na);
    ASSERT_TRUE(a::nas::NasCredentialHolder::HasCredential(naId, storage));
    ASSERT_TRUE(resourceCache.IsAvailable(naId));
    ASSERT_TRUE(nn::os::TryWaitSystemEvent(pEvent));
    nn::os::ClearSystemEvent(pEvent);

    // NA リンク破壊
    NN_LOG("[holder.DegradeNintendoAccountLinkageState()]\n");
    NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(holder.DegradeNintendoAccountLinkageState(user));
    NNT_ACCOUNT_ASSERT_RESULT_INCLUDED(a::ResultNintendoAccountLinkageBroken, holder.CheckAvailability(user));
    NNT_ACCOUNT_ASSERT_RESULT_INCLUDED(a::ResultNotSupported, holder.RecoveryNintendoAccountUserState(user));
    NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(holder.GetLinkedNintendoAccountId(&na, user));
    ASSERT_EQ(naId, na);
    ASSERT_TRUE(a::nas::NasCredentialHolder::HasCredential(naId, storage));
    ASSERT_TRUE(resourceCache.IsAvailable(naId));
    ASSERT_TRUE(nn::os::TryWaitSystemEvent(pEvent));
    nn::os::ClearSystemEvent(pEvent);

    // NA リンク破棄
    NN_LOG("[holder.DismissNintendoAccountLinkageState()]\n");
    NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(holder.DismissNintendoAccountLinkageState(user));
    NNT_ACCOUNT_ASSERT_RESULT_INCLUDED(a::ResultNintendoAccountLinkageRequired, holder.CheckAvailability(user));
    NNT_ACCOUNT_ASSERT_RESULT_INCLUDED(a::ResultNintendoAccountLinkageRequired, holder.RecoveryNintendoAccountUserState(user));
    ASSERT_FALSE(a::nas::NasCredentialHolder::HasCredential(naId, storage));
    ASSERT_FALSE(resourceCache.IsAvailable(naId));
    ASSERT_FALSE(nn::os::TryWaitSystemEvent(pEvent));

    // NA 再リンク
    auto fRelink = [&]() -> bool
    {
        CreateNasInfoCache(&nasCredential, naId, nasVerifier, resourceCache, storage);
        NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(
            holder.InitializeNintendoAccountLinkageState(user, naId, nasCredential, nasVerifier, buffer.GetAddress(), buffer.GetSize()));
        NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(holder.CompleteNintendoAccountLinkageState(user));
        NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(holder.CheckAvailability(user));
        NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(holder.GetLinkedNintendoAccountId(&na, user));
        NNT_ACCOUNT_RETURN_FALSE_UNLESS(naId == na);
        NNT_ACCOUNT_RETURN_FALSE_UNLESS(a::nas::NasCredentialHolder::HasCredential(naId, storage));
        NNT_ACCOUNT_RETURN_FALSE_UNLESS(resourceCache.IsAvailable(naId));
        NNT_ACCOUNT_RETURN_FALSE_UNLESS(nn::os::TryWaitSystemEvent(pEvent));
        nn::os::ClearSystemEvent(pEvent);
        return true;
    };
    NN_LOG("[Re-link]\n");
    ASSERT_TRUE(fRelink());

    // NA リンク破棄(その2)
    NN_LOG("[holder.DismissNintendoAccountLinkageState()]\n");
    NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(holder.DismissNintendoAccountLinkageState(user));
    NNT_ACCOUNT_ASSERT_RESULT_INCLUDED(a::ResultNintendoAccountLinkageRequired, holder.CheckAvailability(user));
    NNT_ACCOUNT_ASSERT_RESULT_INCLUDED(a::ResultNintendoAccountLinkageRequired, holder.RecoveryNintendoAccountUserState(user));
    ASSERT_FALSE(a::nas::NasCredentialHolder::HasCredential(naId, storage));
    ASSERT_FALSE(resourceCache.IsAvailable(naId));
    ASSERT_TRUE(nn::os::TryWaitSystemEvent(pEvent));
    nn::os::ClearSystemEvent(pEvent);

    // NA 再リンク (2)
    NN_LOG("[Re-link]\n");
    ASSERT_TRUE(fRelink());

    // NSA 破壊
    NN_LOG("[holder.DegradeBaasUserState()]\n");
    NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(holder.DegradeBaasUserState(user, a::baas::BaasUserState_Inconsequent));
    NNT_ACCOUNT_ASSERT_RESULT_INCLUDED(a::ResultNetworkServiceAccountCredentialBroken, holder.CheckAvailability(user));
    NNT_ACCOUNT_ASSERT_RESULT_INCLUDED(a::ResultNetworkServiceAccountCredentialBroken, holder.RecoveryNintendoAccountUserState(user));
    ASSERT_FALSE(a::nas::NasCredentialHolder::HasCredential(naId, storage));
    ASSERT_FALSE(resourceCache.IsAvailable(naId));
    ASSERT_TRUE(nn::os::TryWaitSystemEvent(pEvent));
    nn::os::ClearSystemEvent(pEvent);

    // NSA 削除
    NN_LOG("[holder.Unregister()]\n");
    NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(holder.Unregister(user));
    NNT_ACCOUNT_ASSERT_RESULT_INCLUDED(a::ResultNetworkServiceAccountRegistrationRequired, holder.CheckAvailability(user));
    NNT_ACCOUNT_ASSERT_RESULT_INCLUDED(a::ResultNetworkServiceAccountRegistrationRequired, holder.RecoveryNintendoAccountUserState(user));
    ASSERT_FALSE(nn::os::TryWaitSystemEvent(pEvent));

    // NSA 登録, NA 再リンク
    NN_LOG("[Re-link]\n");
    NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(holder.Register(user, nsaId, nsaCredential));
    ASSERT_TRUE(fRelink());

    // NSA 削除
    NN_LOG("[holder.Unregister()]\n");
    NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(holder.Unregister(user));
    NNT_ACCOUNT_ASSERT_RESULT_INCLUDED(a::ResultNetworkServiceAccountRegistrationRequired, holder.CheckAvailability(user));
    NNT_ACCOUNT_ASSERT_RESULT_INCLUDED(a::ResultNetworkServiceAccountRegistrationRequired, holder.RecoveryNintendoAccountUserState(user));
    ASSERT_FALSE(a::nas::NasCredentialHolder::HasCredential(naId, storage));
    ASSERT_FALSE(resourceCache.IsAvailable(naId));
    ASSERT_TRUE(nn::os::TryWaitSystemEvent(pEvent));
    nn::os::ClearSystemEvent(pEvent);

    // NA 連携できない
    NNT_ACCOUNT_ASSERT_RESULT_INCLUDED(
        a::ResultNetworkServiceAccountRegistrationRequired,
        holder.InitializeNintendoAccountLinkageState(user, naId, nasCredential, nasVerifier, buffer.GetAddress(), buffer.GetSize()));

} // NOLINT(readability/fn_size)

TEST(AccountNintendoAccount, UserInfoMulti)
{
    // ストレージ
    t::DefaultTestStorage storage;
    NN_ABORT_UNLESS_RESULT_SUCCESS(storage.Mount());
    storage.Clear();
    NN_ABORT_UNLESS_RESULT_SUCCESS(storage.Setup());
    a::nas::NasUserResourceCache resourceCache;
    NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(resourceCache.Initialize(storage));

    // BaasUserInfoHolder
    a::baas::BaasUserInfoHolder holder;
    NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(holder.Initialize(resourceCache, storage));
    holder.lock();
    NN_UTIL_SCOPE_EXIT
    {
        holder.unlock();
    };
    auto lock = storage.AcquireWriterLock();

    // 認証情報類
    const int NumUsers = a::UserCountMax * 2;
    a::Uid users[NumUsers];
    a::NetworkServiceAccountId nsaIds[NumUsers];
    a::baas::BaasCredential nsaCredentials[NumUsers];
    a::NintendoAccountId naIds[NumUsers];
    a::nas::NasCredentialCache nasCredentials[NumUsers];
    a::http::CodeVerifier nasVerifiers[NumUsers];
    for (auto i = 0; i < NumUsers; ++i)
    {
        auto uid = a::detail::ConvertToUid(storage.GenerateUuidWithContext());
        std::remove_reference<decltype(nsaIds[i])>::type nsaId = {static_cast<uint64_t>(0x10 + i)};
        std::remove_reference<decltype(nsaCredentials[i])>::type nsaCredential;
        nsaCredential.loginId = static_cast<uint64_t>(0x20 + i);
        std::memset(nsaCredential.loginPassword, 'a' + i, sizeof(nsaCredential.loginPassword));
        std::remove_reference<decltype(naIds[i])>::type naId = {static_cast<uint64_t>(0x30 + i)};
        std::remove_reference<decltype(nasCredentials[i])>::type nasCredential;
        std::remove_reference<decltype(nasVerifiers[i])>::type nasVerifier;
        nn::util::SNPrintf(nasVerifier.data, sizeof(nasVerifier.data), "%c", 'a' + i);

        users[i] = uid;
        nsaIds[i] = nsaId;
        nsaCredentials[i] = nsaCredential;
        naIds[i] = naId;
        nasCredentials[i] = nasCredential;
        nasVerifiers[i] = nasVerifier;
        CreateNasInfoCache(&nasCredentials[i], naIds[i], nasVerifiers[i], resourceCache, storage);
    }

    //
    t::Buffer buffer(1024);
    a::NintendoAccountId na;
    nn::os::SystemEventType* pEvent;
    NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(holder.GetBaasUserAvailabilityChangeNotifier(&pEvent));
    NN_UTIL_SCOPE_EXIT
    {
        holder.ReleaseSystemEvent(pEvent);
    };

    // 初期状態
    NN_LOG("[Initiali state]\n");
    for (auto i = 0; i < NumUsers; ++i)
    {
        NNT_ACCOUNT_ASSERT_RESULT_INCLUDED(a::ResultNetworkServiceAccountRegistrationRequired, holder.CheckAvailability(users[i]));
    }
    ASSERT_FALSE(nn::os::TryWaitSystemEvent(pEvent));

    // NSA 登録
    NN_LOG("[holder.Register()]\n");
    for (auto i = 0; i < NumUsers; ++i)
    {
        NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(holder.Register(users[i], nsaIds[i], nsaCredentials[i]));
        for (auto j = 0; j < i; ++j)
        {
            NNT_ACCOUNT_ASSERT_RESULT_INCLUDED(a::ResultNintendoAccountLinkageRequired, holder.CheckAvailability(users[j]));
            NNT_ACCOUNT_ASSERT_RESULT_INCLUDED(a::ResultNintendoAccountLinkageRequired, holder.GetLinkedNintendoAccountId(&na, users[j]));
        }
        for (auto j = i + 1; j < NumUsers; ++j)
        {
            NNT_ACCOUNT_ASSERT_RESULT_INCLUDED(a::ResultNetworkServiceAccountRegistrationRequired, holder.CheckAvailability(users[j]));
            NNT_ACCOUNT_ASSERT_RESULT_INCLUDED(a::ResultNetworkServiceAccountRegistrationRequired, holder.GetLinkedNintendoAccountId(&na, users[j]));
        }
    }
    ASSERT_FALSE(nn::os::TryWaitSystemEvent(pEvent));

    // NA リンク開始
    NN_LOG("[holder.InitializeNintendoAccountLinkageState()]\n");
    for (auto i = 0; i < NumUsers; ++i)
    {
        NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(
            holder.InitializeNintendoAccountLinkageState(users[i], naIds[i], nasCredentials[i], nasVerifiers[i], buffer.GetAddress(), buffer.GetSize()));

        for (auto j = 0; j < i; ++j)
        {
            NNT_ACCOUNT_ASSERT_RESULT_INCLUDED(a::ResultNintendoAccountLinkageBroken, holder.CheckAvailability(users[j]));
            NNT_ACCOUNT_ASSERT_RESULT_INCLUDED(
                a::ResultNintendoAccountAlreadyLinked,
                holder.InitializeNintendoAccountLinkageState(users[j], naIds[i], nasCredentials[i], nasVerifiers[i], buffer.GetAddress(), buffer.GetSize()));
            NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(holder.GetLinkedNintendoAccountId(&na, users[j]));
            EXPECT_EQ(naIds[j], na);
        }
        for (auto j = i + 1; j < NumUsers; ++j)
        {
            NNT_ACCOUNT_ASSERT_RESULT_INCLUDED(a::ResultNintendoAccountLinkageRequired, holder.CheckAvailability(users[j]));
            NNT_ACCOUNT_ASSERT_RESULT_INCLUDED(
                a::ResultDuplicativeNintendoAccount,
                holder.InitializeNintendoAccountLinkageState(users[j], naIds[i], nasCredentials[i], nasVerifiers[i], buffer.GetAddress(), buffer.GetSize()));
        }
    }
    ASSERT_FALSE(nn::os::TryWaitSystemEvent(pEvent));

    // NA リンク確定
    NN_LOG("[holder.CompleteNintendoAccountLinkageState()]\n");
    for (auto i = 0; i < NumUsers; ++i)
    {
        NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(holder.CompleteNintendoAccountLinkageState(users[i]));
        ASSERT_TRUE(nn::os::TryWaitSystemEvent(pEvent));
        nn::os::ClearSystemEvent(pEvent);

        for (auto j = 0; j < i; ++j)
        {
            NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(holder.CheckAvailability(users[j]));
            NNT_ACCOUNT_ASSERT_RESULT_INCLUDED(
                a::ResultNintendoAccountAlreadyLinked,
                holder.InitializeNintendoAccountLinkageState(users[j], naIds[i], nasCredentials[i], nasVerifiers[i], buffer.GetAddress(), buffer.GetSize()));
            NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(holder.GetLinkedNintendoAccountId(&na, users[j]));
            EXPECT_EQ(naIds[j], na);
        }
        for (auto j = i + 1; j < NumUsers; ++j)
        {
            NNT_ACCOUNT_ASSERT_RESULT_INCLUDED(a::ResultNintendoAccountLinkageBroken, holder.CheckAvailability(users[j]));
            NNT_ACCOUNT_ASSERT_RESULT_INCLUDED(
                a::ResultNintendoAccountAlreadyLinked,
                holder.InitializeNintendoAccountLinkageState(users[j], naIds[i], nasCredentials[i], nasVerifiers[i], buffer.GetAddress(), buffer.GetSize()));
            NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(holder.GetLinkedNintendoAccountId(&na, users[j]));
            EXPECT_EQ(naIds[j], na);
        }
    }

    // NA ユーザー状態異常
    NN_LOG("[holder.DegradeNintendoAccountUserState()]\n");
    for (auto i = 0; i < NumUsers; ++i)
    {
        NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(
            holder.DegradeNintendoAccountUserState(users[i], a::baas::NintendoAccountUserState_Banned));
        ASSERT_TRUE(nn::os::TryWaitSystemEvent(pEvent));
        nn::os::ClearSystemEvent(pEvent);

        for (auto j = 0; j < i; ++j)
        {
            NNT_ACCOUNT_ASSERT_RESULT_INCLUDED(a::ResultNintendoAccountStateBanned, holder.CheckAvailability(users[j]));
            NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(holder.GetLinkedNintendoAccountId(&na, users[j]));
            EXPECT_EQ(naIds[j], na);
        }
        for (auto j = i + 1; j < NumUsers; ++j)
        {
            NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(holder.CheckAvailability(users[j]));
            NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(holder.GetLinkedNintendoAccountId(&na, users[j]));
            EXPECT_EQ(naIds[j], na);
        }
    }

    // NA リンク破壊
    NN_LOG("[holder.DegradeNintendoAccountLinkageState()]\n");
    for (auto i = 0; i < NumUsers; ++i)
    {
        NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(holder.DegradeNintendoAccountLinkageState(users[i]));

        for (auto j = 0; j < i; ++j)
        {
            NNT_ACCOUNT_ASSERT_RESULT_INCLUDED(a::ResultNintendoAccountLinkageBroken, holder.CheckAvailability(users[j]));
            NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(holder.GetLinkedNintendoAccountId(&na, users[j]));
            EXPECT_EQ(naIds[j], na);
        }
        for (auto j = i + 1; j < NumUsers; ++j)
        {
            NNT_ACCOUNT_ASSERT_RESULT_INCLUDED(a::ResultNintendoAccountStateBanned, holder.CheckAvailability(users[j]));
            NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(holder.GetLinkedNintendoAccountId(&na, users[j]));
            EXPECT_EQ(naIds[j], na);
        }
    }
    ASSERT_FALSE(nn::os::TryWaitSystemEvent(pEvent));

    // NA 連携解除
    NN_LOG("[holder.DismissNintendoAccountLinkageState()]\n");
    for (auto i = 0; i < NumUsers; ++i)
    {
        NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(holder.DismissNintendoAccountLinkageState(users[i]));

        for (auto j = 0; j < i; ++j)
        {
            NNT_ACCOUNT_ASSERT_RESULT_INCLUDED(a::ResultNintendoAccountLinkageRequired, holder.CheckAvailability(users[j]));
        }
        for (auto j = i + 1; j < NumUsers; ++j)
        {
            NNT_ACCOUNT_ASSERT_RESULT_INCLUDED(a::ResultNintendoAccountLinkageBroken, holder.CheckAvailability(users[j]));
            NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(holder.GetLinkedNintendoAccountId(&na, users[j]));
            EXPECT_EQ(naIds[j], na);

            NNT_ACCOUNT_ASSERT_RESULT_INCLUDED(
                a::ResultDuplicativeNintendoAccount,
                holder.InitializeNintendoAccountLinkageState(users[i], naIds[j], nasCredentials[i], nasVerifiers[i], buffer.GetAddress(), buffer.GetSize()));
        }
    }
    ASSERT_FALSE(nn::os::TryWaitSystemEvent(pEvent));

    // NSA 破壊
    NN_LOG("[holder.DegradeBaasUserState()]\n");
    for (auto i = 0; i < NumUsers; ++i)
    {
        NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(holder.DegradeBaasUserState(users[i], a::baas::BaasUserState_Inconsequent));

        for (auto j = 0; j < i; ++j)
        {
            NNT_ACCOUNT_ASSERT_RESULT_INCLUDED(a::ResultNetworkServiceAccountCredentialBroken, holder.CheckAvailability(users[j]));
        }
        for (auto j = i + 1; j < NumUsers; ++j)
        {
            NNT_ACCOUNT_ASSERT_RESULT_INCLUDED(a::ResultNintendoAccountLinkageRequired, holder.CheckAvailability(users[j]));
        }
    }
    ASSERT_FALSE(nn::os::TryWaitSystemEvent(pEvent));


    // NSA 削除 (user[1])
    NN_LOG("[holder.Unregister()]\n");
    for (auto i = 0; i < NumUsers; ++i)
    {
        NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(holder.Unregister(users[i]));

        for (auto j = 0; j < i; ++j)
        {
            NNT_ACCOUNT_ASSERT_RESULT_INCLUDED(a::ResultNetworkServiceAccountRegistrationRequired, holder.CheckAvailability(users[j]));
        }
        for (auto j = i + 1; j < NumUsers; ++j)
        {
            NNT_ACCOUNT_ASSERT_RESULT_INCLUDED(a::ResultNetworkServiceAccountCredentialBroken, holder.CheckAvailability(users[j]));
        }
    }
    ASSERT_FALSE(nn::os::TryWaitSystemEvent(pEvent));
} // NOLINT(readability/fn_size)

#endif // NNT_ACCOUNT_ENABLE_BAAS_USER_INFO
