﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/account/baas/account_BaasOperator.h>
#include <nn/account/baas/account_ResultForBaas.h>
#include <nn/account/account_RuntimeResource.h>

#include <nn/account/profile/account_ProfileStorage.h>

#include "detail/account_UuidUtil.h"

#include "testAccount_File.h"
#include "testAccount_Mounter.h"
#include "testAccount_Thread.h"
#include "testAccount_Util.h"

#include <atomic>
#include <algorithm>
#include <curl/curl.h>

#include <nn/os/os_Tick.h>
#include <nn/os/os_TimerEvent.h>
#include <nn/util/util_ScopeExit.h>

#include <nn/os/os_ThreadApi.h>

namespace a = nn::account;
namespace t = nnt::account;

#define NNT_ACCOUNT_ENABLE_BAAS_OPERATOR

#if defined(NNT_ACCOUNT_ENABLE_BAAS_OPERATOR)

namespace
{
void ThreadFunction(
    nn::os::Event& terminator,
    a::Executor& executor, a::ExecutionResource& resource) NN_NOEXCEPT
{
    nn::os::MultiWaitType multiWait;
    nn::os::InitializeMultiWait(&multiWait);

    nn::os::MultiWaitHolderType terminatorH;
    nn::os::InitializeMultiWaitHolder(&terminatorH, terminator.GetBase());
    nn::os::LinkMultiWaitHolder(&multiWait, &terminatorH);

    nn::os::MultiWaitHolderType executorH;
    executor.InitializeMultiWaitHolder(&executorH);
    nn::os::LinkMultiWaitHolder(&multiWait, &executorH);

    nn::os::TimerEvent curlTimer(nn::os::EventClearMode_ManualClear);
    nn::os::MultiWaitHolderType curlTimerH;
    nn::os::InitializeMultiWaitHolder(&curlTimerH, curlTimer.GetBase());
    nn::os::LinkMultiWaitHolder(&multiWait, &curlTimerH);

    // CURL ハンドル周りの初期化
    bool isCurlHandleUsed = false;
    curlTimer.StartPeriodic(nn::TimeSpan::FromSeconds(10), nn::TimeSpan::FromSeconds(10));

    while (NN_STATIC_CONDITION(true))
    {
        auto pH = nn::os::WaitAny(&multiWait);
        if (pH == &terminatorH)
        {
            // Terminator は Clear しない。
            NN_LOG("[!!!] Terminate requested\n");
            break;
        }

        if (pH == &curlTimerH)
        {
            curlTimer.Clear();
            if (isCurlHandleUsed && false)
            {
                // CURL ハンドルの再取得
                curl_easy_cleanup(resource.curlHandle);
                resource.curlHandle = curl_easy_init();
                NN_ABORT_UNLESS(resource.curlHandle);
                isCurlHandleUsed = false;
                NN_LOG("[!!!] Curl refreshed\n");
            }
        }
        else if (pH == &executorH)
        {
            executor.ClearSignal();
            executor.Execute(resource);
            isCurlHandleUsed = true;
        }
    }

    // CURL ハンドル周りの解放
    curlTimer.Stop();

    nn::os::UnlinkMultiWaitHolder(&curlTimerH);
    nn::os::FinalizeMultiWaitHolder(&curlTimerH);

    nn::os::UnlinkMultiWaitHolder(&executorH);
    nn::os::FinalizeMultiWaitHolder(&executorH);

    nn::os::UnlinkMultiWaitHolder(&terminatorH);
    nn::os::FinalizeMultiWaitHolder(&terminatorH);

    nn::os::FinalizeMultiWait(&multiWait);
}

#define NNT_ACCOUNT_CREATE_READ_EVENT(name, task, clearMode) \
    nn::os::SystemEvent name((task).GetReadableHandle(), false, nn::os::InvalidNativeHandle, false, clearMode)

bool LoadProfile(
    a::profile::ProfileBase* pOutBase, a::profile::UserData* pOutUserData, t::Buffer* pOutImage,
    const a::Uid& user, const a::profile::ProfileStorage& profileStorage) NN_NOEXCEPT
{
    NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(profileStorage.GetProfile(pOutBase, pOutUserData, user));
    size_t size;
    auto r = profileStorage.GetImageSize(&size, user);
    if (r.IsSuccess() && size > 0)
    {
        *pOutImage = t::Buffer(size);
        NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(profileStorage.LoadImage(&size, pOutImage->GetAddress(), size, user));
    }
    return true;
}
template <typename T>
bool CompareTo(
    const a::Uid& user, const a::profile::ProfileStorage& profileStorage,
    const a::profile::ProfileBase& base, const a::profile::UserData& userData, const T& image) NN_NOEXCEPT
{
    a::profile::ProfileBase tmp;
    a::profile::UserData tmpUserData;
    t::Buffer tmpimg;
    NNT_ACCOUNT_RETURN_FALSE_UNLESS(LoadProfile(&tmp, &tmpUserData, &tmpimg, user, profileStorage));
    EXPECT_TRUE(base.author == tmp.author);
    EXPECT_EQ(base.timeStamp, tmp.timeStamp);
    EXPECT_EQ(0, std::strncmp(base.nickname, tmp.nickname, sizeof(base.nickname)));
    EXPECT_EQ(0, std::memcmp(userData.data, tmpUserData.data, sizeof(userData.data)));
    EXPECT_EQ(image.GetSize(), tmpimg.GetSize());
    NN_LOG("Image original: %zu, downloaded: %zu\n", image.GetSize(), tmpimg.GetSize());
    if (image.GetSize() > 0)
    {
        EXPECT_EQ(0, std::memcmp(image.GetAddress(), tmpimg.GetAddress(), image.GetSize()));
    }
    return true;
}

template <typename T>
bool ModifyAndSync(
    a::profile::ProfileBase* pOutBase, a::profile::UserData* pOutUserData,
    const a::Uid& user, const char* nickname, const int userData, const T& image,
    a::profile::ProfileStorage& profileStorage,
    a::baas::BaasOperator& baasOp, a::Executor& executor) NN_NOEXCEPT
{
    *pOutBase = a::profile::DefaultProfileBase;

    pOutBase->author = user;
    std::strncpy(pOutBase->nickname, nickname, strlen(nickname) + 1);
    pOutBase->timeStamp = nn::os::GetSystemTick().ToTimeSpan().GetMilliSeconds();
    std::memset(pOutUserData->data, userData, sizeof(pOutUserData->data));
    NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(profileStorage.Update(
        user, *pOutBase, *pOutUserData, image.GetAddress(), image.GetSize()));

    a::baas::BaasUserProfileSynchronizationTask syncTask(user, baasOp);
    NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(syncTask.Initialize());
    NNT_ACCOUNT_CREATE_READ_EVENT(syncE, syncTask, nn::os::EventClearMode_ManualClear);
    NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(executor.Register(&syncTask));
    syncE.Wait();
    NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(syncTask.GetResult());
    executor.Unregister(&syncTask);

    EXPECT_TRUE(CompareTo(user, profileStorage, *pOutBase, *pOutUserData, image));
    return true;
}
template <typename T>
bool SyncAndCompare(
    const a::Uid& user,
    const a::profile::ProfileBase& base, const a::profile::UserData& userData, const T& image,
    a::profile::ProfileStorage& profileStorage,
    a::baas::BaasOperator& baasOp, a::Executor& executor) NN_NOEXCEPT
{
    a::baas::BaasUserProfileSynchronizationTask syncTask(user, baasOp);
    NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(syncTask.Initialize());
    NNT_ACCOUNT_CREATE_READ_EVENT(syncE, syncTask, nn::os::EventClearMode_ManualClear);
    NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(executor.Register(&syncTask));
    syncE.Wait();
    NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(syncTask.GetResult());
    executor.Unregister(&syncTask);

    // Resource cache
    nn::TimeSpan span;
    NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(baasOp.GetTimeSpanSinceLastSyncOfUserProfile(&span, user));
    EXPECT_TRUE(span <= nn::TimeSpan::FromMilliSeconds(100));

    nn::os::SleepThread(nn::TimeSpan::FromSeconds(3));
    NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_SUCCESS(baasOp.GetTimeSpanSinceLastSyncOfUserProfile(&span, user));
    EXPECT_TRUE(span > nn::TimeSpan::FromSeconds(3));

    baasOp.InvalidateUserResourceCache(user);
    NNT_ACCOUNT_RETURN_FALSE_UNLESS_RESULT_INCLUDED(
        a::ResultResourceCacheUnavailable,
        baasOp.GetTimeSpanSinceLastSyncOfUserProfile(&span, user));

    EXPECT_TRUE(CompareTo(user, profileStorage, base, userData, image));
    return true;
}

void RunProfileTest(
    a::Executor& executor,
    const a::Uid& user, const a::Uid& userB, a::baas::BaasOperator& baasOp,
    a::profile::ProfileStorage& profileStorage,
    a::baas::BaasUserInfoHolder& userInfoHolder,
    const a::detail::AbstractLocalStorage& storage) NN_NOEXCEPT
{
    NN_LOG(">>> Prepare DA Registration\n");
    bool registered;
    NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(baasOp.IsRegistered(&registered, user));
    ASSERT_TRUE(!registered);

    // DA 作成
    NN_LOG(">>> DA Registration\n");
    a::baas::BaasRegistrationTask regTask(user, baasOp);
    NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(regTask.Initialize());
    NNT_ACCOUNT_CREATE_READ_EVENT(regE, regTask, nn::os::EventClearMode_ManualClear);
    NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(executor.Register(&regTask));
    regE.Wait();
    NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(regTask.GetResult());
    executor.Unregister(&regTask);
    NN_UTIL_SCOPE_EXIT
    {
        NN_LOG(">>> DA Erasure\n");
        a::baas::BaasErasureTask unregTask(user, baasOp);
        NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(unregTask.Initialize());
        NNT_ACCOUNT_CREATE_READ_EVENT(unregE, unregTask, nn::os::EventClearMode_ManualClear);
        NN_ABORT_UNLESS_RESULT_SUCCESS(executor.Register(&unregTask));
        unregE.Wait();
        NN_ABORT_UNLESS_RESULT_SUCCESS(unregTask.GetResult());
        executor.Unregister(&unregTask);
    };

    // (別のユーザーに同じ credential を割り付ける)
    {
        userInfoHolder.lock();
        auto lock = storage.AcquireWriterLock();
        NN_UTIL_SCOPE_EXIT
        {
            userInfoHolder.unlock();
        };

        a::NetworkServiceAccountId id;
        NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(userInfoHolder.GetNetworkServiceAccountId(&id, user));
        a::baas::BaasCredential credential;
        NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(userInfoHolder.GetCredential(&credential, user));
        // コピー
        NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(userInfoHolder.Register(userB, id, credential));
    }

    // 1) 空のプロフィールをアップロード
    NN_LOG(">>> Empty profile\n");
    {
        a::profile::ProfileBase baseA0;
        a::profile::UserData userDataA0;
        t::Buffer imageA0;
        ASSERT_TRUE(LoadProfile(&baseA0, &userDataA0, &imageA0, user, profileStorage));

        // A Synchronize → 空のプロフィールがアップロードされる
        a::baas::BaasUserProfileSynchronizationTask syncTask(user, baasOp);
        NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(syncTask.Initialize());
        NNT_ACCOUNT_CREATE_READ_EVENT(syncE, syncTask, nn::os::EventClearMode_ManualClear);
        NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(executor.Register(&syncTask));
        syncE.Wait();
        NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(syncTask.GetResult());
        executor.Unregister(&syncTask);

        EXPECT_TRUE(CompareTo(user, profileStorage, baseA0, userDataA0, imageA0));

        ASSERT_TRUE(SyncAndCompare(user, baseA0, userDataA0, imageA0, profileStorage, baasOp, executor));
    }

    // 2) 内容のあるプロフィールをアップロード
    NN_LOG(">>> Filled profile\n");
    {
        a::profile::ProfileBase baseA1;
        a::profile::UserData userDataA1;
        t::InFile imageA1("rom:/DevUser_01.jpg");
        ASSERT_TRUE(ModifyAndSync(&baseA1, &userDataA1, user, "DevUser_01", 0xCA, imageA1, profileStorage, baasOp, executor));

        // B はプロフィールが空なので、 A のプロフィールが反映される
        ASSERT_TRUE(SyncAndCompare(userB, baseA1, userDataA1, imageA1, profileStorage, baasOp, executor));
    }

    // 3) ローカルにある古い変更は無視される
    NN_LOG(">>> Discarding local change\n");
    {
        a::profile::ProfileBase baseA2 = a::profile::DefaultProfileBase;
        a::profile::UserData userDataA2 = {{}};
        t::InFile imageA2("rom:/DevUser_02.jpg");

        nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));

        {
            a::profile::ProfileBase baseB1 = a::profile::DefaultProfileBase;
            a::profile::UserData userDataB1 = {{}};
            t::InFile imageB1("rom:/DevUser_09.jpg");
            baseB1.author = userB;
            std::strncpy(baseB1.nickname, "DevUser_09", sizeof("DevUser_09"));
            baseB1.timeStamp = nn::os::GetSystemTick().ToTimeSpan().GetMilliSeconds();
            std::memset(userDataB1.data, 0xCC, sizeof(userDataB1.data));
            NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(profileStorage.Update(userB, baseB1, userDataB1, imageB1.GetAddress(), imageB1.GetSize()));
        }

        ASSERT_TRUE(ModifyAndSync(&baseA2, &userDataA2, user, "DevUser_02", 0xB2, imageA2, profileStorage, baasOp, executor));

        // B への変更は古いので、 A のプロフィールが反映される
        ASSERT_TRUE(SyncAndCompare(userB, baseA2, userDataA2, imageA2, profileStorage, baasOp, executor));
    }

    // 4) ローカルにある新しい変更はアップロードされる
    NN_LOG(">>> Discarding remote change\n");
    {
        {
            a::profile::ProfileBase baseA3 = a::profile::DefaultProfileBase;
            a::profile::UserData userDataA3 = {{}};
            t::InFile imageA3("rom:/DevUser_03.jpg");
            ASSERT_TRUE(ModifyAndSync(&baseA3, &userDataA3, user, "DevUser_03", 0x6A, imageA3, profileStorage, baasOp, executor));
        }

        nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));

        a::profile::ProfileBase baseB2 = a::profile::DefaultProfileBase;
        a::profile::UserData userDataB2 = {{}};
        t::InFile imageB2("rom:/DevUser_08.jpg");
        {
            baseB2.author = userB;
            std::strncpy(baseB2.nickname, "DevUser_08", sizeof("DevUser_08"));
            baseB2.timeStamp = nn::os::GetSystemTick().ToTimeSpan().GetMilliSeconds();
            std::memset(userDataB2.data, 0xF8, sizeof(userDataB2.data));
            NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(profileStorage.Update(userB, baseB2, userDataB2, imageB2.GetAddress(), imageB2.GetSize()));
        }

        // B への変更は新しいので、 B のプロフィールが反映される
        ASSERT_TRUE(SyncAndCompare(userB, baseB2, userDataB2, imageB2, profileStorage, baasOp, executor));

        ASSERT_TRUE(SyncAndCompare(userB, baseB2, userDataB2, imageB2, profileStorage, baasOp, executor));
    }

    // 5) リモートの更新を Upload で上書きしてキャンセルする
    NN_LOG(">>> Force-upload local profile\n");
    {
        a::profile::ProfileBase baseB3 = a::profile::DefaultProfileBase;
        a::profile::UserData userDataB3 = {{}};
        t::InFile imageB3("rom:/DevUser_07.jpg");
        baseB3.author = userB;
        std::strncpy(baseB3.nickname, "DevUser_07", sizeof("DevUser_07"));
        baseB3.timeStamp = nn::os::GetSystemTick().ToTimeSpan().GetMilliSeconds();
        std::memset(userDataB3.data, 0xAC, sizeof(userDataB3.data));
        NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(profileStorage.Update(userB, baseB3, userDataB3, imageB3.GetAddress(), imageB3.GetSize()));

        nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));

        a::profile::ProfileBase baseA4 = a::profile::DefaultProfileBase;
        a::profile::UserData userDataA4 = {{}};
        t::InFile imageA4("rom:/DevUser_04.jpg");
        ASSERT_TRUE(ModifyAndSync(&baseA4, &userDataA4, user, "DevUser_04", 0x34, imageA4, profileStorage, baasOp, executor));

        {
            // B を Upload する
            a::baas::BaasUserProfileUploadTask ulTask(userB, baasOp);
            NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(ulTask.Initialize());
            NNT_ACCOUNT_CREATE_READ_EVENT(ulE, ulTask, nn::os::EventClearMode_ManualClear);
            NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(executor.Register(&ulTask));
            ulE.Wait();
            NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(ulTask.GetResult());
            executor.Unregister(&ulTask);
        }

        // B を sync しても B のまま
        ASSERT_TRUE(SyncAndCompare(userB, baseB3, userDataB3, imageB3, profileStorage, baasOp, executor));

        // 再度 A を sync すると A になる
        ASSERT_TRUE(SyncAndCompare(user, baseA4, userDataA4, imageA4, profileStorage, baasOp, executor));

        // 再度 B を sync すると B になる
        ASSERT_TRUE(SyncAndCompare(userB, baseA4, userDataA4, imageA4, profileStorage, baasOp, executor));
    }
} // NOLINT(readability/fn_size)

void RunChannelTest(
    a::Executor& executor,
    const a::Uid& user, const a::detail::ApplicationInfo& appInfo,
    a::baas::BaasOperator& baasOp) NN_NOEXCEPT
{
    // DA 作成
    NN_LOG(">>> DA Registration\n");
    a::baas::BaasRegistrationTask regTask(user, baasOp);
    NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(regTask.Initialize());
    NNT_ACCOUNT_CREATE_READ_EVENT(regE, regTask, nn::os::EventClearMode_ManualClear);
    NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(executor.Register(&regTask));
    regE.Wait();
    NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(regTask.GetResult());
    executor.Unregister(&regTask);
    bool registered;
    NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(baasOp.IsRegistered(&registered, user));
    ASSERT_TRUE(registered);

    NN_LOG(">>> Register token\n");
    nn::npns::NotificationToken nt;
    std::memset(nt.data, 'a', sizeof(nt.data));
    nt.data[sizeof(nt.data) - 1] = 'Z';

    // NT 登録
    NN_LOG(">>> Register notification token\n");
    a::baas::BaasNotificationRegistrationTask ntRegTask(user, nt, baasOp);
    NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(ntRegTask.Initialize());
    NNT_ACCOUNT_CREATE_READ_EVENT(ntRegE, ntRegTask, nn::os::EventClearMode_ManualClear);
    NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(executor.Register(&ntRegTask));
    ntRegE.Wait();
    NNT_ACCOUNT_EXPECT_RESULT_SUCCESS(ntRegTask.GetResult());
    executor.Unregister(&ntRegTask);

    // NT 取得
    NN_LOG(">>> Check notification token\n");
    nn::npns::NotificationToken nt1;
    a::baas::DebugBaasNotificationDownloadTask ntDlTask(&nt1, user, baasOp);
    NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(ntDlTask.Initialize());
    NNT_ACCOUNT_CREATE_READ_EVENT(ntDlE, ntDlTask, nn::os::EventClearMode_ManualClear);
    NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(executor.Register(&ntDlTask));
    ntDlE.Wait();
    NNT_ACCOUNT_EXPECT_RESULT_SUCCESS(ntDlTask.GetResult());
    executor.Unregister(&ntDlTask);

    // NT 比較
    EXPECT_EQ(0, std::memcmp(nt.data, nt1.data, sizeof(nt.data)));

    // NT 削除
    NN_LOG(">>> Unregister notification token\n");
    a::baas::BaasNotificationErasureTask ntUnregTask(user, baasOp);
    NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(ntUnregTask.Initialize());
    NNT_ACCOUNT_CREATE_READ_EVENT(ntUnregE, ntUnregTask, nn::os::EventClearMode_ManualClear);
    NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(executor.Register(&ntUnregTask));
    ntUnregE.Wait();
    NNT_ACCOUNT_EXPECT_RESULT_SUCCESS(ntUnregTask.GetResult());
    executor.Unregister(&ntUnregTask);

    // NT 取得 -> 失敗
    NN_LOG(">>> Check notification token not found\n");
    a::baas::DebugBaasNotificationDownloadTask ntDlTask1(&nt1, user, baasOp);
    NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(ntDlTask1.Initialize());
    NNT_ACCOUNT_CREATE_READ_EVENT(ntDlE1, ntDlTask1, nn::os::EventClearMode_ManualClear);
    NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(executor.Register(&ntDlTask1));
    ntDlE1.Wait();
    NNT_ACCOUNT_EXPECT_RESULT_INCLUDED(nn::account::baas::ResultBaasStatus404ResourceIsNotFound, ntDlTask1.GetResult());
    executor.Unregister(&ntDlTask1);

    NN_LOG(">>> DA Erasure\n");
    a::baas::BaasErasureTask unregTask(user, baasOp);
    NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(unregTask.Initialize());
    NNT_ACCOUNT_CREATE_READ_EVENT(unregE, unregTask, nn::os::EventClearMode_ManualClear);

    NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(executor.Register(&unregTask));
    unregE.Wait();
    NNT_ACCOUNT_EXPECT_RESULT_SUCCESS(unregTask.GetResult());
    executor.Unregister(&unregTask);
    NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(baasOp.IsRegistered(&registered, user));
    ASSERT_TRUE(!registered);
}

void RunRegistrationAndLoginTest(
    a::Executor& executor,
    const a::Uid& user, const a::detail::ApplicationInfo& appInfo,
    a::baas::BaasOperator& baasOp) NN_NOEXCEPT
{
    // DA 作成
    NN_LOG(">>> DA Registration\n");
    a::baas::BaasRegistrationTask regTask(user, baasOp);
    NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(regTask.Initialize());
    NNT_ACCOUNT_CREATE_READ_EVENT(regE, regTask, nn::os::EventClearMode_ManualClear);
    NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(executor.Register(&regTask));
    regE.Wait();
    NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(regTask.GetResult());
    executor.Unregister(&regTask);
    bool registered;
    NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(baasOp.IsRegistered(&registered, user));
    ASSERT_TRUE(registered);

    NN_LOG(">>> Queueing tasks (login, login with app)\n");
    a::baas::BaasLoginTaskForAccessToken accessTokenTask(user, baasOp);
    NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(accessTokenTask.Initialize());
    NNT_ACCOUNT_CREATE_READ_EVENT(accessTokenE, accessTokenTask, nn::os::EventClearMode_ManualClear);
    a::baas::BaasLoginTaskForIdTokenForApplication  idTokenTask(user, appInfo, baasOp);
    NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(idTokenTask.Initialize());
    NNT_ACCOUNT_CREATE_READ_EVENT(idTokenE, idTokenTask, nn::os::EventClearMode_ManualClear);

    NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(executor.Register(&accessTokenTask));
    NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(executor.Register(&idTokenTask));
    NN_UTIL_SCOPE_EXIT
    {
        accessTokenTask.Cancel();
        executor.Unregister(&accessTokenTask);

        idTokenTask.Cancel();
        executor.Unregister(&idTokenTask);
    };

    NN_LOG(">>> Wait for complete\n");
    accessTokenE.Wait();
    NNT_ACCOUNT_EXPECT_RESULT_SUCCESS(accessTokenTask.GetResult());
    executor.Unregister(&accessTokenTask);
    idTokenE.Wait();
    NNT_ACCOUNT_EXPECT_RESULT_SUCCESS(idTokenTask.GetResult());
    executor.Unregister(&idTokenTask);

    t::Buffer buffer(a::NetworkServiceAccountIdTokenLengthMax);
    size_t sizeActual;
    NNT_ACCOUNT_EXPECT_RESULT_SUCCESS(baasOp.LoadAccessTokenCache(&sizeActual, buffer.Get<char>(), buffer.GetSize(), user));
    EXPECT_TRUE(!(sizeActual < buffer.GetSize()) || buffer.Get<char>()[sizeActual] == '\0');
    NNT_ACCOUNT_EXPECT_RESULT_SUCCESS(baasOp.LoadIdTokenCache(&sizeActual, buffer.Get<char>(), buffer.GetSize(), user, appInfo));
    EXPECT_TRUE(!(sizeActual < buffer.GetSize()) || buffer.Get<char>()[sizeActual] == '\0');

    NN_LOG(">>> DA Erasure\n");
    a::baas::BaasErasureTask unregTask(user, baasOp);
    NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(unregTask.Initialize());
    NNT_ACCOUNT_CREATE_READ_EVENT(unregE, unregTask, nn::os::EventClearMode_ManualClear);

    NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(executor.Register(&unregTask));
    unregE.Wait();
    NNT_ACCOUNT_EXPECT_RESULT_SUCCESS(unregTask.GetResult());
    executor.Unregister(&unregTask);
    NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(baasOp.IsRegistered(&registered, user));
    ASSERT_TRUE(!registered);
}

void RunUserServiceEntryRequirementCacheTest(
    a::Executor& executor,
    const a::Uid& user, const a::detail::ApplicationInfo& appInfo,
    a::baas::BaasOperator& baasOp) NN_NOEXCEPT
{
    NN_LOG(">>> RunUserServiceEntryRequirementCacheTest\n");
    a::baas::BaasUserServiceEntryRequirement requirement;


    // 初期状態
    // - CheckAvailability ... NSA 登録必須
    // - CheckAvailabilityWith... ... NSA 登録必須
    // - GetUserServiceEntryRequirementCache ... None
    NNT_ACCOUNT_ASSERT_RESULT_INCLUDED(
        a::ResultNetworkServiceAccountRegistrationRequired,
        baasOp.CheckAvailability(user));
    NNT_ACCOUNT_ASSERT_RESULT_INCLUDED(
        a::ResultNetworkServiceAccountRegistrationRequired,
        baasOp.CheckAvailabilityWithUserServiceEntryRequirementCache(user, appInfo.launchProperty.id));
    NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(baasOp.GetUserServiceEntryRequirementCache(&requirement, user, appInfo.launchProperty.id));
    ASSERT_EQ(a::baas::BaasUserServiceEntryRequirement_None, requirement);

    // -----------------------------------------------------------------------------------------
    // NSA 登録
    a::baas::BaasRegistrationTask regTask(user, baasOp);
    NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(regTask.Initialize());
    NNT_ACCOUNT_CREATE_READ_EVENT(regE, regTask, nn::os::EventClearMode_ManualClear);
    NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(executor.Register(&regTask));
    regE.Wait();
    NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(regTask.GetResult());
    executor.Unregister(&regTask);
    bool registered;
    NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(baasOp.IsRegistered(&registered, user));
    ASSERT_TRUE(registered);

    // 状態
    // - CheckAvailability ... OK
    // - CheckAvailabilityWith... ... OK
    // - GetUserServiceEntryRequirementCache ... None
    NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(baasOp.CheckAvailability(user));
    NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(baasOp.CheckAvailabilityWithUserServiceEntryRequirementCache(user, appInfo.launchProperty.id));
    NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(baasOp.GetUserServiceEntryRequirementCache(&requirement, user, appInfo.launchProperty.id));
    ASSERT_EQ(a::baas::BaasUserServiceEntryRequirement_None, requirement);

    // -----------------------------------------------------------------------------------------
    // Requirement の設定
    baasOp.SetUserServiceEntryRequirementCache(user, appInfo.launchProperty.id, a::baas::BaasUserServiceEntryRequirement_Op2CommonLicense);

    // 設定後の状態
    // - CheckAvailability ... OK
    // - CheckAvailabilityWith... ... ResultOp2CommonLicenseRequired
    // - GetUserServiceEntryRequirementCache ... Op2CommonLicense
    NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(baasOp.CheckAvailability(user));
    NNT_ACCOUNT_ASSERT_RESULT_INCLUDED(
        a::ResultOp2CommonLicenseRequired,
        baasOp.CheckAvailabilityWithUserServiceEntryRequirementCache(user, appInfo.launchProperty.id));
    NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(baasOp.GetUserServiceEntryRequirementCache(&requirement, user, appInfo.launchProperty.id));
    ASSERT_EQ(a::baas::BaasUserServiceEntryRequirement_Op2CommonLicense, requirement);

    // -----------------------------------------------------------------------------------------
    // Invalidate
    baasOp.InvalidateUserServiceEntryRequirementCache(user, appInfo.launchProperty.id);

    // 状態
    // - CheckAvailability ... OK
    // - CheckAvailabilityWith... ... OK
    // - GetUserServiceEntryRequirementCache ... None
    NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(baasOp.CheckAvailability(user));
    NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(baasOp.CheckAvailabilityWithUserServiceEntryRequirementCache(user, appInfo.launchProperty.id));
    NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(baasOp.GetUserServiceEntryRequirementCache(&requirement, user, appInfo.launchProperty.id));
    ASSERT_EQ(a::baas::BaasUserServiceEntryRequirement_None, requirement);

    // -----------------------------------------------------------------------------------------
    // Requirement 設定 (NSA 削除で削除されることを確認する)
    baasOp.SetUserServiceEntryRequirementCache(user, appInfo.launchProperty.id, a::baas::BaasUserServiceEntryRequirement_Op2CommonLicense);

    // NSA 削除
    a::baas::BaasErasureTask unregTask(user, baasOp);
    NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(unregTask.Initialize());
    NNT_ACCOUNT_CREATE_READ_EVENT(unregE, unregTask, nn::os::EventClearMode_ManualClear);

    NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(executor.Register(&unregTask));
    unregE.Wait();
    NNT_ACCOUNT_EXPECT_RESULT_SUCCESS(unregTask.GetResult());
    executor.Unregister(&unregTask);
    NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(baasOp.IsRegistered(&registered, user));
    ASSERT_TRUE(!registered);

    // 状態
    // - CheckAvailability ... NSA 登録必須
    // - CheckAvailabilityWith... ... NSA 登録必須
    // - GetUserServiceEntryRequirementCache ... None
    NNT_ACCOUNT_ASSERT_RESULT_INCLUDED(
        a::ResultNetworkServiceAccountRegistrationRequired,
        baasOp.CheckAvailability(user));
    NNT_ACCOUNT_ASSERT_RESULT_INCLUDED(
        a::ResultNetworkServiceAccountRegistrationRequired,
        baasOp.CheckAvailabilityWithUserServiceEntryRequirementCache(user, appInfo.launchProperty.id));
    NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(baasOp.GetUserServiceEntryRequirementCache(&requirement, user, appInfo.launchProperty.id));
    ASSERT_EQ(a::baas::BaasUserServiceEntryRequirement_None, requirement);
}

NN_ALIGNAS(4096) char g_Stack[4096 * 8];
} // ~namespace <anonymous>

#include <nn/os/os_MultipleWaitUtility.h>

TEST(AccountBaasOperator, Basic)
{
    // ストレージ
    t::DefaultTestStorage storage;
    NN_ABORT_UNLESS_RESULT_SUCCESS(storage.Mount());
    storage.Clear();
    NN_ABORT_UNLESS_RESULT_SUCCESS(storage.Setup());

    // ユーザーとアプリケーション
    a::Uid user = a::detail::ConvertToUid(storage.GenerateUuidWithContext());
    const auto appInfo = a::detail::ApplicationInfo::Get(0x010000000000B121ull, 0, a::detail::ApplicationMediaType::System);

    // ユーザープロフィール
    a::profile::ProfileStorage profile;
    NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(profile.Initialize(&user, 1, storage));

    // NDAS トークンキャッシュ
    nn::account::ndas::ApplicationAuthenticationCache appAuthCache;
    NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(appAuthCache.Initialize(storage));

    // BaaS トークンキャッシュ
    a::baas::ClientAccessTokenCache clientAccessTokenCache;
    NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(clientAccessTokenCache.Initialize(storage));
    a::baas::UserAccessTokenCache userAccessTokenCache;
    NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(userAccessTokenCache.Initialize(storage));
    a::baas::UserIdTokenCache userIdTokenCache;
    NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(userIdTokenCache.Initialize(storage));
    nn::account::baas::BaasUserInfoHolder userInfoHolder;
    NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(userInfoHolder.Initialize(storage));

    // NDAS, BaaS Operator
    a::ndas::NdasOperator ndasOperator(appAuthCache);
    a::baas::BaasOperator baasOp(clientAccessTokenCache, userAccessTokenCache, userIdTokenCache, userInfoHolder, profile, ndasOperator, storage);

    // グローバルなリソース
    nn::os::Event terminator(nn::os::EventClearMode_ManualClear);

    // 実行時のリソース
    t::Thread thread;
    a::Executor executor;
    t::Buffer workBuffer(std::max(
        a::ndas::NdasDriver::RequiredBufferSize,
        std::max(
            a::baas::BaasLoginDriver::RequiredBufferSize,
            a::baas::BaasRegistrationDriver::RequiredBufferSize)));
    t::Buffer largeBuffer(512 * 1024);
    a::LargeBufferAllocator largeBufferAllocator(largeBuffer.GetAddress(), largeBuffer.GetSize());
    a::ExecutionResource resource = {
        curl_easy_init(),
        {workBuffer.GetAddress(), workBuffer.GetSize()},
        &largeBufferAllocator,
    };

    thread.Initialize(
        [&]{ ThreadFunction(terminator, executor, resource); },
        g_Stack, sizeof(g_Stack), nn::os::DefaultThreadPriority + 1, "AccountBaasOperator:Executor");
    thread.Start();

    // Register テスト実行
    RunRegistrationAndLoginTest(executor, user, appInfo, baasOp);
    bool registered;
    NNT_ACCOUNT_ASSERT_RESULT_SUCCESS(baasOp.IsRegistered(&registered, user));
    EXPECT_TRUE(!registered);

    // Channel テスト実行
    RunChannelTest(executor, user, appInfo, baasOp);

    // Profile テスト実行
    a::Uid userB = a::detail::ConvertToUid(storage.GenerateUuidWithContext());
    profile.Add(userB);
    RunProfileTest(executor, user, userB, baasOp, profile, userInfoHolder, storage);

    // UserServiceEntryRequirementCache テスト
    RunUserServiceEntryRequirementCacheTest(executor, user, appInfo, baasOp);

    // スレッド終了
    terminator.Signal();

    thread.Finalize();
    curl_easy_cleanup(resource.curlHandle);
}

#endif
